# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['aiopinboard']

package_data = \
{'': ['*']}

install_requires = \
['aiohttp>=3.6.2,<4.0.0',
 'async_timeout>=3.0.1,<4.0.0',
 'defusedxml>=0.6.0,<0.7.0',
 'maya>=0.6.1,<0.7.0']

setup_kwargs = {
    'name': 'aiopinboard',
    'version': '0.1.1',
    'description': 'A Python 3, asyncio-based library for the Pinboard API',
    'long_description': '# 📌 aiopinboard: A Python 3 Library for Pinboard\n\n[![CI](https://github.com/bachya/aiopinboard/workflows/CI/badge.svg)](https://github.com/bachya/aiopinboard/actions)\n[![PyPi](https://img.shields.io/pypi/v/aiopinboard.svg)](https://pypi.python.org/pypi/aiopinboard)\n[![Version](https://img.shields.io/pypi/pyversions/aiopinboard.svg)](https://pypi.python.org/pypi/aiopinboard)\n[![License](https://img.shields.io/pypi/l/aiopinboard.svg)](https://github.com/bachya/aiopinboard/blob/master/LICENSE)\n[![Code Coverage](https://codecov.io/gh/bachya/aiopinboard/branch/dev/graph/badge.svg)](https://codecov.io/gh/bachya/aiopinboard)\n[![Maintainability](https://api.codeclimate.com/v1/badges/a03c9e96f19a3dc37f98/maintainability)](https://codeclimate.com/github/bachya/aiopinboard/maintainability)\n[![Say Thanks](https://img.shields.io/badge/SayThanks-!-1EAEDB.svg)](https://saythanks.io/to/bachya)\n\n`aiopinboard` is a Python3, `asyncio`-focused library for interacting with the\n[Pinboard](https://pinboard.in) API.\n\n- [Installation](#installation)\n- [Python Versions](#python-versions)\n- [API Token](#api-token)\n- [Usage](#usage)\n  * [Bookmarks](#bookmarks)\n    + [The `Bookmark` Object](#the--bookmark--object)\n    + [Getting the Last Change Datetime](#getting-the-last-change-datetime)\n    + [Getting Bookmarks](#getting-bookmarks)\n    + [Adding a Bookmark](#adding-a-bookmark)\n    + [Deleting a Bookmark](#deleting-a-bookmark)\n  * [Tags](#tags)\n    + [Getting Tags](#getting-tags)\n    + [Getting Suggested Tags](#getting-suggested-tags)\n    + [Deleting a Tag](#deleting-a-tag)\n    + [Renaming a Tag](#renaming-a-tag)\n  * [Notes](#notes)\n    + [The `Note` Object](#the--note--object)\n    + [Getting Notes](#getting-notes)\n- [Contributing](#contributing)\n\n# Installation\n\n```python\npip install aiopinboard\n```\n\n# Python Versions\n\n`aiopinboard` is currently supported on:\n\n* Python 3.7\n* Python 3.8\n\n# API Token\n\nYou can retrieve your Pinboard API token via\n[your account\'s settings page](https://pinboard.in/settings/password).\n\n# Usage\n\n`aiopinboard` endeavors to replicate all of the endpoints\n[the Pinboard API documentation](https://pinboard.in/api) with sane, usable responses.\n\nAll API usage starts with creating an `API` object that contains your Pinboard API token:\n\n```python\nimport asyncio\n\nfrom aiopinboard import Client\n\n\nasync def main() -> None:\n    api = API("<PINBOARD_API_TOKEN>")\n    # do things!\n\n\nasyncio.run(main())\n```\n\n## Bookmarks\n\n### The `Bookmark` Object\n\nAPI endpoints that retrieve one or more bookmarks will return `Bookmark` objects, which\ncarry all of the expected properties of a bookmark:\n\n* `hash`: the unique identifier of the bookmark\n* `href`: the bookmark\'s URL\n* `title`: the bookmark\'s title\n* `description`: the bookmark\'s description\n* `last_modified`: the UTC date the bookmark was last modified\n* `tags`: a list of tags applied to the bookmark\n* `unread`: whether the bookmark is unread\n* `shared`: whether the bookmark is shared\n\n### Getting the Last Change Datetime\n\nTo get the UTC datetime of the last "change" (bookmark added, updated, or deleted):\n\n```python\nimport asyncio\n\nfrom aiopinboard import Client\n\n\nasync def main() -> None:\n    api = API("<PINBOARD_API_TOKEN>")\n    last_change_dt = await.async_get_last_change_datetime()\n    # >>> datetime.datetime(2020, 9, 3, 13, 7, 19, tzinfo=<UTC>)\n\n\nasyncio.run(main())\n```\n\nThis method should be used to determine whether additional API calls should be made –\nfor example, if nothing has changed since the last time a request was made, the\nimplementing library can halt.\n\n### Getting Bookmarks\n\nTo get a bookmark by its URL:\n\n```python\nimport asyncio\n\nfrom aiopinboard import Client\n\n\nasync def main() -> None:\n    api = API("<PINBOARD_API_TOKEN>")\n    await api.bookmark.async_get_bookmark_by_url("https://my.com/bookmark")\n    # >>> <Bookmark href="https://my.com/bookmark">\n\n\nasyncio.run(main())\n```\n\nTo get all bookmarks\n\n```python\nimport asyncio\n\nfrom aiopinboard import Client\n\n\nasync def main() -> None:\n    api = API("<PINBOARD_API_TOKEN>")\n    await api.bookmark.async_get_all_bookmarks()\n    # >>> [<Bookmark ...>, <Bookmark ...>]\n\n\nasyncio.run(main())\n```\n\nYou can specify several optional parameters while editing a bookmark:\n\n* `tags`: an optional list of tags to filter results by\n* `start`: the optional starting index to return (defaults to the start)\n* `results`: the optional number of results (defaults to all)\n* `from_dt`: the optional datetime to start from\n* `to_dt`: the optional datetime to end at\n\nTo get all bookmarks created on a certain date:\n\n```python\nimport asyncio\nfrom datetime import date\n\nfrom aiopinboard import Client\n\n\nasync def main() -> None:\n    api = API("<PINBOARD_API_TOKEN>")\n    await api.bookmark.async_get_bookmarks_by_date(date.today())\n    # >>> [<Bookmark ...>, <Bookmark ...>]\n\n    # Optionally filter the results with a list of tags – note that only bookmarks that\n    # have all tags will be returned:\n    await api.bookmark.async_get_bookmarks_by_date(date.today(), tags=["tag1", "tag2"])\n    # >>> [<Bookmark ...>, <Bookmark ...>]\n)\n\n\nasyncio.run(main())\n```\n\nTo get recent bookmarks:\n\n```python\nimport asyncio\n\nfrom aiopinboard import Client\n\n\nasync def main() -> None:\n    api = API("<PINBOARD_API_TOKEN>")\n    await api.bookmark.async_get_recent_bookmarks(count=10)\n    # >>> [<Bookmark ...>, <Bookmark ...>]\n\n    # Optionally filter the results with a list of tags – note that only bookmarks that\n    # have all tags will be returned:\n    await api.bookmark.async_get_recent_bookmarks(count=20, tags=["tag1", "tag2"])\n    # >>> [<Bookmark ...>, <Bookmark ...>]\n\n\nasyncio.run(main())\n```\n\nTo get a summary of dates and how many bookmarks were created on those dates:\n\n```python\nimport asyncio\n\nfrom aiopinboard import Client\n\n\nasync def main() -> None:\n    api = API("<PINBOARD_API_TOKEN>")\n    dates = await api.bookmark.async_get_dates()\n    # >>> {datetime.date(2020, 09, 05): 4, ...}\n\n\nasyncio.run(main())\n```\n### Adding a Bookmark\n\nTo add a bookmark:\n\n```python\nimport asyncio\n\nfrom aiopinboard import Client\n\n\nasync def main() -> None:\n    api = API("<PINBOARD_API_TOKEN>")\n    await api.bookmark.async_add_bookmark("https://my.com/bookmark", "My New Bookmark")\n\n\nasyncio.run(main())\n```\n\nYou can specify several optional parameters while editing a bookmark:\n\n* `description`: the optional description of the bookmark\n* `tags`: an optional list of tags to assign to the bookmark\n* `created_datetime`: the optional creation datetime to use (defaults to now)\n* `replace`: whether this should replace a bookmark with the same URL\n* `shared`: whether this bookmark should be shared\n* `toread`: whether this bookmark should be unread\n\n### Deleting a Bookmark\n\nTo delete a bookmark by its URL:\n\n```python\nimport asyncio\n\nfrom aiopinboard import Client\n\n\nasync def main() -> None:\n    api = API("<PINBOARD_API_TOKEN>")\n    await api.bookmark.async_delete_bookmark("https://my.com/bookmark")\n\n\nasyncio.run(main())\n```\n\n## Tags\n\n### Getting Tags\n\nTo get all tags for an account (and a count of how often each tag is used):\n\n```python\nimport asyncio\n\nfrom aiopinboard import Client\n\n\nasync def main() -> None:\n    api = API("<PINBOARD_API_TOKEN>")\n    await api.tag.async_get_tags()\n    # >>> {"tag1": 3, "tag2": 8}\n\n\nasyncio.run(main())\n```\n\n### Getting Suggested Tags\n\nTo get lists of popular (used by the community) and recommended (used by you) tags for a\nparticular URL:\n\n```python\nimport asyncio\n\nfrom aiopinboard import Client\n\n\nasync def main() -> None:\n    api = API("<PINBOARD_API_TOKEN>")\n    await api.bookmark.async_get_suggested_tags("https://my.com/bookmark")\n    # >>> {"popular": ["tag1", "tag2"], "recommended": ["tag3"]}\n\n\nasyncio.run(main())\n```\n\n### Deleting a Tag\n\nTo delete a tag:\n\n```python\nimport asyncio\n\nfrom aiopinboard import Client\n\n\nasync def main() -> None:\n    api = API("<PINBOARD_API_TOKEN>")\n    await api.tag.async_delete_tag("tag1")\n\n\nasyncio.run(main())\n```\n\n### Renaming a Tag\n\nTo rename a tag:\n\n```python\nimport asyncio\n\nfrom aiopinboard import Client\n\n\nasync def main() -> None:\n    api = API("<PINBOARD_API_TOKEN>")\n    await api.tag.async_rename_tag("old-tag", "new-tag")\n\n\nasyncio.run(main())\n```\n\n## Notes\n\n### The `Note` Object\n\nAPI endpoints that retrieve one or more notes will return `Note` objects, which\ncarry all of the expected properties of a note:\n\n* `note_id`: the unique ID\n* `title`: the title\n* `hash`: the computed hash\n* `created_at`: the UTC datetime the note was created\n* `updated_at`: the UTC datetime the note was updated\n* `length`: the length\n\n### Getting Notes\n\nTo get all notes for an account:\n\n```python\nimport asyncio\n\nfrom aiopinboard import Client\n\n\nasync def main() -> None:\n    api = API("<PINBOARD_API_TOKEN>")\n    await api.note.async_get_notes()\n    # >>> [<Note ...>, <Note ...>]\n\n\nasyncio.run(main())\n```\n\n\n# Contributing\n\n1. [Check for open features/bugs](https://github.com/bachya/aiopinboard/issues)\n  or [initiate a discussion on one](https://github.com/bachya/aiopinboard/issues/new).\n2. [Fork the repository](https://github.com/bachya/aiopinboard/fork).\n3. (_optional, but highly recommended_) Create a virtual environment: `python3 -m venv .venv`\n4. (_optional, but highly recommended_) Enter the virtual environment: `source ./.venv/bin/activate`\n5. Install the dev environment: `script/setup`\n6. Code your new feature or bug fix.\n7. Write tests that cover your new functionality.\n8. Run tests and ensure 100% code coverage: `script/test`\n9. Update `README.md` with any new documentation.\n10. Add yourself to `AUTHORS.md`.\n11. Submit a pull request!\n',
    'author': 'Aaron Bach',
    'author_email': 'bachya1208@gmail.com',
    'maintainer': None,
    'maintainer_email': None,
    'url': 'https://github.com/bachya/aiopinboard',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'python_requires': '>=3.7.0,<4.0.0',
}


setup(**setup_kwargs)
