from collections import defaultdict
from sri.d3mglue.d3mwrap import D3MWrappedOperators, D3MWrapperClassFactory
import logging
import os
import sys
import json
CURRENT_DIR = os.path.dirname(os.path.abspath(__file__))
sys.path.append(os.path.dirname(CURRENT_DIR))
sys.path.append(os.path.abspath('config'))
from .config.load_config import LoadConfig
import d3m

#logging.basicConfig(level=logging.DEBUG, format='%(asctime)s [%(levelname)s] %(name)s -- %(message)s')
_logger = logging.getLogger(__name__)

TUNING_PARAMETER = 'https://metadata.datadrivendiscovery.org/types/TuningParameter'

def config_to_d3m(config, operator_advice='strict', d3m_white_list_file=None, family=None):
    """ Converts TPOT or GAMA config dictionary to one that obeys the allowed d3m hyperparameter values.

    D3M wrappers do not always specify all hyperparameters or values, so the original configuration might need to be
    adjusted to account for the differences. Additionally, the primitive metadata may be used to sample possible
    values instead.

    :param config:
    :param operator_advice:
    :param d3m_white_list_file: If specified, this file lists the open d3m primitives that have passed the
                           triage_d3m_primitives CI tests.
    :return:
    """
    if operator_advice not in ['strict', 'query', 'discover']:
        raise ValueError("`operator_advice` must be one of ['strict','query', 'discover'].")

    new_config = defaultdict(lambda: defaultdict(list))
    available_primitives = set()
    primitive_keys = set()

    for primitive_key, hyperparams in config.items():
        if isinstance(primitive_key, str):
            # Import class (based on string) and re-assign as primitive variable
            raise NotImplementedError()
        else:
            primitive = primitive_key
            primitive_key = D3MWrapperClassFactory(primitive_key, primitive_key.__name__)

        primitive_metadata = primitive.metadata.query()
        hyperparams_metadata = primitive_metadata['primitive_code']['class_type_arguments']['Hyperparams']
        specified_hyperparameters = hyperparams_metadata.defaults()

        for hyperparameter, values in hyperparams.items():
            if hyperparameter not in specified_hyperparameters:
                continue
            if operator_advice == 'strict':
                # only consider values specified in config
                for value in values:
                    try:
                        hyperparams_metadata.configuration[hyperparameter].validate(value)
                    except (d3m.exceptions.InvalidArgumentTypeError, d3m.exceptions.InvalidArgumentValueError):
                        print('Throwing out {}.{}={}'.format(primitive.__name__, hyperparameter, value))
                        continue

                    new_config[primitive_key][hyperparameter].append(value)
                    available_primitives.add(primitive_key.__name__)
                    primitive_keys.add(str(primitive))

    from_config = len(available_primitives)
    _logger.info("Loaded %s primitives from the config" % from_config)

    # The d3m_white_list_file is generated by triage_d3m_primitives. It is a list of all primitive that pass a basic
    # usability test. Here, we will load only the primitives that are suitable for use by the optimizer implementation.
    if not d3m_white_list_file is None:
        load_config = LoadConfig()
        json_data = load_config.load_json(d3m_white_list_file)
        d3m_primitives = load_config.parse_data(json_data, family, primitive_keys)
        for primitive in d3m_primitives:
            # if primitive.__name__ == 'AF_SKAdaBoostClassifier':
            if primitive.__name__ not in available_primitives:
                available_primitives.add(primitive.__name__)
                _logger.info("Adding whitelist primitive: %s" % primitive.__name__)
                for hyperparameter in d3m_primitives[primitive]:
                    for value in d3m_primitives[primitive][hyperparameter]:
                        new_config[primitive][hyperparameter].append(value)
                        # For now lets just load 1
                    # break

    _logger.info("Loaded %s primitives from D3M whitelist" % (len(available_primitives) - from_config))

    return new_config


