import logging
import sys
from threading import Lock

logging.basicConfig(level=logging.INFO, format='%(asctime)s [%(levelname)s] %(name)s -- %(message)s')
_logger = logging.getLogger(__name__)

'''
This class is responsible for managing the ids generated by the TA2 to TA3 api exchange. Some of these
ids are related in a chain and are maintained as such
'''
class IdTracker(object):

    # To protect access to data structures
    mutex = Lock()

    # Holds the list of taken ids - this prevents any id duplication
    id_set = set()

    # Holds the active search_id's
    search_id_set = set()

    # Holds the active search_id's for fully specified problems
    search_id_fully_specified_set = set()

    # Holds the active solution_id's for fully specified problems
    solution_id_fully_specified_set = set()

    # Holds the search_id to solution_id lists
    search_id_to_solution_ids_map = {}

    # Maps the solution ids to the ScoreSolution::request_id
    score_solution_request_id_to_solution_id_map = {}

    # Maps the metrics to the ScoreSolution::request_id
    score_solution_request_id_to_metric_map = {}

    # Maps the solution ids to the FitSolution::request_id
    fit_request_id_to_solution_id_map = {}

    # Holds the list of fitted_solution_ids and their corresponding solution_ids
    fitted_solution_id_to_solution_id_map = {}

    produce_request_id_to_fitted_solution_id_map = {}

    produce_request_id_to_solution_id_map = {}


    def __init__(self):
        pass


    '''
    Invoked when we get a new SearchSolutionsRequest from TA3
    '''
    def add_new_search_id(self, search_id):
        # Ensure safe access to the id structures
        self.mutex.acquire()
        try:
            self.search_id_set.add(search_id)
            _logger.info("New search id added: %s, now tracking %s searches" % (search_id, len(self.search_id_set)))
        finally:
            self.mutex.release()


    '''
    Returns True if the supplied search_id is present in the list of tracked search_id's, False otherwise
    '''
    def contains_search_id(self, search_id):
        # Ensure safe access to the id structures
        self.mutex.acquire()
        try:
            present = search_id in self.search_id_set
        finally:
            self.mutex.release()
        return present


    '''
    When the TA3 tells us to drop the specified search id we purge it from the tracked list
    '''
    def remove_search_id(self, search_id):
        # Ensure safe access to the id structures
        self.mutex.acquire()
        try:
            self.search_id_set.remove(search_id)
            #TODO: Should we also purge the solution ids associated with this? Probably..
        finally:
            self.mutex.release()


    '''
    Invoked when we discover a search_id is associated with a fully specified pipeline
    '''
    def search_id_is_fully_specified(self, search_id):
        # Ensure safe access to the id structures
        self.mutex.acquire()
        try:
            self.search_id_fully_specified_set.add(search_id)
            if not search_id in self.search_id_set:
                _logger.warning("A fully specified search id was not in the active searches: %s" % search_id)
            _logger.info("Search id: %s, is being tracked as fully specified. Total fully specified count: %s" %
                         (search_id, len(self.search_id_fully_specified_set)))
        finally:
            self.mutex.release()


    '''
    Returns True if the supplied search_id is present in the list of tracked fully specified search_id's, False otherwise
    '''
    def search_id_fully_specified(self, search_id):
        # Ensure safe access to the id structures
        self.mutex.acquire()
        try:
            present = search_id in self.search_id_fully_specified_set
        finally:
            self.mutex.release()
        return present


    '''
    Get the fitted_solution_id associated with the supplied search_id and at the specified index in that list
    '''
    def get_fitted_solution_id(self, search_id, index):
        # Ensure safe access to the id structures
        self.mutex.acquire()
        try:
            fitted_solution_id = self.search_id_to_solution_ids_map[search_id][index]
        finally:
            self.mutex.release()
        return fitted_solution_id


    '''
    When we start searching for pipelines, associate those ids with the search id that invoked them
    '''
    def add_solution_id_to_search(self, search_id, solution_id):
        solution_ids_list = None
        # Ensure safe access to the id structures
        self.mutex.acquire()
        try:
            if search_id in self.search_id_to_solution_ids_map:
                solution_ids_list = self.search_id_to_solution_ids_map[search_id]
            else:
                solution_ids_list = []
            solution_ids_list.append(solution_id)
            self.search_id_to_solution_ids_map[search_id] = solution_ids_list
            _logger.info("Just added solution id %s to search id %s, list size is now: %s" %
                         (solution_id, search_id, len(solution_ids_list)))
        finally:
            self.mutex.release()


    '''
    Invoked when we discover a solution_id is associated with a fully specified pipeline
    '''
    def set_solution_id_is_fully_specified(self, solution_id):
        # Ensure safe access to the id structures
        self.mutex.acquire()
        try:
            self.solution_id_fully_specified_set.add(solution_id)
            _logger.info("Solution id: %s, is being tracked as fully specified. Total fully specified count: %s" %
                         (solution_id, len(self.solution_id_fully_specified_set)))
        finally:
            self.mutex.release()


    '''
    Check if a solution_id is associated with a fully specified pipeline
    '''
    def solution_id_is_fully_specified(self, solution_id):
        # Ensure safe access to the id structures
        self.mutex.acquire()
        try:
            present = solution_id in self.solution_id_fully_specified_set
            _logger.info("Solution id: %s, %s is being tracked as fully specified. Total fully specified count: %s" %
                         (solution_id, present, len(self.solution_id_fully_specified_set)))
        finally:
            self.mutex.release()
            return present


    '''
    Get the solution_id associated with the supplied search_id - used by 
    '''
    def get_solution_id_for_search_id(self, search_id):
        # Ensure safe access to the id structures
        self.mutex.acquire()
        try:
            solution_id = self.search_id_to_solution_ids_map[search_id]
        finally:
            self.mutex.release()
        return solution_id[0]


    '''
    When a ScoreSolution is called with a solution_id we generate and link a request_id with it
    '''
    def link_solution_id_to_score_request_id(self, solution_id, request_id):
        # Ensure safe access to the id structures
        self.mutex.acquire()
        try:
            self.score_solution_request_id_to_solution_id_map[request_id] = solution_id
        finally:
            self.mutex.release()


    '''
    When a ScoreSolution is called with a solution_id we generate and link a requested metric with it
    '''
    def link_metric_to_score_request_id(self, metric, request_id):
        # Ensure safe access to the id structures
        self.mutex.acquire()
        try:
            self.score_solution_request_id_to_metric_map[request_id] = metric
        finally:
            self.mutex.release()

    '''
    For a given score solution request_id, return the linked solution_id
    '''
    def get_solution_id_for_score_request_id(self, request_id):
        # Ensure safe access to the id structures
        self.mutex.acquire()
        try:
            return self.score_solution_request_id_to_solution_id_map[request_id]
        finally:
            self.mutex.release()


    '''
    For a given score solution request_id, return the linked metric
    '''
    def get_metric_for_score_request_id(self, request_id):
        # Ensure safe access to the id structures
        self.mutex.acquire()
        try:
            return self.score_solution_request_id_to_metric_map[request_id]
        finally:
            self.mutex.release()


    '''
    When a FitRequest is made we generaqte a new request_id. We want to link that request_id with the
    solution_id it is tied to
    '''

    def link_fit_request_id_to_solution_id(self, fit_request_id, solution_id):
        # Ensure safe access to the id structures
        self.mutex.acquire()
        try:
            self.fit_request_id_to_solution_id_map[fit_request_id] = solution_id
        finally:
            self.mutex.release()


    '''
    For a given fit solution request_id, return the linked solution_id
    '''
    def get_solution_id_for_fit_request_id(self, request_id):
        # Ensure safe access to the id structures
        self.mutex.acquire()
        try:
            return self.fit_request_id_to_solution_id_map[request_id]
        finally:
            self.mutex.release()


    '''
    When a GetFitSolutionResults is called and the status changes to COMPLETED, we generate a fitted_solution_id
    and then link it to a solution_id (pipeline identifier)
    '''
    def link_fitted_solution_id_to_solution_id(self, fitted_solution_id, solution_id):
        # Ensure safe access to the id structures
        self.mutex.acquire()
        try:
            self.fitted_solution_id_to_solution_id_map[fitted_solution_id] = solution_id
        finally:
            self.mutex.release()


    def get_solution_id_for_fitted_solution_id(self, fitted_solution_id):
        # Ensure safe access to the id structures
        self.mutex.acquire()
        try:
            return self.fitted_solution_id_to_solution_id_map[fitted_solution_id]
        finally:
            self.mutex.release()


    '''
    When a ProduceSolution is called we create a new request_id for the request and link it to 
    a fitted_solution_id for tracking purposes
    '''
    def link_produce_request_id_with_fitted_solution_id(self, request_id, fitted_solution_id):
        # Ensure safe access to the id structures
        self.mutex.acquire()
        try:
            self.produce_request_id_to_fitted_solution_id_map[request_id] = fitted_solution_id
        finally:
            self.mutex.release()


    '''
    When GetProduceSolutionResults is called with a request id created in ProduceSolution, use it to get 
    the corresponding fitted_solution_id
    '''
    def get_fitted_solution_id_for_produce_request_id(self, request_id):
        # Ensure safe access to the id structures
        self.mutex.acquire()
        try:
            return self.produce_request_id_to_fitted_solution_id_map[request_id]
        finally:
            self.mutex.release()


    def link_produce_request_id_with_solution_id(self, request_id, solution_id):
        # Ensure safe access to the id structures
        self.mutex.acquire()
        try:
            self.produce_request_id_to_solution_id_map[request_id] = solution_id
        finally:
            self.mutex.release()

    def get_solution_id_for_produce_request_id(self, request_id):
        # Ensure safe access to the id structures
        self.mutex.acquire()
        try:
            return self.produce_request_id_to_solution_id_map[request_id]
        finally:
            self.mutex.release()

    '''
    Returns True if the id is already taken, false otherwise. Currently these ids are deterministic which is handy for 
    debugging - when everything is working we will make this more random
    '''
    def id_taken(self, id):
        if id in self.id_set:
            return True
        else:
            self.id_set.add(id)
            return False



class TestIdTracker(object):

    def main(self, argv):
        tracker = IdTracker()

        search_id_1 = "search solution id 1"
        tracker.add_new_search_id(search_id_1)

        tracker.add_solution_id_to_search(search_id_1, "solution 1")
        tracker.add_solution_id_to_search(search_id_1, "solution 2")
        tracker.add_solution_id_to_search(search_id_1, "solution 3")
        tracker.add_solution_id_to_search(search_id_1, "solution 4")



'''
Entry point - required to make python happy
'''
if __name__ == "__main__":
    TestIdTracker().main(sys.argv)