import core_pb2
import core_pb2_grpc
import logging
from google.protobuf.timestamp_pb2 import Timestamp
import random
import string
import os
import sys
import threading
import d3m
import time

from message_converter import MessageConverter
from message_to_pipeline import MessageToPipeline
import problem_pb2
import value_pb2

from core_pb2 import HelloResponse
from core_pb2 import SearchSolutionsResponse
from core_pb2 import GetSearchSolutionsResultsResponse
from core_pb2 import SolutionSearchScore
from core_pb2 import ScoringConfiguration
from core_pb2 import Progress
from core_pb2 import ProgressState
from core_pb2 import ScoreSolutionResponse
from core_pb2 import GetScoreSolutionResultsResponse
from core_pb2 import Score
from core_pb2 import EndSearchSolutionsResponse
from core_pb2 import GetProduceSolutionResultsResponse
from core_pb2 import SolutionExportResponse
from core_pb2 import GetFitSolutionResultsResponse
from core_pb2 import StopSearchSolutionsResponse
from core_pb2 import ListPrimitivesResponse
from core_pb2 import ProduceSolutionResponse
from core_pb2 import FitSolutionResponse
from core_pb2 import DescribeSolutionResponse
from id_tracker import IdTracker

from problem_pb2 import ProblemPerformanceMetric
from problem_pb2 import ProblemTarget

from value_pb2 import Value

from ta3ta2_api.utils import decode_pipeline_description
from d3m.metadata import pipeline as pipeline_module
from d3m.container import Dataset, DataFrame
from d3m.metadata import base as Base
from d3m.runtime import Runtime
from d3m.container import Dataset
from ta3ta2_api.utils import decode_problem_description
from ta3ta2_api.utils import encode_pipeline_description

CURRENT_DIR = os.path.dirname(os.path.abspath(__file__))
sys.path.append(os.path.dirname(CURRENT_DIR))
import autoflowconfig
import pandas
import traceback

logging.basicConfig(level=logging.INFO, format='%(asctime)s [%(levelname)s] %(name)s -- %(message)s')
_logger = logging.getLogger(__name__)


'''
This class implements the CoreServicer base class. The CoreServicer defines the methods that must be supported by a 
TA2 server implementation. The CoreServicer class is generated by grpc using the core.proto descriptor file. See:
https://gitlab.com/datadrivendiscovery/ta3ta2-api.  
'''
class TA2Servicer(core_pb2_grpc.CoreServicer):

    """
    Maps the TA3 enumerated metric names to the underlying TA2 TPOT implementation
    """
    resolve_metric_to_ta2 = {
        "ACCURACY": "accuracy",
        "PRECISION": "", # TODO: Add implementation
        "RECALL": "", # TODO: Add implementation
        "F1": "f1_true",
        "F1_MICRO": "f1_micro",
        "F1_MACRO": "f1_macro",
        "ROC_AUC": "roc_auc",
        "ROC_AUC_MICRO": "recall_micro",
        "ROC_AUC_MACRO": "recall_macro",
        "MEAN_SQUARED_ERROR": "mean_squared_error",
        "ROOT_MEAN_SQUARED_ERROR": "root_mean_squared_error",
        "ROOT_MEAN_SQUARED_ERROR_AVG": "root_mean_squared_error_average",
        "MEAN_ABSOLUTE_ERROR": "mean_absolute_error",
        "R_SQUARED": "r2",
        "NORMALIZED_MUTUAL_INFORMATION": "normalized_mutual_info_score",
        "JACCARD_SIMILARITY_SCORE": "jaccard_similarity_score",
        "PRECISION_AT_TOP_K ": "", # TODO: Add implementation
        "OBJECT_DETECTION_AVERAGE_PRECISION": "", # TODO: Add implementation
        "LOSS": "" # TODO: Add implementation
    }

    '''
    Maps the TA2 metric names to the TA3 enumerated representation         
    '''
    resolve_metric_to_ta3 = {
        "accuracy": "ACCURACY",
        "precision-TODO": "PRECISION",
        "recall-TODO": "RECALL",
        "f1_true": "F1",
        "f1_micro": "F1_MICRO",
        "f1_macro": "F1_MACRO",
        "roc_auc": "ROC_AUC",
        "recall_micro": "ROC_AUC_MICRO",
        "recall_macro": "ROC_AUC_MACRO",
        "mean_squared_error": "MEAN_SQUARED_ERROR",
        "neg_mean_squared_error": "MEAN_SQUARED_ERROR",
        "root_mean_squared_error": "ROOT_MEAN_SQUARED_ERROR",
        "neg_mean_absolute_error": "MEAN_ABSOLUTE_ERROR",
        "mean_absolute_error": "MEAN_ABSOLUTE_ERROR",
        "r2": "R_SQUARED",
        "normalized_mutual_info_score": "NORMALIZED_MUTUAL_INFORMATION",
        "jaccard_similarity_score": "JACCARD_SIMILARITY_SCORE",
        "precesion_at_top_k": "PRECISION_AT_TOP_K",
        "object_detection_average_precisions": "OBJECT_DETECTION_AVERAGE_PRECISION",
        "loss-TODO": "LOSS"
    }

    '''
    Maps the TA3 enumerated type names to the underlying TA2 TPOT implementation
    '''
    resolve_taskType_to_ta2 = {
        "CLASSIFICATION": "classification",
        "REGRESSION": "regression",
        "CLUSTERING": "", # TODO: Add implementation
        "LINK_PREDICTION": "linkPrediction",
        "VERTEX_NOMINATION": "vertexNomination",
        "VERTEX_CLASSIFICATION": "vertexClassification",
        "GRAPH_CLUSTERING": "", # TODO: Add implementation
        "GRAPH_MATCHING": "graphMatching",
        "TIME_SERIES_FORECASTING": "timeSeriesForecasting",
        "COLLABORATIVE_FILTERING": "collaborativeFiltering",
        "OBJECT_DETECTION": "objectDetection",
        "COMMUNITY_DETECTION": "communityDetection",
        "SEMISUPERVISED": "semiSupervisedClassification"
    }


    '''
    Maps the underlying TA2 implementation to the TA3 enumerated types 
    '''
    resolve_state_to_ta3 = {
        "pending": core_pb2.PENDING,
        "running": core_pb2.RUNNING,
        "completed": core_pb2.COMPLETED,
        "errored": core_pb2.ERRORED,
    }


    # How often we yield streamed results to the TA3
    update_interval = 10
    # Used to track the network of request ids sent to and from the TA3
    id_tracker = IdTracker()


    '''
    The __init__ method is used to establish the underlying TA2 libraries to service requests from the TA3 system.
    '''
    def __init__(self, mode="TPOT"):
        self.log_msg("Init invoked (ta2trace) in")
        self.mode = mode
        self.log_msg("Running in %s mode" % mode)
        self.log_msg("Init returning (ta2trace) out")
        # For fully specified pipelines we need to call fit - and so may have a primitive in a supplied pipeline that
        # requires the static volume
        self.static_volumes = None
        if 'D3MSTATICDIR' in os.environ:
            self.static_volumes = os.environ['D3MSTATICDIR']
            logging.info("Setting static_volumes to %s", os.environ['D3MSTATICDIR'])
        self.output_dir = "./"
        if 'D3MOUTPUTDIR' in os.environ:
            self.output_dir = os.environ['D3MOUTPUTDIR']
        self.temp = "%s/%s" % (self.output_dir, "temp")
        if not os.path.exists(self.temp):
            os.makedirs(self.temp, exist_ok=True)


    '''
    Identify a TA2 and get supported features. This call is also suitable for a ping/pong call to check that 
    the gRPC connection to the TA2 is ready.
        Input: HelloRequest
        Output Hello Response
        Non streaming call
    '''
    def Hello(self, request, context):
        self.log_msg(msg="(1)> Hello invoked (ta2trace) in")
        # TODO: Make sure this is activated for eval and CI submission to be sure we dont waste search time loading primitives
        self.ListPrimitives(request=None, context=None)
        self.log_msg(msg="(1)< Hello returning (ta2trace) out")
        return HelloResponse(user_agent="SRI",
                             version="2020.6.2",
                             allowed_value_types=[
                                 "DATASET_URI",
                                 "CSV_URI",
                                 "RAW"
                             ],
                             supported_extensions="")


    '''
    Starts a new solution search. Found solutions have not necessary been fitted on the provided inputs. Problem
    description and inputs are used only to help guide the search process Consider any found solutions to be 
    just a static description of solutions at this stage. Multiple parallel solution searches can happen at the 
    same time.
        Input: SearchSolutionsRequest 
        Output: SearchSolutionsResponse
        Non streaming
        Ids: Creates a search_id for this search instance
    '''
    def SearchSolutions(self, request, context):
        self.log_msg(msg="(2)> SearchSolutions invoked (ta2trace) in, with no id parameter")

        search_id = self.SearchSolutionsEntryPoint(request)

        self.log_msg(msg="(2)< SearchSolutions returning response with search_id: %s (ta2trace) out" % self.search_id)
        return SearchSolutionsResponse(search_id=search_id)


    '''
    This has the non GRPC Search Solutions logic
    '''
    def SearchSolutionsEntryPoint(self, request):
        _logger.info("Entire Message: %s" % request)

        # Create id audit trail
        search_id = self.generateId()
        self.search_id = search_id
        self.id_tracker.add_new_search_id(search_id)

        ta3_preamble = None

        if request.template is not None:
            decoded_pipeline = decode_pipeline_description(request.template, pipeline_module.Resolver())
            if decoded_pipeline is not None:
                is_a_placeholder = decoded_pipeline.has_placeholder()

                if is_a_placeholder:
                    _logger.info("Placeholder pipeline template provided")
                    template = request.template
                    ta3_preamble = MessageConverter().message_to_pipeline(pipeline_message=template)
                else:
                    _logger.info("Fully specified pipeline template provided")
                    # Associate this id with the fact it is a fully specified pipeline
                    self.id_tracker.search_id_is_fully_specified(search_id)
                    self.fully_specified_pipeline = decoded_pipeline

                    # Create new solution id for tracking purposes
                    solution_id = self.generateId()
                    self.id_tracker.add_solution_id_to_search(search_id, solution_id)
                    self.id_tracker.set_solution_id_is_fully_specified(solution_id)

                    uri = request.inputs[0].csv_uri
                    _logger.info("CSV uri is: %s" % uri)
                    df = pandas.read_csv(uri)
                    self.inputs = DataFrame(df, generate_metadata=True)

                    # Nothing more to do here - we have been provided a fully specified pipeline to run
                    return search_id

        _logger.info("Problem Target specified for search")
        # Convert the TA3TA2-api metric enumeration to the TA2 form
        ta2_metric = self.resolveMetricForTA2(request)
        # Convert the TA3TA2-api taskType enumeration to the TA2 form
        ta2_taskType = self.resolveTaskType(request)

        _logger.info("Dataset URI is: %s" % request.inputs[0].dataset_uri)

        config_dict = dict(
            # The problem id is no longer provided, instead the D3MOUTPUTDIR environment variable should be set. Just
            # in case it is not, lets use the dataset_id as a fallback.
            problemId=request.problem.inputs[0].dataset_id,
            metric=ta2_metric,
            # TODO: Look for and populate this field
            pos_label=None,
            taskType=ta2_taskType,
            datasetSchema=request.inputs[0].dataset_uri[7:],
            # This is a default in case the D3MTIMEOUT environment variable is not provided
            timeout=30,
            cpus=1,
            targetName=request.problem.inputs[0].targets[0].column_name,
            cr_backend=self.mode,
            problem=request.problem
        )
        _logger.info("Mode is %s" % self.mode)
        _logger.info("Dataset URI after trimming is: %s" % config_dict.get("datasetSchema"))

        # Initialize the TA2 config
        self.config = autoflowconfig.AutoflowConfig(config_dict=config_dict)
        # Based on the problem spec select an optimizer to use
        self.optimizer = self.config.select_optimizer()
        self.optimizer._initialize_start_time()

        # Create new solution ids for tracking purposes
        solution_count = self.optimizer.get_pipeline_count()
        solution_ids = []
        for i in range (0, solution_count):
            solution_id = self.generateId()
            self.id_tracker.add_solution_id_to_search(search_id, solution_id)
            solution_ids.append(solution_id)

        self.handle = self.optimizer.spinoff_fit(search_id, solution_ids, ta3_preamble)

        return search_id


    '''
    Get all solutions presently identified by the search and start receiving any further solutions also found 
    as well.
        Input: GetSearchSolutionsResultsRequest 
        Output: GetSearchSolutionsResultsResponse
        Streams response to TA3
        Ids: Given the search_id from the SearchSolutions call, return a solution_id, one for each running pipeline
    '''
    def GetSearchSolutionsResults(self, request, context):
        self.log_msg(msg="(3)> GetSearchSolutionsResults invoked with search_id: %s (ta2trace) in" % request.search_id)

        # Check to see if this search_id is associated with a fully specified pipeline - if so special handling is required
        if self.id_tracker.search_id_fully_specified(request.search_id):
            timestamp = Timestamp()
            solution_id = self.id_tracker.get_solution_id_for_search_id(request.search_id)
            self.log_msg(msg="\t\t(3)< GetSearchSolutionsResults for search_id (FSId): %s. State: %s. Returning response with solution_id: %s (ta2trace) out" % (
                    request.search_id, "COMPLETED", solution_id))
            yield GetSearchSolutionsResultsResponse(
                        progress=Progress(state=core_pb2.COMPLETED,
                                          status="COMPLETED",
                                          start=timestamp,
                                          end=timestamp),
                        done_ticks=0,
                        all_ticks=0,
                        solution_id=solution_id,
                    )
        # Regular pipeline search
        else:
            state = None
            while (state != core_pb2.ProgressState.Name(core_pb2.COMPLETED)):

                if not self.id_tracker.contains_search_id(request.search_id):
                    # End Search Solutions has been invoked, exit the GetSearchSolutionsResults loop
                    break

                for pipeline in self.optimizer.running_pipelines():
                    # Check to see if the search process in question is still active
                    if (self.id_tracker.contains_search_id(request.search_id)):
                        state = self.resolveStateForTA3(pipeline.get("state"))
                        solution = GetSearchSolutionsResultsResponse(
                            progress=Progress(state=state,
                                              status=pipeline.get("status"),
                                              start=pipeline.get("start"),
                                              end=pipeline.get("end")),
                            done_ticks=pipeline.get("done_ticks"),
                            all_ticks=pipeline.get("all_ticks"),
                            solution_id=pipeline.get("solution_id"),
                            internal_score=pipeline.get("score"),
                            # TODO: Issue https://gitlab.com/daraghhartnett/tpot-ta2/issues/24 to figure
                            # out what values to use
                            scores=[SolutionSearchScore(
                                scoring_configuration=ScoringConfiguration(
                                    method="HOLDOUT",
                                    folds=10,
                                    train_test_ratio=0.8,
                                    shuffle=False,
                                    stratified=False
                                ),
                                scores=[Score(metric=ProblemPerformanceMetric(metric="RANK",
                                                                          k=0,
                                                                          pos_label="0"),
                                          fold=0,
                                          value=Value(
                                              raw=value_pb2.ValueRaw(
                                                  double=pipeline.get("rank")
                                              )
                                          ))]
                            )]
                        )
                        self.log_msg(msg="\t\t(3)< GetSearchSolutionsResults for search_id: %s. State: %s. Returning response with solution_id: %s (ta2trace) out" % (request.search_id, state, solution.solution_id))
                        # Add some logging here to see what path is being followed
                        yield solution
                    else:
                        _logger.info("\t\tsearch_id: %s removed from active searches" % request.search_id)
                        break
                time.sleep(self.update_interval)
        _logger.info("(3)< GetSearchSolutionsResults EXITED!")


    '''
    Request solution to be scored given inputs. Inputs have to be Dataset container values and pipeline outputs
    have to be predictions. It can internally run multiple fit + produce runs of the pipeline on permutations 
    of inputs data (e.g., for cross-validation). This is also why we cannot expose outputs here.
        Input: ScoreSolutionRequest 
        Output: ScoreSolutionResponse
        Non streaming
        Ids: Given a solution_id (identifies an individual pipeline search), return a request id for that solution_id
    '''
    def ScoreSolution(self, request, context):
        self.log_msg(msg="(4)> ScoreSolution invoked with solution_id: %s (ta2trace) in" % request.solution_id)

        _logger.info("\t\tScoreSolutionRequest data packet: everything: %s" % str(request))

        # Generate valid request id 22 characters long for TA3 tracking
        request_id = self.generateId()

        # Link the solution_id with this ScoreSolution::request_id
        self.id_tracker.link_solution_id_to_score_request_id(request.solution_id, request_id)

        # Register the metric to use with the request_id
        metric = request.performance_metrics[0].metric
        self.id_tracker.link_metric_to_score_request_id(metric, request_id)

        self.log_msg(msg="(4)< ScoreSolution for solution_id: %s returning response with request_id: %s (ta2trace) out" % (request.solution_id, request_id))
        return ScoreSolutionResponse(
            request_id=request_id
        )


    '''
    Get all score results computed until now and start receiving any new score results computed as well.
        Input: GetScoreSolutionResultsRequest 
        Output: GetScoreSolutionResultsResponse
        Streams response to TA3
        Ids: Given a request_id created by the call to ScoreSolution, return a pipeline status for the corresponding 
             solution_id
    '''
    def GetScoreSolutionResults(self, request, context):
        self.log_msg(msg="(5)> GetScoreSolutionResults invoked with request_id: %s (ta2trace) in" % request.request_id)

        # Get the solution_id that goes with this request_id
        solution_id = self.id_tracker.get_solution_id_for_score_request_id(request_id=request.request_id)
        metric = self.id_tracker.get_metric_for_score_request_id(request_id=request.request_id)

        state = None
        while (state != core_pb2.ProgressState.Name(core_pb2.COMPLETED)):
            pipeline = self.optimizer.completed_pipeline(solution_id=solution_id)

            state = self.resolveStateForTA3(pipeline.get("state"))
            if metric == 'RANK':
                score = GetScoreSolutionResultsResponse(
                        progress=Progress(state=state,
                                          status=pipeline.get("status"),
                                          start=pipeline.get("start"),
                                          end=pipeline.get("end")),
                        scores=[Score(metric=ProblemPerformanceMetric(metric="RANK",
                                                                      k=0,
                                                                      pos_label="0"),
                                      fold=0,
                                      value=Value(
                                          raw=value_pb2.ValueRaw(
                                              double=pipeline.get("rank")
                                          )
                                      ))]
                    )
            else:
                ta2_metric = self.resolveMetricForTA3(pipeline.get("metric"))
                score = GetScoreSolutionResultsResponse(
                        progress=Progress(state=state,
                                          status=pipeline.get("status"),
                                          start=pipeline.get("start"),
                                          end=pipeline.get("end")),
                        scores=[Score(metric=ProblemPerformanceMetric(metric=ta2_metric,
                                                                      k=0,
                                                                      pos_label="0"),
                                      fold=0,
                                      value=Value(
                                          raw=value_pb2.ValueRaw(
                                              double=pipeline.get("score")
                                          )
                                      ))]
                    )
            self.log_msg(msg="\t\t(5)< GetScoreSolutionResults for request_id: %s . State: %s. returning response (ta2trace) out" % (request.request_id, state))
            yield score
            time.sleep(self.update_interval)
        _logger.info("(5)< GetScoreSolutionResults EXITED!")


    '''
    Fit the solution on given inputs. If a solution is already fitted on inputs this is a NOOP (if no 
    additional outputs should be exposed). This can happen when a TA2 simultaneously fits the solution as 
    part of the solution search phase.
        Input: FitSolutionRequest 
        Output: FitSolutionResponse
        Non streaming
    '''
    def FitSolution(self, request, context):
        self.log_msg(msg="(7)> FitSolution invoked with solution_id: %s (ta2trace) in" % request.solution_id)

        _logger.info("\t\tFitSolutionRequest data packet: everything: %s" % str(request))

        # Fitting has already occurred so there is no need to pass the data down
        # if self.fully_specified_pipeline is not None:
        if self.id_tracker.solution_id_is_fully_specified(request.solution_id):
            self.fitted_pipeline = Runtime(self.fully_specified_pipeline, context=Base.Context.TESTING, volumes_dir=self.static_volumes)
            self.fit_result = self.fitted_pipeline.fit(inputs=[self.inputs])

        # Create request_id for this for request and link it to the solution_id
        request_id = self.generateId()
        self.id_tracker.link_fit_request_id_to_solution_id(request_id, request.solution_id)

        fitSolutionResponse = FitSolutionResponse(
            request_id =request_id
        )

        self.log_msg(msg="(7)< FitSolution for solution_id: %s returning response with request_id: %s (ta2trace) out" % (request.solution_id, request_id))
        return fitSolutionResponse


    '''
    Get all fitted results currently available and start receiving any further new fitted results
    as well with the specified request_id
        Input: GetFitSolutionResultsRequest 
        Output: GetFitSolutionResultsResponse
        Streams response to TA3
    '''
    def GetFitSolutionResults(self, request, context):
        self.log_msg(msg="(8)> GetFitSolutionResults invoked with request_id: %s (ta2trace) in" % request.request_id)

        # Get the solution_id that goes with this request_id - we will generate a fitted_solution_id to go with this
        # when the run state is COMPLETED.
        solution_id = self.id_tracker.get_solution_id_for_fit_request_id(request_id=request.request_id)

        # if self.fit_result is not None:
        if self.id_tracker.solution_id_is_fully_specified(solution_id):
            # Next get these output paths set to something appropriate
            output = "%s/%s-%s" % (self.temp, solution_id, "output.csv")
            with open(output, 'w') as output_file:
                output_file.write(self.fit_result.values['outputs.0'].to_csv(sep=' ', index=False, header=True))
            exposed_output_map = {"outputs.0": Value(csv_uri="file://%s" % output)}

            timestamp = Timestamp()
            # Create and register a fitted_solution_id associated with this search_id
            fitted_solution_id = self.generateId()
            self.id_tracker.link_fitted_solution_id_to_solution_id(fitted_solution_id, solution_id)
            self.log_msg(
                msg="\t\t(8)< GetFitSolutionResults for request_id (FSId): %s. State: %s. Returning response fitted_solution_id: %s (ta2trace) out" % (
                request.request_id, core_pb2.COMPLETED, fitted_solution_id))

            yield GetFitSolutionResultsResponse(
                    progress =  Progress(
                        state = core_pb2.COMPLETED,
                        status = "COMPLETED",
                        start = timestamp,
                        end = timestamp
                    ),
                    # We may want to expose some more of the pipeline steps progress but reflecting the overall progress
                    # will be good for now
                    steps = [core_pb2.StepProgress(
                        progress = Progress(
                            state=core_pb2.COMPLETED,
                            status="COMPLETED",
                            start=timestamp,
                            end=timestamp
                        ),
                        # steps = None
                    )],
                    exposed_outputs = exposed_output_map,
                    # This will be None until the state reaches COMPLETE
                    fitted_solution_id = fitted_solution_id
                )
        else:
            state = None
            fitted_solution_id = None
            while (state != core_pb2.ProgressState.Name(core_pb2.COMPLETED)):
                fitted_pipeline = self.optimizer.fitted_result(solution_id)

                state = self.resolveStateForTA3(fitted_pipeline.get("state"))
                if (fitted_pipeline.get("state") == "completed"):
                    if fitted_solution_id is None:
                        # Create and register a fitted_solution_id associated with this search_id
                        fitted_solution_id = self.generateId()
                        self.id_tracker.link_fitted_solution_id_to_solution_id(fitted_solution_id, solution_id)
                    output = "%s/%s-%s" % (self.temp, fitted_solution_id, "output.csv")
                    with open(output, 'w') as output_file:
                        output_file.write(fitted_pipeline.get("fit_result").values['outputs.0'].to_csv(sep=' ', index=False, header=True))
                    exposed_output_map = {"outputs.0": Value(csv_uri="file://%s" % output)}

                fitted_result = GetFitSolutionResultsResponse(
                    progress =  Progress(
                        state = state,
                        status = fitted_pipeline.get("status"),
                        start = fitted_pipeline.get("start"),
                        end = fitted_pipeline.get("start")
                    ),
                    # We may want to expose some more of the pipeline steps progress but reflecting the overall progress
                    # will be good for now
                    steps = [core_pb2.StepProgress(
                        progress = Progress(
                            state=self.resolveStateForTA3(fitted_pipeline.get("state")),
                            status=fitted_pipeline.get("status"),
                            start=fitted_pipeline.get("start"),
                            end=fitted_pipeline.get("start")
                        ),
                        # steps = None
                    )],
                    exposed_outputs = exposed_output_map,
                    # This will be None until the state reaches COMPLETE
                    fitted_solution_id = fitted_solution_id
                )
                self.log_msg(msg="\t\t(8)< GetFitSolutionResults for request_id: %s. State: %s. Returning response fitted_solution_id: %s (ta2trace) out" % (request.request_id, state, fitted_solution_id))
                yield fitted_result
                time.sleep(self.update_interval)


    '''
    Produce (execute) the solution on given inputs. A solution has to have been fitted for this to be possible 
    (even if in cases where this is just created by transformations).
        Input: ProduceSolutionRequest 
        Output: ProduceSolutionResponse
        Non streaming
    '''
    def ProduceSolution(self, request, context):
        self.log_msg(
            msg="(9)> ProduceSolution invoked with fitted_solution_id: %s (ta2trace) in" % request.fitted_solution_id)

        _logger.info("\t\tProduceSolutionRequest data packet: all: %s" % str(request))

        # Create and link a new request_id with the fitted_solution_id
        request_id = self.generateId()
        self.id_tracker.link_produce_request_id_with_fitted_solution_id(request_id, request.fitted_solution_id)

        # Get the solution_id that goes with this fitted_solution_id - this is used to get the index of the
        # pipeline that we want
        _logger.info("ProduceSolution: Requesting solution_id for request.fitted_solution_id: %s" % request.fitted_solution_id)
        solution_id = self.id_tracker.get_solution_id_for_fitted_solution_id(request.fitted_solution_id)

        self.id_tracker.link_produce_request_id_with_solution_id(request_id, solution_id)

        # Input to run produce on:
        dataset_schema = request.inputs[0].dataset_uri

        # if self.fit_result is not None:
        if self.id_tracker.solution_id_is_fully_specified(solution_id):
            # self.pipeline_result = self.fit_result.produce(dataset_schema)
            self.pipeline_result = self.fitted_pipeline.produce(inputs=[self.inputs])
        else:
            # Send in the data to produce plus the solution id which tells us which pipeline to run
            self.optimizer.spinoff_produce_solution(dataset_schema, solution_id)

        produceSolutionResponse = ProduceSolutionResponse(
            request_id = request_id
        )

        self.log_msg(
            msg="(9)< ProduceSolution for fitted_solution_id: %s returning reponse with request_id: %s (ta2trace) out" % (request.fitted_solution_id, request_id))
        return produceSolutionResponse


    '''
    Get all producing results computed until now and start receiving any new producing results computed 
    as well with the supplied request_id
        Input: GetProduceSolutionResultsRequest 
        Output: GetProduceSolutionResultsResponse
        Streams response to TA3
    '''
    def GetProduceSolutionResults(self, request, context):
        self.log_msg(msg="(10)> GetProduceSolutionResults invoked with request_id: %s (ta2trace) in" % request.request_id)

        timestamp = Timestamp()
        start = timestamp
        end = timestamp
        start.GetCurrentTime()
        end.GetCurrentTime()

        solution_id = self.id_tracker.get_solution_id_for_produce_request_id(request.request_id)

        # if self.pipeline_result is not None:
        if self.id_tracker.solution_id_is_fully_specified(solution_id):
            output = "%s/%s-%s" % (self.temp, solution_id, "output.csv")
            with open(output, 'w') as output_file:
                output_file.write(self.pipeline_result.values['outputs.0'].to_csv(sep=' ', index=False, header=True))
            exposed_output_map = {"outputs.0": Value(csv_uri="file://%s" % output)}

            solution = GetProduceSolutionResultsResponse(
                progress=Progress(
                    state="COMPLETED",
                    status="successful",
                    start=start,
                    end=end
                ),
                steps=[core_pb2.StepProgress(
                    progress=Progress(
                        state=core_pb2.COMPLETED,
                        status="successful",
                        start=start,
                        end=end
                    ),
                )],
                exposed_outputs=exposed_output_map
            )
            self.log_msg(msg="\t\t(10)< GetProduceSolutionResults for request_id (FSId): %s (ta2trace) out" % request.request_id)
            yield solution
        else:
            if solution_id in self.optimizer.solution_ids_to_predictions:
                pipeline_name = self.optimizer.solution_ids_to_predictions[solution_id]

                exposed_output_map = {"outputs.0": Value(csv_uri="file://%s" % pipeline_name)}

                for i in range(0, 1):
                    solution = GetProduceSolutionResultsResponse(
                        progress=Progress(
                            state="COMPLETED",
                            status="successful",
                            start=start,
                            end=end
                        ),
                        steps = [core_pb2.StepProgress(
                            progress = Progress(
                                state=core_pb2.COMPLETED,
                                status="successful",
                                start=start,
                                end=end
                            ),
                        )],
                        exposed_outputs=exposed_output_map
                    )
                    self.log_msg(msg="\t\t(10)< GetProduceSolutionResults for request_id: %s (ta2trace) out" % request.request_id)
                    yield solution
            else:
                for i in range(0, 1):
                    solution = GetProduceSolutionResultsResponse(
                        progress=Progress(
                            state="ERRORED",
                            status="failed",
                            start=start,
                            end=end
                        ),
                        steps = [core_pb2.StepProgress(
                            progress = Progress(
                                state="ERRORED",
                                status="failed",
                                start=start,
                                end=end
                            ),
                        )],
                    )
                    self.log_msg(msg="\t\t(10)< GetProduceSolutionResults failed for request_id: %s (ta2trace) out" % request.request_id)
                    yield solution


    '''
    Ends the search and releases all resources associated with the solution search.
    If the call is made in parallel with a running search and results are being streamed,
    the search is stopped and the "GetSearchSolutionsResults" stream is closed by TA2
    (as happens when the search is concluded on its own, or when a search is stopped
    by "StopSearchSolutions"). Found solution IDs during the search are no longer valid
    after this call.  
    All TA2's should support and respect this request as part of minimum (baseline) performance.
        Input: EndSearchSolutionsRequest 
        Output: EndSearchSolutionsResponse
        Non streaming
    '''
    def EndSearchSolutions(self, request, context):
        self.log_msg(msg="(11)> EndSearchSolutions invoked with search_id: %s (ta2trace) in" % request.search_id)

        # Kill the search associated with search_id
        self.id_tracker.remove_search_id(request.search_id)
        self.optimizer.end_search(request.search_id)

        self.log_msg(msg="(11)< EndSearchSolutions returning for search_id: %s (ta2trace) out" % request.search_id)
        return EndSearchSolutionsResponse()


    '''
    Exports a solution for evaluation purposes based on NIST specifications specified by fitted_solution_id
        Input: SolutionExportRequest 
        Output: SolutionExportResponse
        Non streaming
    '''
    def SolutionExport(self, request, context):
        self.log_msg(msg="(12)> SolutionExport invoked with solution_id: %s and rank: %s (ta2trace) in" %
                         (request.solution_id, request.rank))
        try:
            self.optimizer.spinoff_export(request.solution_id, request.rank)
        except Exception as e:
            _logger.info("Caught expected TA2 bug that would cause the TA3 to die: %s" % e)
            _logger.warning(traceback.format_exc())

        self.log_msg(msg="(12)< SolutionExport returning for solution_id: %s and rank: %s (ta2trace) out" %
                         (request.solution_id, request.rank))
        return SolutionExportResponse()


    '''
    Stops the search but leaves all currently found solutions available.
    If the call is made in parallel with a running search and results are being streamed,   
    the "GetSearchSolutionsResults" stream is closed by the TA2 (as happens when the search
    is concluded on its own). Search cannot be re-started after it has been stopped.
        Input: StopSearchSolutionsRequest 
        Output: StopSearchSolutionsResponse
        Non streaming
    '''
    def StopSearchSolutions(self, request, context):
        self.log_msg(msg="(13)< StopSearchSolutions invoked with search_id: %s (ta2trace) in" % request.search_id)

        if not self.id_tracker.search_id_is_fully_specified(request.search_id):
            # This basically does the same as EndSearchSolutions but it will do for now
            self.optimizer.end_search(search_id=request.search_id)

        # Remove this is from active searches so the GetSearchSolutionsResults exits as required
        self.id_tracker.remove_search_id(request.search_id)

        self.log_msg(msg="(13)> StopSearchSolutions returning for search_id: %s (ta2trace) out" % request.search_id)
        return StopSearchSolutionsResponse()


    '''
    Request a detailed description of the found solution.
        Input: DescribeSolutionRequest 
        Output: DescribeSolutionResponse
        Non streaming
        
        pipeline_obj = pipeline.Pipeline.from_json(description)
        allowed = [utils.ValueType.RAW,utils.ValueType.CSV_URI,
                   utils.ValueType.DATASET_URI,
                   utils.ValueType.PICKLE_BLOB,
                   utils.ValueType.PICKLE_URI,
                  ]
        pipeline_description = utils.encode_pipeline_description(pipeline_obj, allowed, '.')
        msg = core_pb2.DescribeSolutionResponse(
            pipeline=pipeline_description)
        return msg
    '''
    def DescribeSolution(self, request, context):
        self.log_msg(msg="(15)< DescribeSolution invoked with solution_id: %s (ta2trace) in" % request.solution_id)
        import traceback

        pipeline = self.optimizer.get_pipeline(solution_id=request.solution_id)
        encoded_pipeline_description = None
        if pipeline is not None:
            try:
                _logger.info("Pipeline we want to convert to json: %s" % pipeline[0])
                if self.id_tracker.solution_id_is_fully_specified(request.solution_id):
                    # Hmm - maybe this blows up when it is a provided pipeline?
                    _logger.info("Fully specified!")
                pipeline_json = pipeline[0].json()
                _logger.info("Pipeline json: %s" % pipeline_json)
                pipeline_object = pipeline_module.Pipeline.from_json(pipeline_json)

                allowed = ["RAW",
                           "CSV_URI",
                           "DATASET_URI"]
                encoded_pipeline_description = encode_pipeline_description(pipeline_object, allowed, '.')
            except Exception as e:
                _logger.error("Exception encountered while calling Describe Solutions %s", e)
                _logger.warning(traceback.format_exc())


        describeSolutionResponse = DescribeSolutionResponse(
            pipeline = encoded_pipeline_description
        )

        self.log_msg(msg="(15)> DescribeSolution returning for solution_id: %s (ta2trace) out" % request.solution_id)
        return describeSolutionResponse


    '''
    List all primitives known to TA2, their IDs, versions, names, and digests. Using this information 
    a TA3 should know which primitives may be put into a pipeline template. To narrow down potential primitives 
    to use a TA3 can also ask a TA2 to do a solution search and then observe which primitives the TA2 is using.
    If more metadata about primitives is needed, then a TA3 can use the results of this call to map primitives 
    to metadata (from Python code or primitive annotations) on its own.
        Input: ListPrimitivesRequest 
        Output: ListPrimitivesResponse
        Non streaming
    '''
    def ListPrimitives(self, request, context):
        self.log_msg(msg="(16)< ListPrimitives invoked (ta2trace) in")

        primitives_info = []
        primitives = d3m.index.search()
        for primitive in primitives:
            try:
                primitive_obj = d3m.index.get_primitive(primitive)
            except:
                continue
            if hasattr(primitive_obj, 'metadata'):
                id = primitive_obj.metadata.query()['id']
                version = primitive_obj.metadata.query()['version']
                python_path = primitive_obj.metadata.query()['python_path']
                name = primitive_obj.metadata.query()['name']
                digest = None
                if hasattr(primitive_obj.metadata.query(), 'digest'):
                    digest = primitive_obj.metadata.query()['digest']
                primitive_info = {
                    'id': id,
                    'version': version,
                    'python_path': python_path,
                    'name': name,
                    'digest': digest
                }
                primitives_info.append(primitive_info)
        primitives = ListPrimitivesResponse(primitives=primitives_info)

        self.log_msg(msg="(16)> ListPrimitives returning (ta2trace) out")
        return primitives


    '''
    Handy method for generating pipeline trace logs
    '''
    def log_msg(self, msg):
        msg = str(msg)
        for line in msg.splitlines():
            _logger.info("    | %s" % line)
        _logger.info("    \\_____________")


    '''
    Convenience method for generating 22 character id's
    '''
    def generateId(self):
        id = ''.join(random.choice(string.ascii_uppercase + string.digits) for _ in range(22))
        while (self.id_tracker.id_taken(id)):
            id = ''.join(random.choice(string.ascii_uppercase + string.digits) for _ in range(22))
        return id


    '''
    Convenience method for converting the TA3 Metric Enumerated value into a TA2 representation
    '''
    def resolveMetricForTA2(self, request):
        metric_name = request.problem.problem.performance_metrics[0].metric
        _logger.info("TA3 metric is %s", metric_name)
        ta2_metric = self.resolve_metric_to_ta2.get(metric_name)
        _logger.info("TA3 metric is %s, converted to TA2 metric: %s", metric_name, ta2_metric)
        return ta2_metric


    '''
    Convenience method for converting the TA2 metric representation into a TA3 Metric Enumerated value
    '''
    def resolveMetricForTA3(self, metric):
        _logger.info("resolving metric: %s" % metric)
        ta3_metric_name = self.resolve_metric_to_ta3.get(metric)
        _logger.info("resolving metric name: %s" % ta3_metric_name)
        _logger.info("TA2 metric is %s, converted to TA3 metric: %s", metric, ta3_metric_name)
        return ta3_metric_name


    '''
    Convenience method for converting the TaskType TA3 Enumerated value into a TA2 representation
    '''
    def resolveTaskType(self, request):
        taskkeyword_tasktype_set = {
            "CLASSIFICATION",
            "REGRESSION",
            "LINK_PREDICTION",
            "VERTEX_NOMINATION",
            "VERTEX_CLASSIFICATION",
            "COMMUNITY_DETECTION",
            "GRAPH_MATCHING",
            "FORECASTING",
            "COLLABORATIVE_FILTERING",
            "OBJECT_DETECTION",
            "SEMISUPERVISED",
            # Compound problem types such as timeSeries/Classification/Forecasting get special handling below
        }

        decoded_problem = decode_problem_description(request.problem)

        task_keywords = decoded_problem['problem']['task_keywords']
        task_type = taskkeyword_tasktype_set.intersection(task_keywords)

        if 'TIME_SERIES' in task_keywords and 'FORECASTING' in task_keywords:
            self.task_type = 'timeSeriesForecasting'
            _logger.info("TA3 task_type is TIME_SERIES/FORECASTING, converted to TA2 task_type: %s" % self.task_type)
            return self.task_type
        elif 'TIME_SERIES' in task_keywords and 'CLASSIFICATION' in task_keywords:
            self.task_type = 'timeSeriesClassification'
            _logger.info("TA3 task_type is TIME_SERIES/CLASSIFICATION, converted to TA2 task_type: %s" % self.task_type)
            return self.task_type
        elif 'SEMISUPERVISED' in task_keywords and 'CLASSIFICATION' in task_keywords:
            self.task_type = 'semiSupervisedClassification'
            _logger.info("TA3 task_type is SEMISUPERVISED, converted to TA2 task_type: %s" % self.task_type)
            return self.task_type
        elif len(task_type) == 1:
            self.taskType_name = task_type.pop()
        elif len(task_type) == 0:
            _logger.warning("Supported Task Type is not found in task_keywords: %s from problemDoc" % task_keywords)
            self.taskType_name = None
        elif len(task_type) > 1:
            _logger.warning("More than one Supported Task Type Task Type found in task_keywords: %s from problemDoc."
                            " Continuing with the first one detected" % task_keywords)
            self.taskType_name = task_type.pop()

        ta2_taskType = self.resolve_taskType_to_ta2.get(self.taskType_name)
        _logger.info("TA3 task_type is %s, converted to TA2 task_type: %s", self.taskType_name, ta2_taskType)
        return ta2_taskType


    '''
    Convenience method for converting the State string back into a TA3 Enumerated value
    '''
    def resolveStateForTA3(self, state):
        ta3_state_index = self.resolve_state_to_ta3.get(state)
        ta3_state_name = core_pb2.ProgressState.Name(ta3_state_index)
        # _logger.info("TA2 state is %s, converted to TA3 taskType: %s", state, ta3_state_name)
        return ta3_state_name
