"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const cdk = require("@aws-cdk/cdk");
const scalable_target_1 = require("./scalable-target");
/**
 * Represent an attribute for which autoscaling can be configured
 *
 * This class is basically a light wrapper around ScalableTarget, but with
 * all methods protected instead of public so they can be selectively
 * exposed and/or more specific versions of them can be exposed by derived
 * classes for individual services support autoscaling.
 *
 * Typical use cases:
 *
 * - Hide away the PredefinedMetric enum for target tracking policies.
 * - Don't expose all scaling methods (for example Dynamo tables don't support
 *   Step Scaling, so the Dynamo subclass won't expose this method).
 */
class BaseScalableAttribute extends cdk.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        this.props = props;
        this.target = new scalable_target_1.ScalableTarget(this, 'Target', {
            serviceNamespace: this.props.serviceNamespace,
            scalableDimension: this.props.dimension,
            resourceId: this.props.resourceId,
            role: this.props.role,
            minCapacity: props.minCapacity !== undefined ? props.minCapacity : 1,
            maxCapacity: props.maxCapacity
        });
    }
    /**
     * Scale out or in based on time
     */
    doScaleOnSchedule(id, props) {
        this.target.scaleOnSchedule(id, props);
    }
    /**
     * Scale out or in based on a metric value
     */
    doScaleOnMetric(id, props) {
        this.target.scaleOnMetric(id, props);
    }
    /**
     * Scale out or in in order to keep a metric around a target value
     */
    doScaleToTrackMetric(id, props) {
        this.target.scaleToTrackMetric(id, props);
    }
}
exports.BaseScalableAttribute = BaseScalableAttribute;
//# sourceMappingURL=data:application/json;base64,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