"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
/**
 * Schedule for scheduled scaling actions
 */
class Schedule {
    /**
     * Construct a schedule from a literal schedule expression
     *
     * @param expression The expression to use. Must be in a format that Application AutoScaling will recognize
     */
    static expression(expression) {
        return new LiteralSchedule(expression);
    }
    /**
     * Construct a schedule from an interval and a time unit
     */
    static rate(interval, unit) {
        const unitStr = interval !== 1 ? `${unit}s` : unit;
        return new LiteralSchedule(`rate(${interval} ${unitStr})`);
    }
    /**
     * Construct a Schedule from a moment in time
     */
    static at(moment) {
        return new LiteralSchedule(`at(${formatISO(moment)})`);
    }
    /**
     * Create a schedule from a set of cron fields
     */
    static cron(options) {
        if (options.weekDay !== undefined && options.day !== undefined) {
            throw new Error(`Cannot supply both 'day' and 'weekDay', use at most one`);
        }
        const minute = fallback(options.minute, '*');
        const hour = fallback(options.hour, '*');
        const month = fallback(options.month, '*');
        const year = fallback(options.year, '*');
        // Weekday defaults to '?' if not supplied. If it is supplied, day must become '?'
        const day = fallback(options.day, options.weekDay !== undefined ? '?' : '*');
        const weekDay = fallback(options.weekDay, '?');
        return new LiteralSchedule(`cron(${minute} ${hour} ${day} ${month} ${weekDay} ${year})`);
    }
    constructor() {
    }
}
exports.Schedule = Schedule;
/**
 * What unit to interpret the rate in
 */
var TimeUnit;
(function (TimeUnit) {
    /**
     * The rate is in minutes
     */
    TimeUnit["Minute"] = "minute";
    /**
     * The rate is in hours
     */
    TimeUnit["Hour"] = "hour";
    /**
     * The rate is in days
     */
    TimeUnit["Day"] = "day";
})(TimeUnit = exports.TimeUnit || (exports.TimeUnit = {}));
class LiteralSchedule extends Schedule {
    constructor(expressionString) {
        super();
        this.expressionString = expressionString;
    }
}
function fallback(x, def) {
    return x === undefined ? def : x;
}
function formatISO(date) {
    if (!date) {
        return undefined;
    }
    return date.getUTCFullYear() +
        '-' + pad(date.getUTCMonth() + 1) +
        '-' + pad(date.getUTCDate()) +
        'T' + pad(date.getUTCHours()) +
        ':' + pad(date.getUTCMinutes()) +
        ':' + pad(date.getUTCSeconds());
    function pad(num) {
        if (num < 10) {
            return '0' + num;
        }
        return num;
    }
}
//# sourceMappingURL=data:application/json;base64,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