"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const cdk = require("@aws-cdk/cdk");
const applicationautoscaling_generated_1 = require("./applicationautoscaling.generated");
/**
 * Define a step scaling action
 *
 * This kind of scaling policy adjusts the target capacity in configurable
 * steps. The size of the step is configurable based on the metric's distance
 * to its alarm threshold.
 *
 * This Action must be used as the target of a CloudWatch alarm to take effect.
 */
class StepScalingAction extends cdk.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        this.adjustments = new Array();
        // Cloudformation requires either the ResourceId, ScalableDimension, and ServiceNamespace
        // properties, or the ScalingTargetId property, but not both.
        // https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-applicationautoscaling-scalingpolicy.html
        const resource = new applicationautoscaling_generated_1.CfnScalingPolicy(this, 'Resource', {
            policyName: props.policyName || this.node.uniqueId,
            policyType: 'StepScaling',
            scalingTargetId: props.scalingTarget.scalableTargetId,
            stepScalingPolicyConfiguration: {
                adjustmentType: props.adjustmentType,
                cooldown: props.cooldownSec,
                minAdjustmentMagnitude: props.minAdjustmentMagnitude,
                metricAggregationType: props.metricAggregationType,
                stepAdjustments: cdk.Lazy.anyValue({ produce: () => this.adjustments }),
            }
        });
        this.scalingPolicyArn = resource.refAsString;
    }
    /**
     * Add an adjusment interval to the ScalingAction
     */
    addAdjustment(adjustment) {
        if (adjustment.lowerBound === undefined && adjustment.upperBound === undefined) {
            throw new Error('At least one of lowerBound or upperBound is required');
        }
        this.adjustments.push({
            metricIntervalLowerBound: adjustment.lowerBound,
            metricIntervalUpperBound: adjustment.upperBound,
            scalingAdjustment: adjustment.adjustment,
        });
    }
}
exports.StepScalingAction = StepScalingAction;
/**
 * How adjustment numbers are interpreted
 */
var AdjustmentType;
(function (AdjustmentType) {
    /**
     * Add the adjustment number to the current capacity.
     *
     * A positive number increases capacity, a negative number decreases capacity.
     */
    AdjustmentType["ChangeInCapacity"] = "ChangeInCapacity";
    /**
     * Add this percentage of the current capacity to itself.
     *
     * The number must be between -100 and 100; a positive number increases
     * capacity and a negative number decreases it.
     */
    AdjustmentType["PercentChangeInCapacity"] = "PercentChangeInCapacity";
    /**
     * Make the capacity equal to the exact number given.
     */
    AdjustmentType["ExactCapacity"] = "ExactCapacity";
})(AdjustmentType = exports.AdjustmentType || (exports.AdjustmentType = {}));
/**
 * How the scaling metric is going to be aggregated
 */
var MetricAggregationType;
(function (MetricAggregationType) {
    /**
     * Average
     */
    MetricAggregationType["Average"] = "Average";
    /**
     * Minimum
     */
    MetricAggregationType["Minimum"] = "Minimum";
    /**
     * Maximum
     */
    MetricAggregationType["Maximum"] = "Maximum";
})(MetricAggregationType = exports.MetricAggregationType || (exports.MetricAggregationType = {}));
//# sourceMappingURL=data:application/json;base64,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