"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const cdk = require("@aws-cdk/cdk");
const applicationautoscaling_generated_1 = require("./applicationautoscaling.generated");
class TargetTrackingScalingPolicy extends cdk.Construct {
    constructor(scope, id, props) {
        if ((props.customMetric === undefined) === (props.predefinedMetric === undefined)) {
            throw new Error(`Exactly one of 'customMetric' or 'predefinedMetric' must be specified.`);
        }
        if (props.scaleInCooldownSec !== undefined && props.scaleInCooldownSec < 0) {
            throw new RangeError(`scaleInCooldown cannot be negative, got: ${props.scaleInCooldownSec}`);
        }
        if (props.scaleOutCooldownSec !== undefined && props.scaleOutCooldownSec < 0) {
            throw new RangeError(`scaleOutCooldown cannot be negative, got: ${props.scaleOutCooldownSec}`);
        }
        super(scope, id);
        const resource = new applicationautoscaling_generated_1.CfnScalingPolicy(this, 'Resource', {
            policyName: props.policyName || this.node.uniqueId,
            policyType: 'TargetTrackingScaling',
            scalingTargetId: props.scalingTarget.scalableTargetId,
            targetTrackingScalingPolicyConfiguration: {
                customizedMetricSpecification: renderCustomMetric(props.customMetric),
                disableScaleIn: props.disableScaleIn,
                predefinedMetricSpecification: props.predefinedMetric !== undefined ? {
                    predefinedMetricType: props.predefinedMetric,
                    resourceLabel: props.resourceLabel,
                } : undefined,
                scaleInCooldown: props.scaleInCooldownSec,
                scaleOutCooldown: props.scaleOutCooldownSec,
                targetValue: props.targetValue
            }
        });
        this.scalingPolicyArn = resource.refAsString;
    }
}
exports.TargetTrackingScalingPolicy = TargetTrackingScalingPolicy;
function renderCustomMetric(metric) {
    if (!metric) {
        return undefined;
    }
    const c = metric.toAlarmConfig();
    if (!c.statistic) {
        throw new Error('Can only use Average, Minimum, Maximum, SampleCount, Sum statistic for target tracking');
    }
    return {
        dimensions: c.dimensions,
        metricName: c.metricName,
        namespace: c.namespace,
        statistic: c.statistic,
        unit: c.unit
    };
}
/**
 * One of the predefined autoscaling metrics
 */
var PredefinedMetric;
(function (PredefinedMetric) {
    PredefinedMetric["DynamoDBReadCapacityUtilization"] = "DynamoDBReadCapacityUtilization";
    PredefinedMetric["DynamoDBWriteCapacityUtilization"] = "DynamoDBWriteCapacityUtilization";
    PredefinedMetric["ALBRequestCountPerTarget"] = "ALBRequestCountPerTarget";
    PredefinedMetric["RDSReaderAverageCPUUtilization"] = "RDSReaderAverageCPUUtilization";
    PredefinedMetric["RDSReaderAverageDatabaseConnections"] = "RDSReaderAverageDatabaseConnections";
    PredefinedMetric["EC2SpotFleetRequestAverageCPUUtilization"] = "EC2SpotFleetRequestAverageCPUUtilization";
    PredefinedMetric["EC2SpotFleetRequestAverageNetworkIn"] = "EC2SpotFleetRequestAverageNetworkIn";
    PredefinedMetric["EC2SpotFleetRequestAverageNetworkOut"] = "EC2SpotFleetRequestAverageNetworkOut";
    PredefinedMetric["SageMakerVariantInvocationsPerInstance"] = "SageMakerVariantInvocationsPerInstance";
    PredefinedMetric["ECSServiceAverageCPUUtilization"] = "ECSServiceAverageCPUUtilization";
    PredefinedMetric["ECSServiceAverageMemoryUtilization"] = "ECSServiceAverageMemoryUtilization";
})(PredefinedMetric = exports.PredefinedMetric || (exports.PredefinedMetric = {}));
//# sourceMappingURL=data:application/json;base64,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