import json
from dataclasses import dataclass, field, asdict

from cbaxter1988_utils.models import BaseModel

PROBLEM_DETAIL_HEADER_JSON = {
    "content-type": "application/problem+json"
}


@dataclass
class ProblemDetailModel(BaseModel):
    """


Based on Guidance published in RFC7807 - https://datatracker.ietf.org/doc/html/rfc7807


"type" (string) - A URI reference [RFC3986] that identifies the
problem type.  This specification encourages that, when
dereferenced, it provide human-readable documentation for the
problem type (e.g., using HTML [W3C.REC-html5-20141028]).  When
this member is not present, its value is assumed to be
"about:blank".

"title" (string) - A short, human-readable summary of the problem
type.  It SHOULD NOT change from occurrence to occurrence of the
problem, except for purposes of localization (e.g., using
proactive content negotiation; see [RFC7231], Section 3.4).

"status" (number) - The HTTP status code ([RFC7231], Section 6)
generated by the origin server for this occurrence of the problem.

"detail" (string) - A human-readable explanation specific to this
occurrence of the problem.

"instance" (string) - A URI reference that identifies the specific
occurrence of the problem.  It may or may not yield further
information if dereferenced.

The "status" member, if present, is only advisory; it conveys the
HTTP status code used for the convenience of the consumer.
Generators MUST use the same status code in the actual HTTP response,
to assure that generic HTTP software that does not understand this
format still behaves correctly.  See Section 5 for further caveats
regarding its use.

Consumers can use the status member to determine what the original
status code used by the generator was, in cases where it has been
changed (e.g., by an intermediary or cache), and when message bodies
persist without HTTP information.  Generic HTTP software will still
use the HTTP status code.


The "detail" member, if present, ought to focus on helping the client
correct the problem, rather than giving debugging information.

Consumers SHOULD NOT parse the "detail" member for information;
extensions are more suitable and less error-prone ways to obtain such
information.

Note that both "type" and "instance" accept relative URIs; this means
that they must be resolved relative to the document's base URI, as
per [RFC3986], Section 5.

    """
    type: str = field()
    type_base_uri: str = field(default='about:blank')
    title: str = field(default=None)
    status: int = field(default=None)
    detail: str = field(default=None)
    instance: str = field(default=None)

    header = PROBLEM_DETAIL_HEADER_JSON

    def __post_init__(self):
        self.type = f'{self.type_base_uri}/{self.type}'

    def to_json(self):
        return json.dumps(asdict(self))

    def to_xml(self):
        pass
