import json
import uuid
from flask import jsonify
from .api_dto import ApiDto


class MLModel(ApiDto):
    """
    A trained Machine Learning Model stored to be executed on demand.
    Can contain also a scaler, both object are stored as pickled file.

    :ivar model_id: The UUID of the ML Model.
    :ivar key: Logical String ID of the Model
    :ivar generatedById: The UUID of the Execution from which the ML Model was created.
    :ivar status: 'draft', 'valid' or 'invalid' - When generated model are tested by API, only valid model can be used.
    :ivar needExactColumnNumbers: True by default, define if the model requires exact columns numbers to be executed.
    :ivar needExactColumnNames: True by default, define if the model requires exact columns names to be executed.
    :ivar has_anomalies: False by default, define if a model generate potential anomalies.
    :ivar has_target_feat: False by default, define if a model requires a target features to be trained or executed.
    :ivar input_columns: list of all columns used to trained the model.
    :ivar output_columns: list of all columns generated by the model.
    :ivar label_counts: Count of labels generated by the model.
    :ivar trained_model: The Trained model
    :ivar scaler: The scaler
    :ivar experiment: The Linked experiment to the model.
    """

    def __init__(self, model_id=None, generated_by_id=None,
                 exact_names=True, exact_numbers=True,
                 key = None):
        if model_id is None:
            model_id = uuid.uuid4()
        self.model_id = model_id
        self.key = key

        self.generatedById = generated_by_id

        self.status = 'draft'
        self.input_columns = []
        self.output_columns = []

        self.needExactColumnNumbers = exact_numbers
        self.needExactColumnNames = exact_names
        self.has_anomalies = False
        self.label_counts = 0
        self.has_target_feat = False

        self.trained_model = None
        self.scaler = None

        self.experimentId = None

    def api_id(self) -> str:
        """
        Id of the ML Model (model_id)

        :return: string formatted UUID of the Model ID.
        """
        return str(self.model_id).upper()

    def endpoint(self) -> str:
        """
        Name of the endpoints used to manipulate ML Models.
        :return: Endpoint name.
        """
        return "MLModels"

    def to_json(self):
        obj = {"id": str(self.model_id),
               "status": str(self.status),
               "needExactColumnNames": str(self.needExactColumnNames),
               "needExactColumnNumbers": str(self.needExactColumnNumbers),
               "hasAnomalies": str(self.has_anomalies),
               "hasTargetFeat": str(self.has_target_feat),
               "labelCount": str(self.label_counts)
               }
        if self.key is not None:
            obj["key"] = str(self.key)
        if self.generatedById is not None:
            obj["generatedById"] = str(self.generatedById)
        if self.input_columns is not None:
            obj["inputColumns"] = json.dumps(list(self.input_columns))
        if self.output_columns is not None:
            obj["outputColumns"] = json.dumps(list(self.output_columns))
        if self.experimentId is not None:
            obj["experimentId"] = str(self.experimentId)
        return obj

    def from_json(self, obj):
        """
        Load the ML Model entity from a dictionary representation of the ML Model.

        :param obj: Dict version of the ML Model.
        """
        if "id" in obj.keys():
            self.model_id = obj["id"]
        if "key" in obj.keys() and obj["key"] is not None:
            self.key = obj["key"]
        if "generatedById" in obj.keys() and obj["generatedById"] is not None:
            self.generatedById = uuid.UUID(obj["generatedById"])
        if "experimentId" in obj.keys() and obj["experimentId"] is not None:
            self.experimentId = uuid.UUID(obj["experimentId"])
        if "status" in obj.keys():
            self.status = str(obj["status"]).lower()
        if "inputColumns" in obj.keys():
            self.input_columns = json.loads(obj["inputColumns"])
        if "outputColumns" in obj.keys():
            self.output_columns = json.loads(obj["outputColumns"])
        if "labelCount" in obj.keys():
            self.label_counts = int(obj["labelCount"])
        if "hasAnomalies" in obj.keys():
            self.has_anomalies = bool(obj["hasAnomalies"])
        if "hasTargetFeat" in obj.keys():
            self.has_target_feat = bool(obj["hasTargetFeat"])
        if "needExactColumnNumbers" in obj.keys():
            self.needExactColumnNumbers = bool(obj["needExactColumnNumbers"])
        if "needExactColumnNames" in obj.keys():
            self.needExactColumnNames = bool(obj["needExactColumnNames"])

    def get_sample_payload(self):
        """
        Get a JSON formatted sample payload to call the ML Model.
        :return: JSON formatted sample payload.
        """
        pl_columns = {"timestamp": "[timestamp]"}
        for hardwareId in self.input_columns:
            pl_columns[hardwareId] = "[" + hardwareId + "]"
        pl_json = {
            "id": str(self.model_id),
            "dataset": pl_columns
        }
        return pl_json

