# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['nuntius',
 'nuntius.admin',
 'nuntius.management',
 'nuntius.management.commands',
 'nuntius.migrations',
 'nuntius.models',
 'nuntius.utils']

package_data = \
{'': ['*'],
 'nuntius': ['locale/fr/LC_MESSAGES/*',
             'static/nuntius/*',
             'static/nuntius/mosaico/*',
             'static/nuntius/mosaico/fontawesome/*',
             'static/nuntius/mosaico/img/*',
             'static/nuntius/mosaico/lang/*',
             'static/nuntius/mosaico/notoregular/*',
             'static/nuntius/mosaico/skins/gray-flat/*',
             'static/nuntius/mosaico/skins/gray-flat/fonts/*',
             'static/nuntius/mosaico/skins/gray-flat/img/*',
             'static/nuntius/templates/versafix-1/*',
             'static/nuntius/templates/versafix-1/edres/*',
             'static/nuntius/templates/versafix-1/img/*',
             'static/nuntius/templates/versafix-1/img/icons/*',
             'static/nuntius/templates/versafix-1/img/social_def/*',
             'templates/nuntius/*']}

install_requires = \
['Django>=2.0', 'Pillow', 'django-stdimage', 'html2text', 'tenacity']

extras_require = \
{'anymail': ['django-anymail', 'boto3'],
 'mysql': ['mysqlclient'],
 'push': ['django-push-notifications>=2.0.0,<3.0.0',
          'apns>=2.0.1,<3.0.0',
          'pywebpush>=1.14.0,<2.0.0']}

setup_kwargs = {
    'name': 'nuntius',
    'version': '2.3.0b2',
    'description': 'A newsletter app to be used with Django or standalone.',
    'long_description': '# Nuntius\n\nNuntius is a newsletter / push notification campaign application for Django.\n\nNuntius integrates with your Django project.\nIt is very agnostic about your subscribers and subscriber lists models.\n\nIt features [Mosaico](https://mosaico.io/), a drag-and-drop email\neditor, for sending beautiful emails to your subscribers and \npush notification support through Apple Push Notification service (APNs) \nand Google Cloud Messaging (GCM).\n\n## How it works\n\nNuntius is agnostic about your subscribers model. You can use your current\nuse model, as long as it implements a few required methods.\n\nTo allow your end-users to choose recipients, it is your choice to implement\none or more "segment" models. Segment models implement a required method\n`get_subscribers_queryset`.\n\nYou can then create campaigns in the Django admin panel, and send them to\nexisting segments.\n\n## Installation\n\n1. Add "push_notifications" and "nuntius" to your INSTALLED_APPS setting like this:\n    ````python\n        INSTALLED_APPS = [\n            \'django.contrib.admin\',\n            \'django.contrib.auth\',\n            \'django.contrib.contenttypes\',\n            \'django.contrib.sessions\',\n            \'django.contrib.messages\',\n            \'django.contrib.staticfiles\',\n            ...\n            \'push_notifications\',\n            \'nuntius\',\n        ]\n    ````\n2. Include Nuntius urlconf in your project `urls.py` like this:\n    ````python\n        path(\'nuntius/\', include(\'nuntius.urls\')),\n    ````\n3. Define your subscriber model so it works with Nuntius.\n   You must inherit from\n   [`nuntius.models.AbstractSubscriber`](https://github.com/lafranceinsoumise/nuntius/blob/master/nuntius/models.py#L174)\n   and implement all the necessary methods. An easy way to do this is to use directly or to extend\n   [`BaseSubscriber`](https://github.com/lafranceinsoumise/nuntius/blob/master/nuntius/models.py#L204), but\n   you can implement the methods of `AbstractSubscriber` the way you want.\n    \n    Here are the methods you must implement :\n    \n    * `get_subscriber_status()`\n        must return one of `AbstractSubscriber.STATUS_CHOICES`. You can also simply\n        define a `subscriber_status` attribute.\n        \n    * `get_subscriber_email()`\n        must return a unique email address for the subscriber. You can also simply\n        define an `email` attribute.\n\n    * `get_subscriber_data()`\n        must return the dictionnary of values which can be used as substitution in\n        the emails. Default is `{"email": self.get_subscriber_email()}`.\n    \n    * `get_subscriber_push_devices()` (optional)\n        must return a list of `django-push-notifications.APNSDevice` \n        and `django-push-notifications.GCMDevice` model instances \n        (cf. [the `django-push-notifications` documentation](https://github.com/jazzband/django-push-notifications))\n        \n\n \n4. Tell Nuntius how to find your subscriber model in `settings.py`\n    ````python\n    NUNTIUS_SUBSCRIBER_MODEL = \'myapp.MySubscriberModel\'\n    ````\n\n5. Launch the nuntius worker in the background. In a production setting, this should be done through\n   a process monitor like upstart or systemd.\n    ```shell script\n    export DJANGO_SETTINGS_MODULE=myapp.settings\n    python ./manage.py nuntius_worker\n    ```\n\n6.  Unless you are using a custom admin site, admin panels for Nuntius will be\n[autodiscovered](https://docs.djangoproject.com/en/2.0/ref/contrib/admin/#discovery-of-admin-files)\nand added to you admin site. If you use a custom admin site, you need to register\nNuntius models with something like:\n\n    ```python\n    admin_site.register(nuntius.models.Campaign, nuntius.admin.CampaignAdmin)\n    admin_site.register(nuntius.models.CampaignSentEvent, nuntius.admin.CampaignSentEventAdmin)\n    ```\n\n## Other settings\nUse `NUNTIUS_DEFAULT_FROM_EMAIL`, `NUNTIUS_DEFAULT_FROM_NAME`, `NUNTIUS_DEFAULT_REPLY_TO_EMAIL`,\n`NUNTIUS_DEFAULT_REPLY_TO_NAME` to change default field values in the admin form.\n\nUse `NUNTIUS_ENABLED_CAMPAIGN_TYPES` to choose which types of campaign \nyou want to enable by default (`email` *default*, `push` or `email,push`)\n\nIn order to use push notifications, `NUNTIUS_PUSH_NOTIFICATION_SETTINGS` must be specified\n(cf. [the `django-push-notifications` documentation](https://github.com/jazzband/django-push-notifications#settings-list))\n```py\nNUNTIUS_PUSH_NOTIFICATIONS_SETTINGS = {\n    "FCM_API_KEY": "[your api key]",\n    "GCM_API_KEY": "[your api key]",\n    "APNS_CERTIFICATE": "/path/to/your/certificate.pem",\n    "APNS_TOPIC": "com.example.push_test",\n    # ...\n}\n```\n\n## Advanced usage\n\n### List segments\n\nIf you want to have more control on your recipients, you can create a\nsegment model.\n\nOne example of segment is a simple model which holds a Many-to-Many relation\nto subscribers.\n\nAnother example is a segment model which filters subscribers depending on\nthe date of their last login :\n\n```python\nfrom django.db import models\nfrom django.db.models import fields\nfrom datetime import datetime\n\nfrom nuntius.models import BaseSegment\n\n\nclass LastLoginDateSegment(BaseSegment, models.Model):\n     last_login_duration = fields.DurationField()\n     \n     def get_display_name(self):\n         return f\'Last login : {str(datetime.now() - self.last_login_duration)}\'\n         \n     def get_subscribers_queryset(self):\n        return MySubscriberClass.objects.filter(last_login__gt=datetime.now() - self.last_login_duration)\n        \n     def get_subscribers_count(self):\n        return MySubscriberClass.objects.filter(last_login__gt=datetime.now() - self.last_login_duration, subscribed=True)\n\n```\n \n* `get_subscribers_queryset` is allowed to return subscribers regardless of their\n    `subscriber_status`, as `get_subscriber_status` will be called on each instance.\n* `get_subscribers_count` is only there for convenience in the admin panel, it does not\n    have to be accurate. If you want to have it accurate, you should however take\n    your subscribers status into account.\n\nThen, add your segment model to Nuntius settings :\n````python\nNUNTIUS_SEGMENT_MODEL = \'myapp.lastlogindatesegment\'\n````\n\n### Custom template\n\nYou can write your own Mosaico template to fit your needs. To make it available in the admin,\nlist the public URL path of the template in `NUNTIUS_MOSAICO_TEMPLATES`. The template can be served\nby Django static files system, or any other way at your preference.\n\n```python\nNUNTIUS_MOSAICO_TEMPLATES = [\n    ("/static/mosaico_templates/versafix-2/template-versafix-2.html", "Custom template")\n]\n```\n\n### Sending parameters\n\nThe worker will spawn several subprocesses to speed up the sending of campaigns. The number of\nprocesses that will send emails concurrently can be configured using the `NUNTIUS_MAX_CONCURRENT_SENDERS`\nsetting.\n\nMost ESP enforce a maximum send rate. Nuntius won\'t sent messages faster than`NUNTIUS_MAX_SENDING_RATE`,\nin messages per second.\n\nWhen using SMTP, some ESP limit the number of emails that can be sent using a single connection.\n`NUNTIUS_MAX_MESSAGES_PER_CONNECTION` will force Nuntius to reset the connection after sending that\nmany messages.\n\nThe Nuntius worker checks every `NUNTIUS_POLLING_INTERVAL` seconds if any sending has been scheduled\nor canceled. The default value of 2 seconds should be find for most usages.\n\nTo help you configure these parameters, you can send SIGUSR1 to the main worker process and it will\nprint sending statistics on `stderr`. Pay special attention to the current sending rate and to the\ncurrent bucket capacity: if your sending rate is lower than the maximum you configured, it most\nlikely means the value you chose for `NUNTIUS_MAX_CONCURRENT_SENDERS` is not high enough given\nthe latency you\'re getting with your ESP.\n\n### ESP and Webhooks\n\nMaintaining your own SMTP server to send your newsletter is probably\na bad idea if you have more than a few subscribers. You can use\n[Anymail](https://github.com/anymail/django-anymail) along with Nuntius\nin order to use an email service provider. Anymail supports\na lot of ESP, like Amazon SES, Mailgun, Mailjet, Postmark, SendGrid,\nSendinBlue, or SparkPost.\n\nRefer to the steps in [Anymail 1-2-3](https://anymail.readthedocs.io/en/stable/quickstart/)\nto install Anymail. If you want to configure Anymail just for Nuntius and keep\nthe default email backend for other usage, you can use the setting `NUNTIUS_EMAIL_BACKEND`\nrather than the default `EMAIL_BACKEND`.\n\nIn addition, configuring Nuntius with Anymail will allow you to use ESP tracking features\nand to track status of your email once it is sent.\n\n#### Webhooks\n\nConfiguring webhhoks allows Nuntius to track email status and to\ngive you statistics on campaign, as well as updating subscriber status\nwhen they bounce.\n\n1. Configure email tracking as described in\n[Anymail documentation](https://anymail.readthedocs.io/en/stable/installation/#configuring-tracking-and-inbound-webhooks).\n2. Implement the method `set_subscriber_status(self, email, status)` on your subscriber\nmodel manager.\n\nNuntius will automatically listen to Anymail signals and call this method if needed.\n\n##### Handling of non-nuntius events (optional)\n\nIf you send emails to your subscribers by other means than Nuntius (for example,\ntransactional emails), you will receive webhooks events which are not related to\na campaign you sent. By default, Nuntius will create a campaign result event recording\nthe email and the event type, but it will not link it to a campaign nor to a subscriber\nmodel.\n\nIf you want your events to always be linked to a subscriber model, you must implement\na `get_subscriber(self, email_address)` method on your subscriber model manager.\n\n\n##### BaseSubscriberManager\n\nNuntius is packaged with a BaseSubscriberManager, which implements both\n`set_subscriber_status` and `get_subscriber`, assuming you have an `email` field\non your subscriber model. This is the default manager used by `BaseSubscriber`.\n\n\n#### Bounce handling\n\nMost ESP gives you a reputation based on your hard bounce rate.\nMosaico handles bounces smartly to change your subscribers status\nwhen necessary.\n\nIf Nuntius receive a bounce event on an email address which has no\nother sending event, `set_subscriber_status(email, status)` is called\nwith `AbstractSubscriber.STATUS_BOUNCED`.\n\nIf a successful sending event exists for this address,\nthree parameters are taken into account :\n* if during the last `duration` days, there has been no more bounces than `limit`\nand at least one successful sending, no action is taken\n* if there has been at least one successful sending in the last\n`consecutive` events, no action is taken\n* otherwise, `set_subscriber_status(email, status)` is called\nwith `AbstractSubscriber.STATUS_BOUNCED`\n\n\nYou can change thoses default values :\n```python\nNUNTIUS_BOUNCE_PARAMS = {\n    "consecutive": 1,\n    "duration": 7,\n    "limit": 3\n}\n```\n\n**Example :**\n\n* You send 3 campaigns a week. After a few months, a subscriber\nhas a full mailbox. On first and second bounced campaign, no action\nis taken because there is a successful sending in the last 7 days,\nand no more than 3 bounces. On the third campaign, if the user has empty\ntheir mailbox, everything is fine. Otherwise, the subscriber is marked\nas permanently bounced.\n* You send one campaign a day. A user has a buggy email server.\nThis week, the user has already 3 bounces. When you receive the 4th\nbounce, if there has been a successful sending just before,\neverything is fine. Otherwise, the subscriber is marked\nas permanently bounced.\n\n## Tracking\n\nOpening and clicks are tracked by adding a white pixel and replacing links in emails, and by using a proxy URL on\npush notification clicks.\n\nNuntius also adds [UTM parameters](https://en.wikipedia.org/wiki/UTM_parameters) to every URL with the following values:\n* `utm_source`: *"nuntius"*\n* `utm_medium`: *"email"*\n* `utm_campaign`: value configured by user at the campaign level\n* `utm_content`: *"link-{number}"* based on the link position in the email\n* `utm_term`: attribute `utm_term` of the segment object, or empty string if attribute does not exist\n\nIn some situations, two details may be important for you:\n\n1. `utm_campaign`, `utm_content`, and `utm_term`, those are just defaults values, and can also be set directly on\nthe link. `utm_source` and `utm_medium` will always be overwritten.\n2. `utm_content` and `utm_term` are set at sending time and cannot change afterwards. `utm_campaign` is\nset at click time, during the redirection from nuntius tracking URL to target URL, so if you change the value\nat the campaign level after sending, the value will change for all new clicks.\n\n## License\n\nCopyright is owned by Jill Royer and Arthur Cheysson.\n\nYou can use Nuntius under GPLv3 terms.\n',
    'author': 'Jill Royer',
    'author_email': 'hello@jillroyer.me',
    'maintainer': None,
    'maintainer_email': None,
    'url': 'https://github.com/lafranceinsoumise/nuntius',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'extras_require': extras_require,
    'python_requires': '>=3.6,<4.0',
}


setup(**setup_kwargs)
