"use strict";
/**
 *  Copyright 2021 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.createGlueDatabase = exports.createGlueTable = exports.createGlueJobRole = exports.deployGlueJob = exports.buildGlueJob = exports.SinkStoreType = void 0;
const glue = require("@aws-cdk/aws-glue");
const aws_iam_1 = require("@aws-cdk/aws-iam");
const aws_s3_1 = require("@aws-cdk/aws-s3");
const core_1 = require("@aws-cdk/core");
const defaults = require("../");
const utils_1 = require("./utils");
/**
 * Enumeration of data store types that could include S3, DynamoDB, DocumentDB, RDS or Redshift. Current
 * construct implementation only supports S3, but potential to add other output types in the future
 */
var SinkStoreType;
(function (SinkStoreType) {
    SinkStoreType["S3"] = "S3";
})(SinkStoreType = exports.SinkStoreType || (exports.SinkStoreType = {}));
function buildGlueJob(scope, props) {
    if (!props.existingCfnJob) {
        if (props.glueJobProps) {
            return deployGlueJob(scope, props.glueJobProps, props.database, props.table, props.outputDataStore);
        }
        else {
            throw Error('Either glueJobProps or existingCfnJob is required');
        }
    }
    else {
        return [props.existingCfnJob, aws_iam_1.Role.fromRoleArn(scope, 'ExistingRole', props.existingCfnJob.role)];
    }
}
exports.buildGlueJob = buildGlueJob;
function deployGlueJob(scope, glueJobProps, database, table, outputDataStore) {
    let _glueSecurityConfigName;
    if (glueJobProps.securityConfiguration === undefined) {
        _glueSecurityConfigName = 'ETLJobSecurityConfig';
        const _glueKMSKey = `arn:${core_1.Aws.PARTITION}:kms:${core_1.Aws.REGION}:${core_1.Aws.ACCOUNT_ID}:alias/aws/glue`;
        new glue.CfnSecurityConfiguration(scope, 'GlueSecurityConfig', {
            name: _glueSecurityConfigName,
            encryptionConfiguration: {
                jobBookmarksEncryption: {
                    jobBookmarksEncryptionMode: 'CSE-KMS',
                    kmsKeyArn: _glueKMSKey
                },
                s3Encryptions: [{
                        s3EncryptionMode: 'SSE-S3'
                    }]
            }
        });
    }
    else {
        _glueSecurityConfigName = glueJobProps.securityConfiguration;
    }
    const _glueJobPolicy = new aws_iam_1.Policy(scope, 'LogPolicy', {
        statements: [
            new aws_iam_1.PolicyStatement({
                effect: aws_iam_1.Effect.ALLOW,
                actions: ['logs:CreateLogGroup', 'logs:CreateLogStream', 'logs:PutLogEvents'],
                resources: [`arn:${core_1.Aws.PARTITION}:logs:${core_1.Aws.REGION}:${core_1.Aws.ACCOUNT_ID}:log-group:/aws-glue/*`]
            })
        ]
    });
    let _jobRole;
    if (glueJobProps.role) {
        _jobRole = aws_iam_1.Role.fromRoleArn(scope, 'JobRole', glueJobProps.role);
    }
    else {
        _jobRole = defaults.createGlueJobRole(scope);
    }
    _glueJobPolicy.attachToRole(_jobRole);
    let _outputLocation;
    if (outputDataStore !== undefined && outputDataStore.datastoreType === SinkStoreType.S3) {
        if (outputDataStore.existingS3OutputBucket !== undefined) {
            _outputLocation = [outputDataStore.existingS3OutputBucket, undefined];
        }
        else {
            _outputLocation = defaults.buildS3Bucket(scope, { bucketProps: outputDataStore.outputBucketProps });
        }
    }
    else {
        _outputLocation = defaults.buildS3Bucket(scope, {});
    }
    _outputLocation[0].grantReadWrite(_jobRole);
    const _jobArgumentsList = {
        "--enable-metrics": true,
        "--enable-continuous-cloudwatch-log": true,
        "--database_name": database.ref,
        "--table_name": table.ref,
        ...((outputDataStore === undefined || (outputDataStore && outputDataStore.datastoreType === SinkStoreType.S3)) &&
            { '--output_path': `s3a://${_outputLocation[0].bucketName}/output/` }),
        ...glueJobProps.defaultArguments
    };
    const _newGlueJobProps = utils_1.overrideProps(defaults.DefaultGlueJobProps(_jobRole, glueJobProps.command, _glueSecurityConfigName, _jobArgumentsList, glueJobProps.glueVersion), glueJobProps);
    let _scriptLocation;
    if (isJobCommandProperty(_newGlueJobProps.command)) {
        if (_newGlueJobProps.command.scriptLocation) {
            _scriptLocation = _newGlueJobProps.command.scriptLocation;
        }
        else {
            throw Error('Script location has to be provided as an s3 Url location. Script location cannot be empty');
        }
    }
    const _scriptBucketLocation = aws_s3_1.Bucket.fromBucketArn(scope, 'ScriptLocaiton', getS3ArnfromS3Url(_scriptLocation));
    _scriptBucketLocation.grantRead(_jobRole);
    const _glueJob = new glue.CfnJob(scope, 'KinesisETLJob', _newGlueJobProps);
    return [_glueJob, _jobRole];
}
exports.deployGlueJob = deployGlueJob;
/**
 * This is a helper method to create the Role required for the Glue Job. If a role is already created then this
 * method is not required to be called.
 *
 * @param scope - The AWS Construct under which the role is to be created
 */
function createGlueJobRole(scope) {
    return new aws_iam_1.Role(scope, 'JobRole', {
        assumedBy: new aws_iam_1.ServicePrincipal('glue.amazonaws.com'),
        description: 'Service role that Glue custom ETL jobs will assume for exeuction',
    });
}
exports.createGlueJobRole = createGlueJobRole;
/**
 * This method creates an AWS Glue table. The method is called when an existing Glue table is not provided
 */
function createGlueTable(scope, database, tableProps, fieldSchema, sourceType, parameters) {
    return defaults.DefaultGlueTable(scope, tableProps !== undefined ? tableProps :
        defaults.DefaultGlueTableProps(database, fieldSchema, sourceType, parameters));
}
exports.createGlueTable = createGlueTable;
/**
 * This method creates an AWS Glue database. The method is only called with an existing Glue database type is not provided.
 * The method uses the user provided props to override the defaul props for the Glue database
 *
 * @param scope
 * @param databaseProps
 */
function createGlueDatabase(scope, databaseProps) {
    const _mergedDBProps = (databaseProps !== undefined) ? utils_1.overrideProps(defaults.DefaultGlueDatabaseProps(), databaseProps) :
        defaults.DefaultGlueDatabaseProps();
    return defaults.DefaultGlueDatabase(scope, _mergedDBProps);
}
exports.createGlueDatabase = createGlueDatabase;
/**
 * A utility method to generate the S3 Arn from an S3 Url.
 *
 * @param s3Url
 */
function getS3ArnfromS3Url(s3Url) {
    const splitString = s3Url.slice('s3://'.length);
    return `arn:${core_1.Aws.PARTITION}:s3:::${splitString}`;
}
/**
 * A utility method to type check CfnJob.JobCommandProperty type.
 *
 * @param command
 */
function isJobCommandProperty(command) {
    if (command.name ||
        command.pythonVersion ||
        command.scriptLocation) {
        return true;
    }
    else {
        defaults.printWarning('command not of type JobCommandProperty type');
        return false;
    }
}
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiZ2x1ZS1qb2ItaGVscGVyLmpzIiwic291cmNlUm9vdCI6IiIsInNvdXJjZXMiOlsiZ2x1ZS1qb2ItaGVscGVyLnRzIl0sIm5hbWVzIjpbXSwibWFwcGluZ3MiOiI7QUFBQTs7Ozs7Ozs7Ozs7R0FXRzs7O0FBRUgsMENBQTBDO0FBQzFDLDhDQUFrRztBQUNsRyw0Q0FBK0Q7QUFDL0Qsd0NBQTREO0FBQzVELGdDQUFnQztBQUNoQyxtQ0FBd0M7QUFFeEM7OztHQUdHO0FBQ0gsSUFBWSxhQUVYO0FBRkQsV0FBWSxhQUFhO0lBQ3ZCLDBCQUFTLENBQUE7QUFDWCxDQUFDLEVBRlcsYUFBYSxHQUFiLHFCQUFhLEtBQWIscUJBQWEsUUFFeEI7QUF5QkQsU0FBZ0IsWUFBWSxDQUFDLEtBQWdCLEVBQUUsS0FBd0I7SUFDckUsSUFBSSxDQUFDLEtBQUssQ0FBQyxjQUFjLEVBQUU7UUFDekIsSUFBSSxLQUFLLENBQUMsWUFBWSxFQUFFO1lBQ3RCLE9BQU8sYUFBYSxDQUFDLEtBQUssRUFBRSxLQUFLLENBQUMsWUFBWSxFQUFFLEtBQUssQ0FBQyxRQUFTLEVBQUUsS0FBSyxDQUFDLEtBQU0sRUFBRSxLQUFLLENBQUMsZUFBZ0IsQ0FBQyxDQUFDO1NBQ3hHO2FBQU07WUFDTCxNQUFNLEtBQUssQ0FBQyxtREFBbUQsQ0FBQyxDQUFDO1NBQ2xFO0tBQ0Y7U0FBTTtRQUNMLE9BQU8sQ0FBQyxLQUFLLENBQUMsY0FBYyxFQUFFLGNBQUksQ0FBQyxXQUFXLENBQUMsS0FBSyxFQUFFLGNBQWMsRUFBRSxLQUFLLENBQUMsY0FBYyxDQUFDLElBQUksQ0FBQyxDQUFDLENBQUM7S0FDbkc7QUFDSCxDQUFDO0FBVkQsb0NBVUM7QUFFRCxTQUFnQixhQUFhLENBQUMsS0FBZ0IsRUFBRSxZQUE4QixFQUFFLFFBQTBCLEVBQUUsS0FBb0IsRUFDOUgsZUFBbUM7SUFFbkMsSUFBSSx1QkFBK0IsQ0FBQztJQUVwQyxJQUFJLFlBQVksQ0FBQyxxQkFBcUIsS0FBSyxTQUFTLEVBQUU7UUFDcEQsdUJBQXVCLEdBQUcsc0JBQXNCLENBQUM7UUFDakQsTUFBTSxXQUFXLEdBQUcsT0FBTyxVQUFHLENBQUMsU0FBUyxRQUFRLFVBQUcsQ0FBQyxNQUFNLElBQUksVUFBRyxDQUFDLFVBQVUsaUJBQWlCLENBQUM7UUFFOUYsSUFBSSxJQUFJLENBQUMsd0JBQXdCLENBQUMsS0FBSyxFQUFFLG9CQUFvQixFQUFFO1lBQzdELElBQUksRUFBRSx1QkFBdUI7WUFDN0IsdUJBQXVCLEVBQUU7Z0JBQ3ZCLHNCQUFzQixFQUFFO29CQUN0QiwwQkFBMEIsRUFBRSxTQUFTO29CQUNyQyxTQUFTLEVBQUUsV0FBVztpQkFDdkI7Z0JBQ0QsYUFBYSxFQUFFLENBQUM7d0JBQ2QsZ0JBQWdCLEVBQUUsUUFBUTtxQkFDM0IsQ0FBQzthQUNIO1NBQ0YsQ0FBQyxDQUFDO0tBQ0o7U0FBTTtRQUNMLHVCQUF1QixHQUFHLFlBQVksQ0FBQyxxQkFBcUIsQ0FBQztLQUM5RDtJQUVELE1BQU0sY0FBYyxHQUFHLElBQUksZ0JBQU0sQ0FBQyxLQUFLLEVBQUUsV0FBVyxFQUFFO1FBQ3BELFVBQVUsRUFBRTtZQUNWLElBQUkseUJBQWUsQ0FBQztnQkFDbEIsTUFBTSxFQUFFLGdCQUFNLENBQUMsS0FBSztnQkFDcEIsT0FBTyxFQUFFLENBQUUscUJBQXFCLEVBQUUsc0JBQXNCLEVBQUUsbUJBQW1CLENBQUU7Z0JBQy9FLFNBQVMsRUFBRSxDQUFFLE9BQU8sVUFBRyxDQUFDLFNBQVMsU0FBUyxVQUFHLENBQUMsTUFBTSxJQUFJLFVBQUcsQ0FBQyxVQUFVLHdCQUF3QixDQUFFO2FBQ2pHLENBQUM7U0FDSDtLQUNGLENBQUMsQ0FBQztJQUVILElBQUksUUFBZSxDQUFDO0lBQ3BCLElBQUksWUFBWSxDQUFDLElBQUksRUFBRTtRQUNyQixRQUFRLEdBQUcsY0FBSSxDQUFDLFdBQVcsQ0FBQyxLQUFLLEVBQUUsU0FBUyxFQUFFLFlBQVksQ0FBQyxJQUFJLENBQUMsQ0FBQztLQUNsRTtTQUFNO1FBQ0wsUUFBUSxHQUFHLFFBQVEsQ0FBQyxpQkFBaUIsQ0FBQyxLQUFLLENBQUMsQ0FBQztLQUM5QztJQUVELGNBQWMsQ0FBQyxZQUFZLENBQUMsUUFBUSxDQUFDLENBQUM7SUFFdEMsSUFBSSxlQUFvQyxDQUFDO0lBQ3pDLElBQUksZUFBZSxLQUFLLFNBQVMsSUFBSSxlQUFlLENBQUMsYUFBYSxLQUFLLGFBQWEsQ0FBQyxFQUFFLEVBQUU7UUFDdkYsSUFBSSxlQUFlLENBQUMsc0JBQXNCLEtBQUssU0FBUyxFQUFFO1lBQ3hELGVBQWUsR0FBRyxDQUFFLGVBQWUsQ0FBQyxzQkFBc0IsRUFBRSxTQUFTLENBQUUsQ0FBQztTQUN6RTthQUFNO1lBQ0wsZUFBZSxHQUFHLFFBQVEsQ0FBQyxhQUFhLENBQUMsS0FBSyxFQUFFLEVBQUUsV0FBVyxFQUFFLGVBQWUsQ0FBQyxpQkFBaUIsRUFBRSxDQUFFLENBQUM7U0FDdEc7S0FDRjtTQUFNO1FBQ0wsZUFBZSxHQUFHLFFBQVEsQ0FBQyxhQUFhLENBQUMsS0FBSyxFQUFFLEVBQUUsQ0FBQyxDQUFDO0tBQ3JEO0lBRUQsZUFBZSxDQUFDLENBQUMsQ0FBQyxDQUFDLGNBQWMsQ0FBQyxRQUFRLENBQUMsQ0FBQztJQUU1QyxNQUFNLGlCQUFpQixHQUFHO1FBQ3hCLGtCQUFrQixFQUFHLElBQUk7UUFDekIsb0NBQW9DLEVBQUcsSUFBSTtRQUMzQyxpQkFBaUIsRUFBRSxRQUFRLENBQUMsR0FBRztRQUMvQixjQUFjLEVBQUUsS0FBSyxDQUFDLEdBQUc7UUFDekIsR0FBRyxDQUFDLENBQUMsZUFBZSxLQUFLLFNBQVMsSUFBSSxDQUFDLGVBQWUsSUFBSSxlQUFlLENBQUMsYUFBYSxLQUFLLGFBQWEsQ0FBQyxFQUFFLENBQUMsQ0FBQztZQUM1RyxFQUFFLGVBQWUsRUFBRyxTQUFTLGVBQWUsQ0FBQyxDQUFDLENBQUMsQ0FBQyxVQUFVLFVBQVUsRUFBRSxDQUFDO1FBQ3pFLEdBQUcsWUFBWSxDQUFDLGdCQUFnQjtLQUNqQyxDQUFDO0lBRUYsTUFBTSxnQkFBZ0IsR0FBcUIscUJBQWEsQ0FBQyxRQUFRLENBQUMsbUJBQW1CLENBQUMsUUFBUyxFQUFFLFlBQVksQ0FBQyxPQUFPLEVBQ25ILHVCQUF1QixFQUFFLGlCQUFpQixFQUFFLFlBQVksQ0FBQyxXQUFXLENBQUMsRUFBRSxZQUFZLENBQUMsQ0FBQztJQUV2RixJQUFJLGVBQXVCLENBQUM7SUFDNUIsSUFBSSxvQkFBb0IsQ0FBQyxnQkFBZ0IsQ0FBQyxPQUFPLENBQUMsRUFBRTtRQUNsRCxJQUFJLGdCQUFnQixDQUFDLE9BQU8sQ0FBQyxjQUFjLEVBQUU7WUFDM0MsZUFBZSxHQUFHLGdCQUFnQixDQUFDLE9BQU8sQ0FBQyxjQUFjLENBQUM7U0FDM0Q7YUFBTTtZQUNMLE1BQU0sS0FBSyxDQUFDLDJGQUEyRixDQUFDLENBQUM7U0FDMUc7S0FDRjtJQUVELE1BQU0scUJBQXFCLEdBQVksZUFBTSxDQUFDLGFBQWEsQ0FBQyxLQUFLLEVBQUUsZ0JBQWdCLEVBQUUsaUJBQWlCLENBQUMsZUFBZ0IsQ0FBQyxDQUFDLENBQUM7SUFDMUgscUJBQXFCLENBQUMsU0FBUyxDQUFDLFFBQVEsQ0FBQyxDQUFDO0lBRTFDLE1BQU0sUUFBUSxHQUFnQixJQUFJLElBQUksQ0FBQyxNQUFNLENBQUMsS0FBSyxFQUFFLGVBQWUsRUFBRSxnQkFBZ0IsQ0FBQyxDQUFDO0lBQ3hGLE9BQU8sQ0FBQyxRQUFRLEVBQUUsUUFBUSxDQUFDLENBQUM7QUFDOUIsQ0FBQztBQXBGRCxzQ0FvRkM7QUFFRDs7Ozs7R0FLRztBQUNILFNBQWdCLGlCQUFpQixDQUFDLEtBQWdCO0lBQ2hELE9BQU8sSUFBSSxjQUFJLENBQUMsS0FBSyxFQUFFLFNBQVMsRUFBRTtRQUNoQyxTQUFTLEVBQUUsSUFBSSwwQkFBZ0IsQ0FBQyxvQkFBb0IsQ0FBQztRQUNyRCxXQUFXLEVBQUUsa0VBQWtFO0tBQ2hGLENBQUMsQ0FBQztBQUNMLENBQUM7QUFMRCw4Q0FLQztBQUVEOztHQUVHO0FBQ0gsU0FBZ0IsZUFBZSxDQUFDLEtBQWdCLEVBQUUsUUFBMEIsRUFBRSxVQUErQixFQUMzRyxXQUE2QyxFQUFFLFVBQW1CLEVBQUUsVUFBZ0I7SUFDcEYsT0FBTyxRQUFRLENBQUMsZ0JBQWdCLENBQUMsS0FBSyxFQUFFLFVBQVUsS0FBSyxTQUFTLENBQUMsQ0FBQyxDQUFDLFVBQVUsQ0FBQyxDQUFDO1FBQzdFLFFBQVEsQ0FBQyxxQkFBcUIsQ0FBQyxRQUFRLEVBQUUsV0FBWSxFQUFFLFVBQVUsRUFBRSxVQUFVLENBQUMsQ0FBQyxDQUFDO0FBQ3BGLENBQUM7QUFKRCwwQ0FJQztBQUVEOzs7Ozs7R0FNRztBQUNILFNBQWdCLGtCQUFrQixDQUFDLEtBQWdCLEVBQUcsYUFBcUM7SUFDekYsTUFBTSxjQUFjLEdBQTBCLENBQUMsYUFBYSxLQUFLLFNBQVMsQ0FBQyxDQUFDLENBQUMsQ0FBQyxxQkFBYSxDQUFDLFFBQVEsQ0FBQyx3QkFBd0IsRUFBRSxFQUFFLGFBQWEsQ0FBQyxDQUFDLENBQUM7UUFDL0ksUUFBUSxDQUFDLHdCQUF3QixFQUFFLENBQUM7SUFDdEMsT0FBTyxRQUFRLENBQUMsbUJBQW1CLENBQUMsS0FBSyxFQUFFLGNBQWMsQ0FBQyxDQUFDO0FBQzdELENBQUM7QUFKRCxnREFJQztBQUVEOzs7O0dBSUc7QUFDSCxTQUFTLGlCQUFpQixDQUFDLEtBQWE7SUFDdEMsTUFBTSxXQUFXLEdBQVcsS0FBSyxDQUFDLEtBQUssQ0FBQyxPQUFPLENBQUMsTUFBTSxDQUFDLENBQUM7SUFDeEQsT0FBTyxPQUFPLFVBQUcsQ0FBQyxTQUFTLFNBQVMsV0FBVyxFQUFFLENBQUM7QUFDcEQsQ0FBQztBQUVEOzs7O0dBSUc7QUFDSCxTQUFTLG9CQUFvQixDQUFDLE9BQXFEO0lBQ2pGLElBQUssT0FBMEMsQ0FBQyxJQUFJO1FBQ2pELE9BQTBDLENBQUMsYUFBYTtRQUN4RCxPQUEwQyxDQUFDLGNBQWMsRUFBRTtRQUM1RCxPQUFPLElBQUksQ0FBQztLQUNiO1NBQU07UUFDTCxRQUFRLENBQUMsWUFBWSxDQUFDLDZDQUE2QyxDQUFDLENBQUM7UUFDckUsT0FBTyxLQUFLLENBQUM7S0FDZDtBQUNILENBQUMiLCJzb3VyY2VzQ29udGVudCI6WyIvKipcbiAqICBDb3B5cmlnaHQgMjAyMSBBbWF6b24uY29tLCBJbmMuIG9yIGl0cyBhZmZpbGlhdGVzLiBBbGwgUmlnaHRzIFJlc2VydmVkLlxuICpcbiAqICBMaWNlbnNlZCB1bmRlciB0aGUgQXBhY2hlIExpY2Vuc2UsIFZlcnNpb24gMi4wICh0aGUgXCJMaWNlbnNlXCIpLiBZb3UgbWF5IG5vdCB1c2UgdGhpcyBmaWxlIGV4Y2VwdCBpbiBjb21wbGlhbmNlXG4gKiAgd2l0aCB0aGUgTGljZW5zZS4gQSBjb3B5IG9mIHRoZSBMaWNlbnNlIGlzIGxvY2F0ZWQgYXRcbiAqXG4gKiAgICAgIGh0dHA6Ly93d3cuYXBhY2hlLm9yZy9saWNlbnNlcy9MSUNFTlNFLTIuMFxuICpcbiAqICBvciBpbiB0aGUgJ2xpY2Vuc2UnIGZpbGUgYWNjb21wYW55aW5nIHRoaXMgZmlsZS4gVGhpcyBmaWxlIGlzIGRpc3RyaWJ1dGVkIG9uIGFuICdBUyBJUycgQkFTSVMsIFdJVEhPVVQgV0FSUkFOVElFU1xuICogIE9SIENPTkRJVElPTlMgT0YgQU5ZIEtJTkQsIGV4cHJlc3Mgb3IgaW1wbGllZC4gU2VlIHRoZSBMaWNlbnNlIGZvciB0aGUgc3BlY2lmaWMgbGFuZ3VhZ2UgZ292ZXJuaW5nIHBlcm1pc3Npb25zXG4gKiAgYW5kIGxpbWl0YXRpb25zIHVuZGVyIHRoZSBMaWNlbnNlLlxuICovXG5cbmltcG9ydCAqIGFzIGdsdWUgZnJvbSAnQGF3cy1jZGsvYXdzLWdsdWUnO1xuaW1wb3J0IHsgRWZmZWN0LCBJUm9sZSwgUG9saWN5LCBQb2xpY3lTdGF0ZW1lbnQsIFJvbGUsIFNlcnZpY2VQcmluY2lwYWwgfSBmcm9tICdAYXdzLWNkay9hd3MtaWFtJztcbmltcG9ydCB7IEJ1Y2tldCwgQnVja2V0UHJvcHMsIElCdWNrZXQgfSBmcm9tICdAYXdzLWNkay9hd3MtczMnO1xuaW1wb3J0IHsgQXdzLCBDb25zdHJ1Y3QsIElSZXNvbHZhYmxlIH0gZnJvbSAnQGF3cy1jZGsvY29yZSc7XG5pbXBvcnQgKiBhcyBkZWZhdWx0cyBmcm9tICcuLi8nO1xuaW1wb3J0IHsgb3ZlcnJpZGVQcm9wcyB9IGZyb20gJy4vdXRpbHMnO1xuXG4vKipcbiAqIEVudW1lcmF0aW9uIG9mIGRhdGEgc3RvcmUgdHlwZXMgdGhhdCBjb3VsZCBpbmNsdWRlIFMzLCBEeW5hbW9EQiwgRG9jdW1lbnREQiwgUkRTIG9yIFJlZHNoaWZ0LiBDdXJyZW50XG4gKiBjb25zdHJ1Y3QgaW1wbGVtZW50YXRpb24gb25seSBzdXBwb3J0cyBTMywgYnV0IHBvdGVudGlhbCB0byBhZGQgb3RoZXIgb3V0cHV0IHR5cGVzIGluIHRoZSBmdXR1cmVcbiAqL1xuZXhwb3J0IGVudW0gU2lua1N0b3JlVHlwZSB7XG4gIFMzID0gJ1MzJ1xufVxuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbmV4cG9ydCBpbnRlcmZhY2UgU2lua0RhdGFTdG9yZVByb3BzIHtcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcmVhZG9ubHkgZGF0YXN0b3JlVHlwZTogU2lua1N0b3JlVHlwZTtcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICByZWFkb25seSBleGlzdGluZ1MzT3V0cHV0QnVja2V0PzogQnVja2V0XG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICByZWFkb25seSBvdXRwdXRCdWNrZXRQcm9wcz86IEJ1Y2tldFByb3BzO1xufVxuXG5leHBvcnQgaW50ZXJmYWNlIEJ1aWxkR2x1ZUpvYlByb3BzIHtcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHJlYWRvbmx5IGdsdWVKb2JQcm9wcz86IGdsdWUuQ2ZuSm9iUHJvcHMgfCBhbnlcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICByZWFkb25seSBleGlzdGluZ0NmbkpvYj86IGdsdWUuQ2ZuSm9iO1xuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICByZWFkb25seSB0YWJsZTogZ2x1ZS5DZm5UYWJsZTtcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcmVhZG9ubHkgZGF0YWJhc2U6IGdsdWUuQ2ZuRGF0YWJhc2U7XG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcmVhZG9ubHkgb3V0cHV0RGF0YVN0b3JlPzogU2lua0RhdGFTdG9yZVByb3BzXG59XG5cbmV4cG9ydCBmdW5jdGlvbiBidWlsZEdsdWVKb2Ioc2NvcGU6IENvbnN0cnVjdCwgcHJvcHM6IEJ1aWxkR2x1ZUpvYlByb3BzKTogW2dsdWUuQ2ZuSm9iLCBJUm9sZV0ge1xuICBpZiAoIXByb3BzLmV4aXN0aW5nQ2ZuSm9iKSB7XG4gICAgaWYgKHByb3BzLmdsdWVKb2JQcm9wcykge1xuICAgICAgcmV0dXJuIGRlcGxveUdsdWVKb2Ioc2NvcGUsIHByb3BzLmdsdWVKb2JQcm9wcywgcHJvcHMuZGF0YWJhc2UhLCBwcm9wcy50YWJsZSEsIHByb3BzLm91dHB1dERhdGFTdG9yZSEpO1xuICAgIH0gZWxzZSB7XG4gICAgICB0aHJvdyBFcnJvcignRWl0aGVyIGdsdWVKb2JQcm9wcyBvciBleGlzdGluZ0NmbkpvYiBpcyByZXF1aXJlZCcpO1xuICAgIH1cbiAgfSBlbHNlIHtcbiAgICByZXR1cm4gW3Byb3BzLmV4aXN0aW5nQ2ZuSm9iLCBSb2xlLmZyb21Sb2xlQXJuKHNjb3BlLCAnRXhpc3RpbmdSb2xlJywgcHJvcHMuZXhpc3RpbmdDZm5Kb2Iucm9sZSldO1xuICB9XG59XG5cbmV4cG9ydCBmdW5jdGlvbiBkZXBsb3lHbHVlSm9iKHNjb3BlOiBDb25zdHJ1Y3QsIGdsdWVKb2JQcm9wczogZ2x1ZS5DZm5Kb2JQcm9wcywgZGF0YWJhc2U6IGdsdWUuQ2ZuRGF0YWJhc2UsIHRhYmxlOiBnbHVlLkNmblRhYmxlLFxuICBvdXRwdXREYXRhU3RvcmU6IFNpbmtEYXRhU3RvcmVQcm9wcyk6IFtnbHVlLkNmbkpvYiwgSVJvbGVdIHtcblxuICBsZXQgX2dsdWVTZWN1cml0eUNvbmZpZ05hbWU6IHN0cmluZztcblxuICBpZiAoZ2x1ZUpvYlByb3BzLnNlY3VyaXR5Q29uZmlndXJhdGlvbiA9PT0gdW5kZWZpbmVkKSB7XG4gICAgX2dsdWVTZWN1cml0eUNvbmZpZ05hbWUgPSAnRVRMSm9iU2VjdXJpdHlDb25maWcnO1xuICAgIGNvbnN0IF9nbHVlS01TS2V5ID0gYGFybjoke0F3cy5QQVJUSVRJT059Omttczoke0F3cy5SRUdJT059OiR7QXdzLkFDQ09VTlRfSUR9OmFsaWFzL2F3cy9nbHVlYDtcblxuICAgIG5ldyBnbHVlLkNmblNlY3VyaXR5Q29uZmlndXJhdGlvbihzY29wZSwgJ0dsdWVTZWN1cml0eUNvbmZpZycsIHtcbiAgICAgIG5hbWU6IF9nbHVlU2VjdXJpdHlDb25maWdOYW1lLFxuICAgICAgZW5jcnlwdGlvbkNvbmZpZ3VyYXRpb246IHtcbiAgICAgICAgam9iQm9va21hcmtzRW5jcnlwdGlvbjoge1xuICAgICAgICAgIGpvYkJvb2ttYXJrc0VuY3J5cHRpb25Nb2RlOiAnQ1NFLUtNUycsXG4gICAgICAgICAga21zS2V5QXJuOiBfZ2x1ZUtNU0tleVxuICAgICAgICB9LFxuICAgICAgICBzM0VuY3J5cHRpb25zOiBbe1xuICAgICAgICAgIHMzRW5jcnlwdGlvbk1vZGU6ICdTU0UtUzMnXG4gICAgICAgIH1dXG4gICAgICB9XG4gICAgfSk7XG4gIH0gZWxzZSB7XG4gICAgX2dsdWVTZWN1cml0eUNvbmZpZ05hbWUgPSBnbHVlSm9iUHJvcHMuc2VjdXJpdHlDb25maWd1cmF0aW9uO1xuICB9XG5cbiAgY29uc3QgX2dsdWVKb2JQb2xpY3kgPSBuZXcgUG9saWN5KHNjb3BlLCAnTG9nUG9saWN5Jywge1xuICAgIHN0YXRlbWVudHM6IFtcbiAgICAgIG5ldyBQb2xpY3lTdGF0ZW1lbnQoe1xuICAgICAgICBlZmZlY3Q6IEVmZmVjdC5BTExPVyxcbiAgICAgICAgYWN0aW9uczogWyAnbG9nczpDcmVhdGVMb2dHcm91cCcsICdsb2dzOkNyZWF0ZUxvZ1N0cmVhbScsICdsb2dzOlB1dExvZ0V2ZW50cycgXSxcbiAgICAgICAgcmVzb3VyY2VzOiBbIGBhcm46JHtBd3MuUEFSVElUSU9OfTpsb2dzOiR7QXdzLlJFR0lPTn06JHtBd3MuQUNDT1VOVF9JRH06bG9nLWdyb3VwOi9hd3MtZ2x1ZS8qYCBdXG4gICAgICB9KVxuICAgIF1cbiAgfSk7XG5cbiAgbGV0IF9qb2JSb2xlOiBJUm9sZTtcbiAgaWYgKGdsdWVKb2JQcm9wcy5yb2xlKSB7XG4gICAgX2pvYlJvbGUgPSBSb2xlLmZyb21Sb2xlQXJuKHNjb3BlLCAnSm9iUm9sZScsIGdsdWVKb2JQcm9wcy5yb2xlKTtcbiAgfSBlbHNlIHtcbiAgICBfam9iUm9sZSA9IGRlZmF1bHRzLmNyZWF0ZUdsdWVKb2JSb2xlKHNjb3BlKTtcbiAgfVxuXG4gIF9nbHVlSm9iUG9saWN5LmF0dGFjaFRvUm9sZShfam9iUm9sZSk7XG5cbiAgbGV0IF9vdXRwdXRMb2NhdGlvbjogWyBCdWNrZXQsIEJ1Y2tldD8gXTtcbiAgaWYgKG91dHB1dERhdGFTdG9yZSAhPT0gdW5kZWZpbmVkICYmIG91dHB1dERhdGFTdG9yZS5kYXRhc3RvcmVUeXBlID09PSBTaW5rU3RvcmVUeXBlLlMzKSB7XG4gICAgaWYgKG91dHB1dERhdGFTdG9yZS5leGlzdGluZ1MzT3V0cHV0QnVja2V0ICE9PSB1bmRlZmluZWQpIHtcbiAgICAgIF9vdXRwdXRMb2NhdGlvbiA9IFsgb3V0cHV0RGF0YVN0b3JlLmV4aXN0aW5nUzNPdXRwdXRCdWNrZXQsIHVuZGVmaW5lZCBdO1xuICAgIH0gZWxzZSB7XG4gICAgICBfb3V0cHV0TG9jYXRpb24gPSBkZWZhdWx0cy5idWlsZFMzQnVja2V0KHNjb3BlLCB7IGJ1Y2tldFByb3BzOiBvdXRwdXREYXRhU3RvcmUub3V0cHV0QnVja2V0UHJvcHMgfSApO1xuICAgIH1cbiAgfSBlbHNlIHtcbiAgICBfb3V0cHV0TG9jYXRpb24gPSBkZWZhdWx0cy5idWlsZFMzQnVja2V0KHNjb3BlLCB7fSk7XG4gIH1cblxuICBfb3V0cHV0TG9jYXRpb25bMF0uZ3JhbnRSZWFkV3JpdGUoX2pvYlJvbGUpO1xuXG4gIGNvbnN0IF9qb2JBcmd1bWVudHNMaXN0ID0ge1xuICAgIFwiLS1lbmFibGUtbWV0cmljc1wiIDogdHJ1ZSxcbiAgICBcIi0tZW5hYmxlLWNvbnRpbnVvdXMtY2xvdWR3YXRjaC1sb2dcIiA6IHRydWUsXG4gICAgXCItLWRhdGFiYXNlX25hbWVcIjogZGF0YWJhc2UucmVmLFxuICAgIFwiLS10YWJsZV9uYW1lXCI6IHRhYmxlLnJlZixcbiAgICAuLi4oKG91dHB1dERhdGFTdG9yZSA9PT0gdW5kZWZpbmVkIHx8IChvdXRwdXREYXRhU3RvcmUgJiYgb3V0cHV0RGF0YVN0b3JlLmRhdGFzdG9yZVR5cGUgPT09IFNpbmtTdG9yZVR5cGUuUzMpKSAmJlxuICAgICAgeyAnLS1vdXRwdXRfcGF0aCcgOiBgczNhOi8vJHtfb3V0cHV0TG9jYXRpb25bMF0uYnVja2V0TmFtZX0vb3V0cHV0L2AgfSksXG4gICAgLi4uZ2x1ZUpvYlByb3BzLmRlZmF1bHRBcmd1bWVudHNcbiAgfTtcblxuICBjb25zdCBfbmV3R2x1ZUpvYlByb3BzOiBnbHVlLkNmbkpvYlByb3BzID0gb3ZlcnJpZGVQcm9wcyhkZWZhdWx0cy5EZWZhdWx0R2x1ZUpvYlByb3BzKF9qb2JSb2xlISwgZ2x1ZUpvYlByb3BzLmNvbW1hbmQsXG4gICAgX2dsdWVTZWN1cml0eUNvbmZpZ05hbWUsIF9qb2JBcmd1bWVudHNMaXN0LCBnbHVlSm9iUHJvcHMuZ2x1ZVZlcnNpb24pLCBnbHVlSm9iUHJvcHMpO1xuXG4gIGxldCBfc2NyaXB0TG9jYXRpb246IHN0cmluZztcbiAgaWYgKGlzSm9iQ29tbWFuZFByb3BlcnR5KF9uZXdHbHVlSm9iUHJvcHMuY29tbWFuZCkpIHtcbiAgICBpZiAoX25ld0dsdWVKb2JQcm9wcy5jb21tYW5kLnNjcmlwdExvY2F0aW9uKSB7XG4gICAgICBfc2NyaXB0TG9jYXRpb24gPSBfbmV3R2x1ZUpvYlByb3BzLmNvbW1hbmQuc2NyaXB0TG9jYXRpb247XG4gICAgfSBlbHNlIHtcbiAgICAgIHRocm93IEVycm9yKCdTY3JpcHQgbG9jYXRpb24gaGFzIHRvIGJlIHByb3ZpZGVkIGFzIGFuIHMzIFVybCBsb2NhdGlvbi4gU2NyaXB0IGxvY2F0aW9uIGNhbm5vdCBiZSBlbXB0eScpO1xuICAgIH1cbiAgfVxuXG4gIGNvbnN0IF9zY3JpcHRCdWNrZXRMb2NhdGlvbjogSUJ1Y2tldCA9IEJ1Y2tldC5mcm9tQnVja2V0QXJuKHNjb3BlLCAnU2NyaXB0TG9jYWl0b24nLCBnZXRTM0FybmZyb21TM1VybChfc2NyaXB0TG9jYXRpb24hKSk7XG4gIF9zY3JpcHRCdWNrZXRMb2NhdGlvbi5ncmFudFJlYWQoX2pvYlJvbGUpO1xuXG4gIGNvbnN0IF9nbHVlSm9iOiBnbHVlLkNmbkpvYiA9IG5ldyBnbHVlLkNmbkpvYihzY29wZSwgJ0tpbmVzaXNFVExKb2InLCBfbmV3R2x1ZUpvYlByb3BzKTtcbiAgcmV0dXJuIFtfZ2x1ZUpvYiwgX2pvYlJvbGVdO1xufVxuXG4vKipcbiAqIFRoaXMgaXMgYSBoZWxwZXIgbWV0aG9kIHRvIGNyZWF0ZSB0aGUgUm9sZSByZXF1aXJlZCBmb3IgdGhlIEdsdWUgSm9iLiBJZiBhIHJvbGUgaXMgYWxyZWFkeSBjcmVhdGVkIHRoZW4gdGhpc1xuICogbWV0aG9kIGlzIG5vdCByZXF1aXJlZCB0byBiZSBjYWxsZWQuXG4gKlxuICogQHBhcmFtIHNjb3BlIC0gVGhlIEFXUyBDb25zdHJ1Y3QgdW5kZXIgd2hpY2ggdGhlIHJvbGUgaXMgdG8gYmUgY3JlYXRlZFxuICovXG5leHBvcnQgZnVuY3Rpb24gY3JlYXRlR2x1ZUpvYlJvbGUoc2NvcGU6IENvbnN0cnVjdCk6IFJvbGUge1xuICByZXR1cm4gbmV3IFJvbGUoc2NvcGUsICdKb2JSb2xlJywge1xuICAgIGFzc3VtZWRCeTogbmV3IFNlcnZpY2VQcmluY2lwYWwoJ2dsdWUuYW1hem9uYXdzLmNvbScpLFxuICAgIGRlc2NyaXB0aW9uOiAnU2VydmljZSByb2xlIHRoYXQgR2x1ZSBjdXN0b20gRVRMIGpvYnMgd2lsbCBhc3N1bWUgZm9yIGV4ZXVjdGlvbicsXG4gIH0pO1xufVxuXG4vKipcbiAqIFRoaXMgbWV0aG9kIGNyZWF0ZXMgYW4gQVdTIEdsdWUgdGFibGUuIFRoZSBtZXRob2QgaXMgY2FsbGVkIHdoZW4gYW4gZXhpc3RpbmcgR2x1ZSB0YWJsZSBpcyBub3QgcHJvdmlkZWRcbiAqL1xuZXhwb3J0IGZ1bmN0aW9uIGNyZWF0ZUdsdWVUYWJsZShzY29wZTogQ29uc3RydWN0LCBkYXRhYmFzZTogZ2x1ZS5DZm5EYXRhYmFzZSwgdGFibGVQcm9wcz86IGdsdWUuQ2ZuVGFibGVQcm9wcyxcbiAgZmllbGRTY2hlbWE/OiBnbHVlLkNmblRhYmxlLkNvbHVtblByb3BlcnR5IFtdLCBzb3VyY2VUeXBlPzogc3RyaW5nLCBwYXJhbWV0ZXJzPzogYW55KTogZ2x1ZS5DZm5UYWJsZSB7XG4gIHJldHVybiBkZWZhdWx0cy5EZWZhdWx0R2x1ZVRhYmxlKHNjb3BlLCB0YWJsZVByb3BzICE9PSB1bmRlZmluZWQgPyB0YWJsZVByb3BzIDpcbiAgICBkZWZhdWx0cy5EZWZhdWx0R2x1ZVRhYmxlUHJvcHMoZGF0YWJhc2UsIGZpZWxkU2NoZW1hISwgc291cmNlVHlwZSwgcGFyYW1ldGVycykpO1xufVxuXG4vKipcbiAqIFRoaXMgbWV0aG9kIGNyZWF0ZXMgYW4gQVdTIEdsdWUgZGF0YWJhc2UuIFRoZSBtZXRob2QgaXMgb25seSBjYWxsZWQgd2l0aCBhbiBleGlzdGluZyBHbHVlIGRhdGFiYXNlIHR5cGUgaXMgbm90IHByb3ZpZGVkLlxuICogVGhlIG1ldGhvZCB1c2VzIHRoZSB1c2VyIHByb3ZpZGVkIHByb3BzIHRvIG92ZXJyaWRlIHRoZSBkZWZhdWwgcHJvcHMgZm9yIHRoZSBHbHVlIGRhdGFiYXNlXG4gKlxuICogQHBhcmFtIHNjb3BlXG4gKiBAcGFyYW0gZGF0YWJhc2VQcm9wc1xuICovXG5leHBvcnQgZnVuY3Rpb24gY3JlYXRlR2x1ZURhdGFiYXNlKHNjb3BlOiBDb25zdHJ1Y3QsICBkYXRhYmFzZVByb3BzPzogZ2x1ZS5DZm5EYXRhYmFzZVByb3BzKTogZ2x1ZS5DZm5EYXRhYmFzZSB7XG4gIGNvbnN0IF9tZXJnZWREQlByb3BzOiBnbHVlLkNmbkRhdGFiYXNlUHJvcHMgPSAoZGF0YWJhc2VQcm9wcyAhPT0gdW5kZWZpbmVkKSA/IG92ZXJyaWRlUHJvcHMoZGVmYXVsdHMuRGVmYXVsdEdsdWVEYXRhYmFzZVByb3BzKCksIGRhdGFiYXNlUHJvcHMpIDpcbiAgICBkZWZhdWx0cy5EZWZhdWx0R2x1ZURhdGFiYXNlUHJvcHMoKTtcbiAgcmV0dXJuIGRlZmF1bHRzLkRlZmF1bHRHbHVlRGF0YWJhc2Uoc2NvcGUsIF9tZXJnZWREQlByb3BzKTtcbn1cblxuLyoqXG4gKiBBIHV0aWxpdHkgbWV0aG9kIHRvIGdlbmVyYXRlIHRoZSBTMyBBcm4gZnJvbSBhbiBTMyBVcmwuXG4gKlxuICogQHBhcmFtIHMzVXJsXG4gKi9cbmZ1bmN0aW9uIGdldFMzQXJuZnJvbVMzVXJsKHMzVXJsOiBzdHJpbmcpOiBzdHJpbmcge1xuICBjb25zdCBzcGxpdFN0cmluZzogc3RyaW5nID0gczNVcmwuc2xpY2UoJ3MzOi8vJy5sZW5ndGgpO1xuICByZXR1cm4gYGFybjoke0F3cy5QQVJUSVRJT059OnMzOjo6JHtzcGxpdFN0cmluZ31gO1xufVxuXG4vKipcbiAqIEEgdXRpbGl0eSBtZXRob2QgdG8gdHlwZSBjaGVjayBDZm5Kb2IuSm9iQ29tbWFuZFByb3BlcnR5IHR5cGUuXG4gKlxuICogQHBhcmFtIGNvbW1hbmRcbiAqL1xuZnVuY3Rpb24gaXNKb2JDb21tYW5kUHJvcGVydHkoY29tbWFuZDogZ2x1ZS5DZm5Kb2IuSm9iQ29tbWFuZFByb3BlcnR5IHwgSVJlc29sdmFibGUpOiBjb21tYW5kIGlzIGdsdWUuQ2ZuSm9iLkpvYkNvbW1hbmRQcm9wZXJ0eSB7XG4gIGlmICgoY29tbWFuZCBhcyBnbHVlLkNmbkpvYi5Kb2JDb21tYW5kUHJvcGVydHkpLm5hbWUgfHxcbiAgICAoY29tbWFuZCBhcyBnbHVlLkNmbkpvYi5Kb2JDb21tYW5kUHJvcGVydHkpLnB5dGhvblZlcnNpb24gfHxcbiAgICAoY29tbWFuZCBhcyBnbHVlLkNmbkpvYi5Kb2JDb21tYW5kUHJvcGVydHkpLnNjcmlwdExvY2F0aW9uKSB7XG4gICAgcmV0dXJuIHRydWU7XG4gIH0gZWxzZSB7XG4gICAgZGVmYXVsdHMucHJpbnRXYXJuaW5nKCdjb21tYW5kIG5vdCBvZiB0eXBlIEpvYkNvbW1hbmRQcm9wZXJ0eSB0eXBlJyk7XG4gICAgcmV0dXJuIGZhbHNlO1xuICB9XG59Il19