"use strict";
/**
 *  Copyright 2021 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.buildStepFunctionCWAlarms = exports.buildStateMachine = void 0;
const cdk = require("@aws-cdk/core");
const smDefaults = require("./step-function-defaults");
const sfn = require("@aws-cdk/aws-stepfunctions");
const utils_1 = require("./utils");
const iam = require("@aws-cdk/aws-iam");
const cloudwatch = require("@aws-cdk/aws-cloudwatch");
const cloudwatch_log_group_helper_1 = require("./cloudwatch-log-group-helper");
/**
 * Builds and returns a StateMachine.
 * @param scope - the construct to which the StateMachine should be attached to.
 * @param stateMachineProps - user-specified properties to override the default properties.
 */
function buildStateMachine(scope, stateMachineProps, logGroupProps) {
    var _a, _b;
    let logGroup;
    let _smProps;
    // If they sent a logGroup in stateMachineProps
    if ((_a = stateMachineProps.logs) === null || _a === void 0 ? void 0 : _a.destination) {
        logGroup = (_b = stateMachineProps.logs) === null || _b === void 0 ? void 0 : _b.destination;
        _smProps = stateMachineProps;
    }
    else {
        // Three possibilities
        // 1) logGroupProps not provided - create logGroupProps with just logGroupName
        // 2) logGroupProps provided with no logGroupName - override logGroupProps.logGroupName
        // 3) logGroupProps provided with logGroupName - pass unaltered logGroupProps
        let consolidatedLogGroupProps = logGroupProps;
        if (!consolidatedLogGroupProps) {
            consolidatedLogGroupProps = {};
        }
        if (!(consolidatedLogGroupProps === null || consolidatedLogGroupProps === void 0 ? void 0 : consolidatedLogGroupProps.logGroupName)) {
            const logGroupPrefix = '/aws/vendedlogs/states/';
            const maxResourceNameLength = 255 - logGroupPrefix.length;
            const nameParts = [
                cdk.Stack.of(scope).stackName,
                scope.node.id,
                'StateMachineLog' // Literal string for log group name portion
            ];
            const logGroupName = logGroupPrefix + utils_1.generateResourceName(nameParts, maxResourceNameLength);
            consolidatedLogGroupProps = utils_1.overrideProps(consolidatedLogGroupProps, { logGroupName });
        }
        // Create new Cloudwatch log group for Step function State Machine
        logGroup = cloudwatch_log_group_helper_1.buildLogGroup(scope, 'StateMachineLogGroup', consolidatedLogGroupProps);
        // Override the defaults with the user provided props
        _smProps = utils_1.overrideProps(smDefaults.DefaultStateMachineProps(logGroup), stateMachineProps);
    }
    // Override the Cloudwatch permissions to make it more fine grained
    const _sm = new sfn.StateMachine(scope, 'StateMachine', _smProps);
    const role = _sm.node.findChild('Role');
    const cfnDefaultPolicy = role.node.findChild('DefaultPolicy').node.defaultChild;
    // Reduce the scope of actions for the existing DefaultPolicy
    cfnDefaultPolicy.addPropertyOverride('PolicyDocument.Statement.0.Action', [
        "logs:CreateLogDelivery",
        'logs:GetLogDelivery',
        'logs:UpdateLogDelivery',
        'logs:DeleteLogDelivery',
        'logs:ListLogDeliveries'
    ]);
    // Override Cfn Nag warning W12: IAM policy should not allow * resource
    cfnDefaultPolicy.cfnOptions.metadata = {
        cfn_nag: {
            rules_to_suppress: [{
                    id: 'W12',
                    reason: `The 'LogDelivery' actions do not support resource-level authorizations`
                }]
        }
    };
    // Add a new policy with logging permissions for the given cloudwatch log group
    _sm.addToRolePolicy(new iam.PolicyStatement({
        actions: [
            'logs:PutResourcePolicy',
            'logs:DescribeResourcePolicies',
            'logs:DescribeLogGroups'
        ],
        resources: [`arn:${cdk.Aws.PARTITION}:logs:${cdk.Aws.REGION}:${cdk.Aws.ACCOUNT_ID}:*`]
    }));
    return [_sm, logGroup];
}
exports.buildStateMachine = buildStateMachine;
function buildStepFunctionCWAlarms(scope, sm) {
    // Setup CW Alarms for State Machine
    const alarms = new Array();
    // Sum of number of executions that failed is >= 1 for 5 minutes, 1 consecutive time
    alarms.push(new cloudwatch.Alarm(scope, 'ExecutionFailedAlarm', {
        metric: sm.metricFailed(),
        threshold: 1,
        evaluationPeriods: 1,
        statistic: 'Sum',
        period: cdk.Duration.seconds(300),
        comparisonOperator: cloudwatch.ComparisonOperator.GREATER_THAN_OR_EQUAL_TO_THRESHOLD,
        alarmDescription: 'Alarm for the number of executions that failed exceeded the threshold of 1. '
    }));
    // Sum of number of executions that failed maximum is >= 1 for 5 minute, 1 consecutive time
    alarms.push(new cloudwatch.Alarm(scope, 'ExecutionThrottledAlarm', {
        metric: sm.metricThrottled(),
        threshold: 1,
        evaluationPeriods: 1,
        statistic: 'Sum',
        period: cdk.Duration.seconds(300),
        comparisonOperator: cloudwatch.ComparisonOperator.GREATER_THAN_OR_EQUAL_TO_THRESHOLD,
        alarmDescription: 'Alarm for the number of executions that throttled exceeded the threshold of 1. '
    }));
    // Number of executions that aborted maximum is >= 1 for 5 minute, 1 consecutive time
    alarms.push(new cloudwatch.Alarm(scope, 'ExecutionAbortedAlarm', {
        metric: sm.metricAborted(),
        threshold: 1,
        evaluationPeriods: 1,
        statistic: 'Maximum',
        period: cdk.Duration.seconds(300),
        comparisonOperator: cloudwatch.ComparisonOperator.GREATER_THAN_OR_EQUAL_TO_THRESHOLD,
        alarmDescription: 'Alarm for the number of executions that aborted exceeded the threshold of 1. '
    }));
    return alarms;
}
exports.buildStepFunctionCWAlarms = buildStepFunctionCWAlarms;
//# sourceMappingURL=data:application/json;base64,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