"use strict";
/**
 *  Copyright 2021 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
const assert_1 = require("@aws-cdk/assert");
const core_1 = require("@aws-cdk/core");
const cloudfront = require("@aws-cdk/aws-cloudfront");
const lambda = require("@aws-cdk/aws-lambda");
const cloudfront_distribution_helper_1 = require("../lib/cloudfront-distribution-helper");
const s3_bucket_helper_1 = require("../lib/s3-bucket-helper");
require("@aws-cdk/assert/jest");
const aws_s3_1 = require("@aws-cdk/aws-s3");
const origins = require("@aws-cdk/aws-cloudfront-origins");
const acm = require("@aws-cdk/aws-certificatemanager");
const aws_cloudfront_1 = require("@aws-cdk/aws-cloudfront");
test('cloudfront distribution with default params', () => {
    const stack = new core_1.Stack();
    const [sourceBucket] = s3_bucket_helper_1.buildS3Bucket(stack, {});
    cloudfront_distribution_helper_1.CloudFrontDistributionForS3(stack, sourceBucket);
    expect(assert_1.SynthUtils.toCloudFormation(stack)).toMatchSnapshot();
});
test('check bucket policy metadata', () => {
    const stack = new core_1.Stack();
    const [sourceBucket] = s3_bucket_helper_1.buildS3Bucket(stack, {});
    cloudfront_distribution_helper_1.CloudFrontDistributionForS3(stack, sourceBucket);
    expect(stack).toHaveResource('AWS::S3::BucketPolicy', {
        Metadata: {
            cfn_nag: {
                rules_to_suppress: [
                    {
                        id: "F16",
                        reason: "Public website bucket policy requires a wildcard principal"
                    }
                ]
            }
        }
    }, assert_1.ResourcePart.CompleteDefinition);
});
test('check bucket metadata', () => {
    const stack = new core_1.Stack();
    const [sourceBucket] = s3_bucket_helper_1.buildS3Bucket(stack, {});
    cloudfront_distribution_helper_1.CloudFrontDistributionForS3(stack, sourceBucket);
    expect(stack).toHaveResource('AWS::S3::Bucket', {
        Metadata: {
            cfn_nag: {
                rules_to_suppress: [
                    {
                        id: "W35",
                        reason: "This S3 bucket is used as the access logging bucket for CloudFront Distribution"
                    }
                ]
            }
        }
    }, assert_1.ResourcePart.CompleteDefinition);
});
test('test cloudfront check bucket policy', () => {
    const stack = new core_1.Stack();
    const [sourceBucket] = s3_bucket_helper_1.buildS3Bucket(stack, {});
    cloudfront_distribution_helper_1.CloudFrontDistributionForS3(stack, sourceBucket);
    expect(stack).toHaveResourceLike("AWS::S3::BucketPolicy", {
        PolicyDocument: {
            Statement: [
                {
                    Action: "*",
                    Condition: {
                        Bool: {
                            "aws:SecureTransport": "false"
                        }
                    },
                    Effect: "Deny",
                    Principal: "*",
                    Resource: {
                        "Fn::Join": [
                            "",
                            [
                                {
                                    "Fn::GetAtt": [
                                        "S3Bucket07682993",
                                        "Arn"
                                    ]
                                },
                                "/*"
                            ]
                        ]
                    }
                }
            ],
            Version: "2012-10-17"
        }
    });
});
test('test cloudfront with no security headers ', () => {
    const stack = new core_1.Stack();
    const [sourceBucket] = s3_bucket_helper_1.buildS3Bucket(stack, {});
    cloudfront_distribution_helper_1.CloudFrontDistributionForS3(stack, sourceBucket, {}, false);
    expect(stack).toHaveResourceLike("AWS::CloudFront::Distribution", {
        DistributionConfig: {
            DefaultCacheBehavior: {
                CachePolicyId: "658327ea-f89d-4fab-a63d-7e88639e58f6",
                Compress: true,
                TargetOriginId: "CloudFrontDistributionOrigin176EC3A12",
                ViewerProtocolPolicy: "redirect-to-https"
            },
            DefaultRootObject: "index.html",
            Enabled: true,
            HttpVersion: "http2",
            IPV6Enabled: true,
            Logging: {
                Bucket: {
                    "Fn::GetAtt": [
                        "CloudfrontLoggingBucket3C3EFAA7",
                        "RegionalDomainName"
                    ]
                }
            },
            Origins: [
                {
                    DomainName: {
                        "Fn::GetAtt": [
                            "S3Bucket07682993",
                            "RegionalDomainName"
                        ]
                    },
                    Id: "CloudFrontDistributionOrigin176EC3A12",
                    S3OriginConfig: {
                        OriginAccessIdentity: {
                            "Fn::Join": [
                                "",
                                [
                                    "origin-access-identity/cloudfront/",
                                    {
                                        Ref: "CloudFrontDistributionOrigin1S3Origin3D9CA0E9"
                                    }
                                ]
                            ]
                        }
                    }
                }
            ]
        }
    });
});
test('test cloudfront override cloudfront custom domain names ', () => {
    const stack = new core_1.Stack();
    const [sourceBucket] = s3_bucket_helper_1.buildS3Bucket(stack, {});
    const certificate = acm.Certificate.fromCertificateArn(stack, 'Cert', 'arn:aws:acm:us-east-1:123456789012:certificate/12345678-1234-1234-1234-123456789012');
    const myprops = {
        domainNames: ['mydomains'],
        certificate
    };
    cloudfront_distribution_helper_1.CloudFrontDistributionForS3(stack, sourceBucket, myprops);
    expect(stack).toHaveResourceLike("AWS::CloudFront::Distribution", {
        DistributionConfig: {
            Aliases: [
                "mydomains"
            ],
            DefaultCacheBehavior: {
                CachePolicyId: "658327ea-f89d-4fab-a63d-7e88639e58f6",
                Compress: true,
                LambdaFunctionAssociations: [
                    {
                        EventType: "origin-response",
                        LambdaFunctionARN: {
                            Ref: "SetHttpSecurityHeadersVersion660E2F72"
                        }
                    }
                ],
                TargetOriginId: "CloudFrontDistributionOrigin176EC3A12",
                ViewerProtocolPolicy: "redirect-to-https"
            },
            DefaultRootObject: "index.html",
            Enabled: true,
            HttpVersion: "http2",
            IPV6Enabled: true,
            Logging: {
                Bucket: {
                    "Fn::GetAtt": [
                        "CloudfrontLoggingBucket3C3EFAA7",
                        "RegionalDomainName"
                    ]
                }
            },
            Origins: [
                {
                    DomainName: {
                        "Fn::GetAtt": [
                            "S3Bucket07682993",
                            "RegionalDomainName"
                        ]
                    },
                    Id: "CloudFrontDistributionOrigin176EC3A12",
                    S3OriginConfig: {
                        OriginAccessIdentity: {
                            "Fn::Join": [
                                "",
                                [
                                    "origin-access-identity/cloudfront/",
                                    {
                                        Ref: "CloudFrontDistributionOrigin1S3Origin3D9CA0E9"
                                    }
                                ]
                            ]
                        }
                    }
                }
            ],
            ViewerCertificate: {
                AcmCertificateArn: "arn:aws:acm:us-east-1:123456789012:certificate/12345678-1234-1234-1234-123456789012",
                MinimumProtocolVersion: "TLSv1.2_2019",
                SslSupportMethod: "sni-only"
            }
        }
    });
});
test('test cloudfront override cloudfront logging bucket ', () => {
    const stack = new core_1.Stack();
    const [sourceBucket] = s3_bucket_helper_1.buildS3Bucket(stack, {});
    const logBucket = new aws_s3_1.Bucket(stack, 'loggingbucket');
    const myprops = {
        enableLogging: true,
        logBucket
    };
    cloudfront_distribution_helper_1.CloudFrontDistributionForS3(stack, sourceBucket, myprops);
    expect(stack).toHaveResourceLike("AWS::CloudFront::Distribution", {
        DistributionConfig: {
            DefaultCacheBehavior: {
                CachePolicyId: "658327ea-f89d-4fab-a63d-7e88639e58f6",
                Compress: true,
                LambdaFunctionAssociations: [
                    {
                        EventType: "origin-response",
                        LambdaFunctionARN: {
                            Ref: "SetHttpSecurityHeadersVersion660E2F72"
                        }
                    }
                ],
                TargetOriginId: "CloudFrontDistributionOrigin176EC3A12",
                ViewerProtocolPolicy: "redirect-to-https"
            },
            DefaultRootObject: "index.html",
            Enabled: true,
            HttpVersion: "http2",
            IPV6Enabled: true,
            Logging: {
                Bucket: {
                    "Fn::GetAtt": [
                        "loggingbucket6D73BD53",
                        "RegionalDomainName"
                    ]
                }
            },
            Origins: [
                {
                    DomainName: {
                        "Fn::GetAtt": [
                            "S3Bucket07682993",
                            "RegionalDomainName"
                        ]
                    },
                    Id: "CloudFrontDistributionOrigin176EC3A12",
                    S3OriginConfig: {
                        OriginAccessIdentity: {
                            "Fn::Join": [
                                "",
                                [
                                    "origin-access-identity/cloudfront/",
                                    {
                                        Ref: "CloudFrontDistributionOrigin1S3Origin3D9CA0E9"
                                    }
                                ]
                            ]
                        }
                    }
                }
            ]
        }
    });
});
test('test cloudfront override properties', () => {
    const stack = new core_1.Stack();
    const [sourceBucket] = s3_bucket_helper_1.buildS3Bucket(stack, {});
    const props = {
        defaultBehavior: {
            origin: new origins.S3Origin(sourceBucket),
            viewerProtocolPolicy: cloudfront.ViewerProtocolPolicy.REDIRECT_TO_HTTPS,
            allowedMethods: cloudfront.AllowedMethods.ALLOW_ALL,
            cachedMethods: cloudfront.CachedMethods.CACHE_GET_HEAD_OPTIONS
        },
    };
    cloudfront_distribution_helper_1.CloudFrontDistributionForS3(stack, sourceBucket, props);
    expect(stack).toHaveResourceLike("AWS::CloudFront::Distribution", {
        DistributionConfig: {
            DefaultCacheBehavior: {
                AllowedMethods: [
                    "GET",
                    "HEAD",
                    "OPTIONS",
                    "PUT",
                    "PATCH",
                    "POST",
                    "DELETE"
                ],
                CachePolicyId: "658327ea-f89d-4fab-a63d-7e88639e58f6",
                CachedMethods: [
                    "GET",
                    "HEAD",
                    "OPTIONS"
                ],
                Compress: true,
                LambdaFunctionAssociations: [
                    {
                        EventType: "origin-response",
                        LambdaFunctionARN: {
                            Ref: "SetHttpSecurityHeadersVersion660E2F72"
                        }
                    }
                ],
                TargetOriginId: "CloudFrontDistributionOrigin176EC3A12",
                ViewerProtocolPolicy: "redirect-to-https"
            },
            DefaultRootObject: "index.html",
            Enabled: true,
            HttpVersion: "http2",
            IPV6Enabled: true,
            Logging: {
                Bucket: {
                    "Fn::GetAtt": [
                        "CloudfrontLoggingBucket3C3EFAA7",
                        "RegionalDomainName"
                    ]
                }
            },
            Origins: [
                {
                    DomainName: {
                        "Fn::GetAtt": [
                            "S3Bucket07682993",
                            "RegionalDomainName"
                        ]
                    },
                    Id: "CloudFrontDistributionOrigin176EC3A12",
                    S3OriginConfig: {
                        OriginAccessIdentity: {
                            "Fn::Join": [
                                "",
                                [
                                    "origin-access-identity/cloudfront/",
                                    {
                                        Ref: "CloudFrontDistributionOrigin1S3Origin3D9CA0E9"
                                    }
                                ]
                            ]
                        }
                    }
                }
            ]
        }
    });
});
test('test override cloudfront add custom lambda@edge', () => {
    const stack = new core_1.Stack();
    const [sourceBucket] = s3_bucket_helper_1.buildS3Bucket(stack, {});
    // custom lambda@edg function
    const handler = new lambda.Function(stack, 'SomeHandler', {
        functionName: 'SomeHandler',
        runtime: lambda.Runtime.NODEJS_12_X,
        handler: 'index.handler',
        code: lambda.Code.fromAsset(`${__dirname}/lambda`),
    });
    const handlerVersion = new lambda.Version(stack, 'SomeHandlerVersion', {
        lambda: handler,
    });
    cloudfront_distribution_helper_1.CloudFrontDistributionForS3(stack, sourceBucket, {
        defaultBehavior: {
            edgeLambdas: [
                {
                    eventType: aws_cloudfront_1.LambdaEdgeEventType.VIEWER_REQUEST,
                    includeBody: false,
                    functionVersion: handlerVersion,
                }
            ]
        }
    });
    expect(stack).toHaveResource("AWS::CloudFront::Distribution", {
        DistributionConfig: {
            DefaultCacheBehavior: {
                CachePolicyId: "658327ea-f89d-4fab-a63d-7e88639e58f6",
                Compress: true,
                LambdaFunctionAssociations: [
                    {
                        EventType: "origin-response",
                        LambdaFunctionARN: {
                            Ref: "SetHttpSecurityHeadersVersion660E2F72"
                        }
                    },
                    {
                        EventType: "viewer-request",
                        IncludeBody: false,
                        LambdaFunctionARN: {
                            Ref: "SomeHandlerVersionDA986E41"
                        }
                    }
                ],
                TargetOriginId: "CloudFrontDistributionOrigin176EC3A12",
                ViewerProtocolPolicy: "redirect-to-https"
            },
            DefaultRootObject: "index.html",
            Enabled: true,
            HttpVersion: "http2",
            IPV6Enabled: true,
            Logging: {
                Bucket: {
                    "Fn::GetAtt": [
                        "CloudfrontLoggingBucket3C3EFAA7",
                        "RegionalDomainName"
                    ]
                }
            },
            Origins: [
                {
                    DomainName: {
                        "Fn::GetAtt": [
                            "S3Bucket07682993",
                            "RegionalDomainName"
                        ]
                    },
                    Id: "CloudFrontDistributionOrigin176EC3A12",
                    S3OriginConfig: {
                        OriginAccessIdentity: {
                            "Fn::Join": [
                                "",
                                [
                                    "origin-access-identity/cloudfront/",
                                    {
                                        Ref: "CloudFrontDistributionOrigin1S3Origin3D9CA0E9"
                                    }
                                ]
                            ]
                        }
                    }
                }
            ]
        }
    });
});
test('test override cloudfront replace custom lambda@edge', () => {
    const stack = new core_1.Stack();
    const [sourceBucket] = s3_bucket_helper_1.buildS3Bucket(stack, {});
    // custom lambda@edg function
    const handler = new lambda.Function(stack, 'SomeHandler', {
        functionName: 'SomeHandler',
        runtime: lambda.Runtime.NODEJS_12_X,
        handler: 'index.handler',
        code: lambda.Code.fromAsset(`${__dirname}/lambda`),
    });
    const handlerVersion = new lambda.Version(stack, 'SomeHandlerVersion', {
        lambda: handler,
    });
    cloudfront_distribution_helper_1.CloudFrontDistributionForS3(stack, sourceBucket, {
        defaultBehavior: {
            edgeLambdas: [
                {
                    eventType: aws_cloudfront_1.LambdaEdgeEventType.VIEWER_REQUEST,
                    includeBody: false,
                    functionVersion: handlerVersion,
                }
            ]
        }
    }, false);
    expect(stack).toHaveResource("AWS::CloudFront::Distribution", {
        DistributionConfig: {
            DefaultCacheBehavior: {
                CachePolicyId: "658327ea-f89d-4fab-a63d-7e88639e58f6",
                Compress: true,
                LambdaFunctionAssociations: [
                    {
                        EventType: "viewer-request",
                        IncludeBody: false,
                        LambdaFunctionARN: {
                            Ref: "SomeHandlerVersionDA986E41"
                        }
                    }
                ],
                TargetOriginId: "CloudFrontDistributionOrigin176EC3A12",
                ViewerProtocolPolicy: "redirect-to-https"
            },
            DefaultRootObject: "index.html",
            Enabled: true,
            HttpVersion: "http2",
            IPV6Enabled: true,
            Logging: {
                Bucket: {
                    "Fn::GetAtt": [
                        "CloudfrontLoggingBucket3C3EFAA7",
                        "RegionalDomainName"
                    ]
                }
            },
            Origins: [
                {
                    DomainName: {
                        "Fn::GetAtt": [
                            "S3Bucket07682993",
                            "RegionalDomainName"
                        ]
                    },
                    Id: "CloudFrontDistributionOrigin176EC3A12",
                    S3OriginConfig: {
                        OriginAccessIdentity: {
                            "Fn::Join": [
                                "",
                                [
                                    "origin-access-identity/cloudfront/",
                                    {
                                        Ref: "CloudFrontDistributionOrigin1S3Origin3D9CA0E9"
                                    }
                                ]
                            ]
                        }
                    }
                }
            ]
        }
    });
});
//# sourceMappingURL=data:application/json;base64,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