"use strict";
/**
 *  Copyright 2021 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.generateIntegStackName = exports.CreateScrapBucket = void 0;
// Imports
const aws_s3_1 = require("@aws-cdk/aws-s3");
const core_1 = require("@aws-cdk/core");
const utils_1 = require("../lib/utils");
const path = require("path");
// Creates a bucket used for testing - minimal properties, destroyed after test
function CreateScrapBucket(scope, props) {
    const defaultProps = {
        versioned: true,
        removalPolicy: core_1.RemovalPolicy.DESTROY,
    };
    let synthesizedProps;
    if (props) {
        synthesizedProps = utils_1.overrideProps(defaultProps, props);
    }
    else {
        synthesizedProps = defaultProps;
    }
    const scriptBucket = new aws_s3_1.Bucket(scope, "existingScriptLocation", synthesizedProps);
    scriptBucket.node.defaultChild.cfnOptions.metadata = {
        cfn_nag: {
            rules_to_suppress: [
                {
                    id: "W51",
                    reason: "This S3 bucket is created for unit/ integration testing purposes only and not part of \
      the actual construct implementation",
                },
                {
                    id: "W35",
                    reason: "This S3 bucket is created for unit/ integration testing purposes only and not part of \
      the actual construct implementation",
                },
                {
                    id: "W41",
                    reason: "This S3 bucket is created for unit/ integration testing purposes only and not part of \
      the actual construct",
                },
            ],
        },
    };
    return scriptBucket;
}
exports.CreateScrapBucket = CreateScrapBucket;
/**
 * @summary Creates a stack name for Integration tests
 * @param {string} filename - the filename of the integ test
 * @returns {string} - a string with current filename after removing anything before the prefix '.' and suffix '.js'
 * e.g. 'integ.apigateway-dynamodb-CRUD.js' will return 'apigateway-dynamodb-CRUD'
 */
function generateIntegStackName(filename) {
    const file = path.basename(filename, path.extname(filename));
    const stackname = file.slice(file.lastIndexOf('.') + 1).replace(/_/g, '-');
    return stackname;
}
exports.generateIntegStackName = generateIntegStackName;
//# sourceMappingURL=data:application/json;base64,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