# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['alphaess_modbus']

package_data = \
{'': ['*']}

install_requires = \
['AsyncioMinimalModbus>=1.0.0,<2.0.0']

setup_kwargs = {
    'name': 'alphaess-modbus',
    'version': '0.1.0',
    'description': 'Async Python 3 library to read ModBus from an AlphaESS inverter',
    'long_description': '# AlphaESS ModBus reader\n\nAsync Python 3 library to read ModBus from an AlphaESS inverter.\n\nUses [asynciominimalmodbus](https://github.com/guyradford/asynciominimalmodbus) for ModBus/RS485 communication.\n\nCompatible with:\n\n| **Device**  | **Baud** | **Tested** |\n|-------------|----------|------------|\n| SMILE5      | 9600     |      ✅     |\n| SMILE-B3    | 9600     |            |\n| SMILE-T10   | 9600     |            |\n| Storion T30 | 19200    |            |\n\n## Hardware\n\n⚠️⚠️ This worked for me, so do at your own risk!! ⚠️⚠️\n\nMore information (and pictures) in the [Notes](#my-setup) section below.\n\n- Use the inverter menu to enable modbus in slave mode.\n- Snip one end of an ethernet cable off and connect (may vary):\n    - Blue/white to RS485 A\n    - Blue to RS485 B    \n    - RS485 RX to GPIO 15\n    - RS485 TX to GPIO 14\n- Enable serial port on Raspberry Pi with `raspi-config`.\n- Connect other end of ethernet cable to the inverter CAN port.\n\n## Quick start\n\n### PIP\n\nInstall with:\n\n``` bash\npython3 -m pip install alphaess-modbus\n```\n\nCheckout `example.py` to get started\n\n### Clone\n\nClone repo and run `example.py`:\n\n``` bash\ngit clone git@github.com:SorX14/alphaess_modbus.git\ncd ./alphaess_modbus\n./example.py\n```\n\n``` bash\n[Sun, 20 Nov 2022 21:36:54] INFO [example.py.main:27] PV: 0W GRID: 1078W USE: 1078W Battery: 0W\n```\n\nDone! 🎉\n\n## Architecture\n\nThis library concentrates on reading data, but [writing](#writing-values) is possible.\n\nUses a JSON definition file containing all the ModBus registers and how to parse them - lookup the register you want from the [PDF](https://www.alpha-ess.de/images/downloads/handbuecher/AlphaESS-Handbuch_SMILET30_ModBus_RTU_V123-DE.pdf) and request it using the reader functions below.\n\nFor example, to get the capacity of your installed system, find the item in the PDF:\n\n![PDF entry](https://raw.githubusercontent.com/SorX14/alphaess_modbus/main/docs/pdf_register.png)\n\nCopy the name - `PV Capacity of Grid Inverter` - and request with `.get_value("PV Capacity of Grid Inverter")`\n\n### Definitions\n\n``` json\n  ...\n  {\n    "name": "pv2_current",\n    "address": 1058,\n    "hex": "0x0422",\n    "type": "register",\n    "signed": false,\n    "decimals": 1,\n    "units": "A"\n  },\n  ...\n```\n\ncalled with:\n\n``` python\nawait reader.get_value("PV2 current") # or await reader.get_value("pv2_current")\n```\n\nwill read register `0x0422`, process the result as unsigned, divide it by 10, and optionally add `A` as the units.\n\nThe default JSON file was created with [alphaess_pdf_parser](https://github.com/SorX14/alphaess_pdf_parser). You can override the default JSON file with `Reader(json_file=location)`\n\n## Reading values\n\n### `Reader()`\n\nCreate a new reader\n\n``` python\nimport asyncio\nfrom alphaess_modbus import Reader\n\nasync def main():\n    reader: Reader = Reader()\n\n    definition = await reader.get_definition("pv2_voltage")\n    print(definition)\n\nasyncio.run(main())\n```\n\nOptionally change the defaults with:\n\n- `decimalAddress=85`\n- `serial=\'/dev/serial0\'`\n- `debug=False`\n- `baud=9600`\n- `json_file=None`\n- `formatter=None`\n\n### `get_value(name) -> int`\n\nRequests a value from the inverter.\n\n``` python\ngrid = await reader.get_value("total_active_power_grid_meter")\nprint(grid)\n\n# 1234\n```\n\nPrints the current grid usage as an integer.\n\n### `get_units(name) -> str`\n\nGet units (if any) for a register name.\n\n``` python\ngrid_units = await reader.get_units("total_active_power_grid_meter")\nprint(grid_units)\n\n# W\n```\n\n### `get_formatted_value(name, use_formatter=True)`\n\nSame as `get_value()` but returns a string with units. If a [formatter](#formatters) is defined for the register, a different return type is possible.\n\n``` python\ngrid = await reader.get_formatted_value("total_active_power_grid_meter")\nprint(grid)\n\n# 1234W\n```\n\nSet `use_formatter` to `False` to prevent a formatter from being invoked.\n\n### `get_definition(name) -> dict`\n\nGet the JSON entry for an item. Useful if you\'re trying to [write](#writing-values) a register.\n\n``` python\nitem = await reader.get_definition("inverter_power_total")\nprint(item)\n\n# {\'name\': \'inverter_power_total\', \'address\': 1036, \'hex\': \'0x040C\', \'type\': \'long\', \'signed\': True, \'decimals\': 0, \'units\': \'W\'}\n```\n\n## Formatters\n\nSome registers are special and not just simple numbers - they could contain ASCII, hex-encoded numbers or another format.\n\nFor example, `0x0809` `Local IP` returns 4 bytes of the current IP, e.g. `0xC0，0xA8，0x01，0x01` (`192.168.1.1`).\n\nTo help, there is a built-in formatter which will be invoked when calling `.get_formatted_value()` e.g:\n\n``` python\nip = await reader.get_formatted_value("Local IP")\nprint(ip)\n\n# 192.168.0.1\n```\n\nNot all registers have a formatter, and you might have a preference on how the value is returned (e.g. time-format). To help with this, you can pass a `formatter` to `Reader()` and override or add to the default:\n\n``` python\n\nclass my_custom_formatter:\n  def local_ip(self, val) -> str:\n    bytes = val.to_bytes(4, byteorder=\'big\')\n    return f"IP of device: {int(bytes[0])} - {int(bytes[1])} - {int(bytes[2])} - {int(bytes[3])}"\n\nreader: Reader = Reader(formatter=my_customer_formatter)\n\nlocal_ip = await reader.get_formatted_value("local_ip")\nprint(local_ip)\n\n# IP of device: 192 - 168 - 0 - 0\n```\n\nEach formatting function is based on the conformed name of a register. You can find the conformed name of a register by searching `registers.json` or by using `await reader.get_definition(name)`\n\n## Writing values\n\n☠️ ModBus gives full control of the inverter. There are device-level protections in place but be very careful ☠️\n\nThis library is intended to read values, but you can get a reference to the  [internal ModBus library](https://pypi.org/project/AsyncioMinimalModbus/) with `reader.instrument`:\n\n``` python\nread = await reader.instrument.read_long(int(0x0021), 3, False)\nprint(read)\n```\n\nRead the library docs for what to do next: https://minimalmodbus.readthedocs.io/en/stable/\n\nUse the [AlphaESS manual](https://www.alpha-ess.de/images/downloads/handbuecher/AlphaESS-Handbuch_SMILET30_ModBus_RTU_V123-DE.pdf) for how each register works.\n\n## Notes\n\n### Definitions\n\nWhile [my parsing script](https://github.com/SorX14/alphaess_pdf_parser) did its best, there are likely to be many faults and missing entries. I only need a few basic registers so haven\'t tested them all.\n\nSome registers are longer than the default 4 bytes and won\'t work- you\'ll have to use the internal reader instead.\n\nPR\'s are welcome 🙂\n\n### Registers always returning 0\n\nThere are a lot of registers, but they might not all be relevant depending on your system setup. For example, the PV meter section is useless if your AlphaESS is in DC mode. \n\n### Error handling\n\nI\'ve had the connection break a few times while testing, make sure you handle reconnecting correctly. `example.py` will output the full exception should one happen.\n\n### My setup\n\nI used a [m5stamp RS485 module](https://shop.m5stack.com/products/m5stamp-rs485-module) with a digital isolator and DC/DC isolator.\n\n![RS485 adapter](https://raw.githubusercontent.com/SorX14/alphaess_modbus/main/docs/rs485_adapter.png)\n\nInstalled in an enclosure with a PoE adapter to power the Pi and provide connectivity.\n\n![Enclosure](https://raw.githubusercontent.com/SorX14/alphaess_modbus/main/docs/enclosure.png)\n\nEnabled ModBus interface on the inverter. You\'ll need the service password, mine was set to the default of `1111`.\n\n![Modbus enable](https://raw.githubusercontent.com/SorX14/alphaess_modbus/main/docs/modbus_enable.png)\n\nThen connected to the CAN port.\n\n![Installed](https://raw.githubusercontent.com/SorX14/alphaess_modbus/main/docs/installed.png)\n\n# Credit and thanks\n\nSpecial thanks go to https://github.com/CharlesGillanders/alphaess where I originally started\nplaying around with my PV system. Their project uses the AlphaESS dashboard backend API to unofficially get inverter values from the cloud.\n\nInvaluable resource for discussing with other users. Highly\nrecommend reading https://github.com/CharlesGillanders/alphaess/issues/9 which ended up with\nAlphaESS creating an official API to retrieve data - https://github.com/alphaess-developer/alphacloud_open_api\n\nAnother great resource is https://github.com/dxoverdy/Alpha2MQTT which uses a ESP8266 instead\nof a Raspberry PI to communicate with the inverter - again - highly recommended.\n\nhttps://github.com/scanapi/scanapi for \'helping\' with github actions (I used their workflow actions as templates for this project).\n\n',
    'author': 'steve.parker',
    'author_email': None,
    'maintainer': None,
    'maintainer_email': None,
    'url': 'https://github.com/SorX14/alphaess_modbus',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'python_requires': '>=3.8,<4.0',
}


setup(**setup_kwargs)
