"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.collectRuntimeInformation = void 0;
const path_1 = require("path");
const node_version_1 = require("./node-version");
// list of NPM scopes included in version reporting e.g. @aws-cdk and @aws-solutions-konstruk
const WHITELIST_SCOPES = ['@aws-cdk', '@aws-cdk-containers', '@aws-solutions-konstruk', '@aws-solutions-constructs', '@amzn'];
// list of NPM packages included in version reporting
const WHITELIST_PACKAGES = ['aws-rfdk', 'aws-cdk-lib'];
/**
 * Returns a list of loaded modules and their versions.
 */
function collectRuntimeInformation() {
    const libraries = {};
    for (const fileName of Object.keys(require.cache)) {
        const pkg = findNpmPackage(fileName);
        if (pkg && !pkg.private) {
            libraries[pkg.name] = pkg.version;
        }
    }
    // include only libraries that are in the whitelistLibraries list
    for (const name of Object.keys(libraries)) {
        let foundMatch = false;
        for (const scope of WHITELIST_SCOPES) {
            if (name.startsWith(`${scope}/`)) {
                foundMatch = true;
            }
        }
        foundMatch = foundMatch || WHITELIST_PACKAGES.includes(name);
        if (!foundMatch) {
            delete libraries[name];
        }
    }
    // add jsii runtime version
    libraries['jsii-runtime'] = getJsiiAgentVersion();
    return { libraries };
}
exports.collectRuntimeInformation = collectRuntimeInformation;
/**
 * Determines which NPM module a given loaded javascript file is from.
 *
 * The only infromation that is available locally is a list of Javascript files,
 * and every source file is associated with a search path to resolve the further
 * ``require`` calls made from there, which includes its own directory on disk,
 * and parent directories - for example:
 *
 * [ '...repo/packages/aws-cdk-resources/lib/cfn/node_modules',
 *   '...repo/packages/aws-cdk-resources/lib/node_modules',
 *   '...repo/packages/aws-cdk-resources/node_modules',
 *   '...repo/packages/node_modules',
 *   // etc...
 * ]
 *
 * We are looking for ``package.json`` that is anywhere in the tree, except it's
 * in the parent directory, not in the ``node_modules`` directory. For this
 * reason, we strip the ``/node_modules`` suffix off each path and use regular
 * module resolution to obtain a reference to ``package.json``.
 *
 * @param fileName a javascript file name.
 * @returns the NPM module infos (aka ``package.json`` contents), or
 *      ``undefined`` if the lookup was unsuccessful.
 */
function findNpmPackage(fileName) {
    const mod = require.cache[fileName];
    if (!(mod === null || mod === void 0 ? void 0 : mod.paths)) {
        // sometimes this can be undefined. for example when querying for .json modules
        // inside a jest runtime environment.
        // see https://github.com/aws/aws-cdk/issues/7657
        // potentially we can remove this if it turns out to be a bug in how jest implemented the 'require' module.
        return undefined;
    }
    // For any path in ``mod.paths`` that is a node_modules folder, use its parent directory instead.
    const paths = mod === null || mod === void 0 ? void 0 : mod.paths.map((path) => path_1.basename(path) === 'node_modules' ? path_1.dirname(path) : path);
    try {
        const packagePath = require.resolve(
        // Resolution behavior changed in node 12.0.0 - https://github.com/nodejs/node/issues/27583
        node_version_1.major >= 12 ? './package.json' : 'package.json', { paths });
        // eslint-disable-next-line @typescript-eslint/no-require-imports
        return require(packagePath);
    }
    catch (e) {
        return undefined;
    }
}
function getJsiiAgentVersion() {
    let jsiiAgent = process.env.JSII_AGENT;
    // if JSII_AGENT is not specified, we will assume this is a node.js runtime
    // and plug in our node.js version
    if (!jsiiAgent) {
        jsiiAgent = `node.js/${process.version}`;
    }
    return jsiiAgent;
}
//# sourceMappingURL=data:application/json;base64,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