"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.ZoneDelegationRecord = exports.MxRecord = exports.CaaAmazonRecord = exports.CaaRecord = exports.CaaTag = exports.SrvRecord = exports.TxtRecord = exports.CnameRecord = exports.AaaaRecord = exports.ARecord = exports.AddressRecordTarget = exports.RecordSet = exports.RecordTarget = exports.RecordType = void 0;
const core_1 = require("@aws-cdk/core");
const route53_generated_1 = require("./route53.generated");
const util_1 = require("./util");
/**
 * The record type.
 */
var RecordType;
(function (RecordType) {
    /**
     * route traffic to a resource, such as a web server, using an IPv4 address in dotted decimal
     * notation
     *
     * @see https://docs.aws.amazon.com/Route53/latest/DeveloperGuide/ResourceRecordTypes.html#AFormat
     */
    RecordType["A"] = "A";
    /**
     * route traffic to a resource, such as a web server, using an IPv6 address in colon-separated
     * hexadecimal format
     *
     * @see https://docs.aws.amazon.com/Route53/latest/DeveloperGuide/ResourceRecordTypes.html#AAAAFormat
     */
    RecordType["AAAA"] = "AAAA";
    /**
     * A CAA record specifies which certificate authorities (CAs) are allowed to issue certificates
     * for a domain or subdomain
     *
     * @see https://docs.aws.amazon.com/Route53/latest/DeveloperGuide/ResourceRecordTypes.html#CAAFormat
     */
    RecordType["CAA"] = "CAA";
    /**
     * A CNAME record maps DNS queries for the name of the current record, such as acme.example.com,
     * to another domain (example.com or example.net) or subdomain (acme.example.com or zenith.example.org).
     *
     * @see https://docs.aws.amazon.com/Route53/latest/DeveloperGuide/ResourceRecordTypes.html#CNAMEFormat
     */
    RecordType["CNAME"] = "CNAME";
    /**
     * An MX record specifies the names of your mail servers and, if you have two or more mail servers,
     * the priority order.
     *
     * @see https://docs.aws.amazon.com/Route53/latest/DeveloperGuide/ResourceRecordTypes.html#MXFormat
     */
    RecordType["MX"] = "MX";
    /**
     * A Name Authority Pointer (NAPTR) is a type of record that is used by Dynamic Delegation Discovery
     * System (DDDS) applications to convert one value to another or to replace one value with another.
     * For example, one common use is to convert phone numbers into SIP URIs.
     *
     * @see https://docs.aws.amazon.com/Route53/latest/DeveloperGuide/ResourceRecordTypes.html#NAPTRFormat
     */
    RecordType["NAPTR"] = "NAPTR";
    /**
     * An NS record identifies the name servers for the hosted zone
     *
     * @see https://docs.aws.amazon.com/Route53/latest/DeveloperGuide/ResourceRecordTypes.html#NSFormat
     */
    RecordType["NS"] = "NS";
    /**
     * A PTR record maps an IP address to the corresponding domain name.
     *
     * @see https://docs.aws.amazon.com/Route53/latest/DeveloperGuide/ResourceRecordTypes.html#PTRFormat
     */
    RecordType["PTR"] = "PTR";
    /**
     * A start of authority (SOA) record provides information about a domain and the corresponding Amazon
     * Route 53 hosted zone
     *
     * @see https://docs.aws.amazon.com/Route53/latest/DeveloperGuide/ResourceRecordTypes.html#SOAFormat
     */
    RecordType["SOA"] = "SOA";
    /**
     * SPF records were formerly used to verify the identity of the sender of email messages.
     * Instead of an SPF record, we recommend that you create a TXT record that contains the applicable value.
     *
     * @see https://docs.aws.amazon.com/Route53/latest/DeveloperGuide/ResourceRecordTypes.html#SPFFormat
     */
    RecordType["SPF"] = "SPF";
    /**
     * An SRV record Value element consists of four space-separated values. The first three values are
     * decimal numbers representing priority, weight, and port. The fourth value is a domain name.
     *
     * @see https://docs.aws.amazon.com/Route53/latest/DeveloperGuide/ResourceRecordTypes.html#SRVFormat
     */
    RecordType["SRV"] = "SRV";
    /**
     * A TXT record contains one or more strings that are enclosed in double quotation marks (").
     *
     * @see https://docs.aws.amazon.com/Route53/latest/DeveloperGuide/ResourceRecordTypes.html#TXTFormat
     */
    RecordType["TXT"] = "TXT";
})(RecordType = exports.RecordType || (exports.RecordType = {}));
/**
 * Type union for a record that accepts multiple types of target.
 */
class RecordTarget {
    /**
     *
     * @param values correspond with the chosen record type (e.g. for 'A' Type, specify one ore more IP addresses)
     * @param aliasTarget alias for targets such as CloudFront distribution to route traffic to
     */
    constructor(values, aliasTarget) {
        this.values = values;
        this.aliasTarget = aliasTarget;
    }
    /**
     * Use string values as target.
     */
    static fromValues(...values) {
        return new RecordTarget(values);
    }
    /**
     * Use an alias as target.
     */
    static fromAlias(aliasTarget) {
        return new RecordTarget(undefined, aliasTarget);
    }
    /**
     * Use ip addresses as target.
     */
    static fromIpAddresses(...ipAddresses) {
        return RecordTarget.fromValues(...ipAddresses);
    }
}
exports.RecordTarget = RecordTarget;
/**
 * A record set.
 */
class RecordSet extends core_1.Resource {
    constructor(scope, id, props) {
        super(scope, id);
        const ttl = props.target.aliasTarget ? undefined : ((props.ttl && props.ttl.toSeconds()) || 1800).toString();
        const recordSet = new route53_generated_1.CfnRecordSet(this, 'Resource', {
            hostedZoneId: props.zone.hostedZoneId,
            name: util_1.determineFullyQualifiedDomainName(props.recordName || props.zone.zoneName, props.zone),
            type: props.recordType,
            resourceRecords: props.target.values,
            aliasTarget: props.target.aliasTarget && props.target.aliasTarget.bind(this),
            ttl,
            comment: props.comment,
        });
        this.domainName = recordSet.ref;
    }
}
exports.RecordSet = RecordSet;
/**
 * Target for a DNS A Record
 *
 * @deprecated Use RecordTarget
 */
class AddressRecordTarget extends RecordTarget {
}
exports.AddressRecordTarget = AddressRecordTarget;
/**
 * A DNS A record
 *
 * @resource AWS::Route53::RecordSet
 */
class ARecord extends RecordSet {
    constructor(scope, id, props) {
        super(scope, id, {
            ...props,
            recordType: RecordType.A,
            target: props.target,
        });
    }
}
exports.ARecord = ARecord;
/**
 * A DNS AAAA record
 *
 * @resource AWS::Route53::RecordSet
 */
class AaaaRecord extends RecordSet {
    constructor(scope, id, props) {
        super(scope, id, {
            ...props,
            recordType: RecordType.AAAA,
            target: props.target,
        });
    }
}
exports.AaaaRecord = AaaaRecord;
/**
 * A DNS CNAME record
 *
 * @resource AWS::Route53::RecordSet
 */
class CnameRecord extends RecordSet {
    constructor(scope, id, props) {
        super(scope, id, {
            ...props,
            recordType: RecordType.CNAME,
            target: RecordTarget.fromValues(props.domainName),
        });
    }
}
exports.CnameRecord = CnameRecord;
/**
 * A DNS TXT record
 *
 * @resource AWS::Route53::RecordSet
 */
class TxtRecord extends RecordSet {
    constructor(scope, id, props) {
        super(scope, id, {
            ...props,
            recordType: RecordType.TXT,
            target: RecordTarget.fromValues(...props.values.map(v => JSON.stringify(v))),
        });
    }
}
exports.TxtRecord = TxtRecord;
/**
 * A DNS SRV record
 *
 * @resource AWS::Route53::RecordSet
 */
class SrvRecord extends RecordSet {
    constructor(scope, id, props) {
        super(scope, id, {
            ...props,
            recordType: RecordType.SRV,
            target: RecordTarget.fromValues(...props.values.map(v => `${v.priority} ${v.weight} ${v.port} ${v.hostName}`)),
        });
    }
}
exports.SrvRecord = SrvRecord;
/**
 * The CAA tag.
 */
var CaaTag;
(function (CaaTag) {
    /**
     * Explicity authorizes a single certificate authority to issue a
     * certificate (any type) for the hostname.
     */
    CaaTag["ISSUE"] = "issue";
    /**
     * Explicity authorizes a single certificate authority to issue a
     * wildcard certificate (and only wildcard) for the hostname.
     */
    CaaTag["ISSUEWILD"] = "issuewild";
    /**
     * Specifies a URL to which a certificate authority may report policy
     * violations.
     */
    CaaTag["IODEF"] = "iodef";
})(CaaTag = exports.CaaTag || (exports.CaaTag = {}));
/**
 * A DNS CAA record
 *
 * @resource AWS::Route53::RecordSet
 */
class CaaRecord extends RecordSet {
    constructor(scope, id, props) {
        super(scope, id, {
            ...props,
            recordType: RecordType.CAA,
            target: RecordTarget.fromValues(...props.values.map(v => `${v.flag} ${v.tag} "${v.value}"`)),
        });
    }
}
exports.CaaRecord = CaaRecord;
/**
 * A DNS Amazon CAA record.
 *
 * A CAA record to restrict certificate authorities allowed
 * to issue certificates for a domain to Amazon only.
 *
 * @resource AWS::Route53::RecordSet
 */
class CaaAmazonRecord extends CaaRecord {
    constructor(scope, id, props) {
        super(scope, id, {
            ...props,
            values: [
                {
                    flag: 0,
                    tag: CaaTag.ISSUE,
                    value: 'amazon.com',
                },
            ],
        });
    }
}
exports.CaaAmazonRecord = CaaAmazonRecord;
/**
 * A DNS MX record
 *
 * @resource AWS::Route53::RecordSet
 */
class MxRecord extends RecordSet {
    constructor(scope, id, props) {
        super(scope, id, {
            ...props,
            recordType: RecordType.MX,
            target: RecordTarget.fromValues(...props.values.map(v => `${v.priority} ${v.hostName}`)),
        });
    }
}
exports.MxRecord = MxRecord;
/**
 * A record to delegate further lookups to a different set of name servers.
 */
class ZoneDelegationRecord extends RecordSet {
    constructor(scope, id, props) {
        super(scope, id, {
            ...props,
            recordType: RecordType.NS,
            target: RecordTarget.fromValues(...core_1.Token.isUnresolved(props.nameServers)
                ? props.nameServers // Can't map a string-array token!
                : props.nameServers.map(ns => (core_1.Token.isUnresolved(ns) || ns.endsWith('.')) ? ns : `${ns}.`)),
            ttl: props.ttl || core_1.Duration.days(2),
        });
    }
}
exports.ZoneDelegationRecord = ZoneDelegationRecord;
//# sourceMappingURL=data:application/json;base64,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