import json
import pickle
import uuid

import pandas
import wizata_dsapi

from .api_dto import ApiDto
from .dataframe_toolkit import DataFrameToolkit
from .ds_dataframe import DSDataFrame
from .mlmodel import MLModel
from .plot import Plot
from .request import Request
from .script import Script


class Execution(ApiDto):

    def __init__(self, execution_id=None):

        # Id
        if execution_id is None:
            execution_id = uuid.uuid4()
        self.execution_id = execution_id

        # Experiment information (required only for queries processed from backend)
        self.experiment_id = None

        # Inputs Properties (load)
        self.script = None
        self.request = None
        self.input_ds_dataframe = None
        self.ml_model = None
        self.function = None
        self.isAnomalyDetection = False

        # Outputs Properties (generated by Execution)
        self.models = []
        self.anomalies = []
        self.plots = []
        self.output_ds_dataframe = None

    def _get_dataframe(self):
        if self.input_ds_dataframe is None:
            return None
        else:
            return self.input_ds_dataframe.dataframe

    def _set_dataframe(self, value):
        if not isinstance(value, pandas.DataFrame):
            raise ValueError("dataframe must be a panda dataframe.")
        self.input_ds_dataframe = DSDataFrame()
        self.input_ds_dataframe.dataframe = value

    def _del_dataframe(self):
        del self.input_ds_dataframe

    dataframe = property(
        fget=_get_dataframe,
        fset=_set_dataframe,
        fdel=_del_dataframe,
        doc="Input Pandas Dataframe (for id fetch 'input_ds_dataframe.df_id')"
    )

    def _get_result_dataframe(self):
        if self.output_ds_dataframe is None:
            return None
        else:
            return self.output_ds_dataframe.dataframe

    def _set_result_dataframe(self, value):
        if not isinstance(value, pandas.DataFrame):
            raise ValueError("dataframe must be a panda dataframe.")
        self.output_ds_dataframe = DSDataFrame()
        self.output_ds_dataframe.dataframe = value

    def _del_result_dataframe(self):
        del self.output_ds_dataframe

    result_dataframe = property(
        fget=_get_result_dataframe,
        fset=_set_result_dataframe,
        fdel=_del_result_dataframe,
        doc="Output Pandas Dataframe (for id fetch 'output_ds_dataframe.df_id')"
    )

    def append_plot(self, figure, name="Unkwown"):
        plot = Plot()
        plot.name = name
        plot.experiment_id = self.experiment_id
        plot.figure = figure
        self.plots.append(plot)
        return plot

    def append_model(self, trained_model, input_columns, output_columns=None, has_anomalies=False, scaler=None):
        ml_model = MLModel()

        ml_model.trained_model = trained_model
        ml_model.scaler = scaler

        ml_model.input_columns = input_columns
        ml_model.output_columns = output_columns

        ml_model.has_anomalies = has_anomalies

        self.models.append(ml_model)
        return ml_model

    def api_id(self) -> str:
        return str(self.execution_id).upper()

    def endpoint(self) -> str:
        return "Executions"

    def to_json(self, result=False, api=False):
        """
        Convert to a json version of Execution definition.
        By default, use DS API format.
        :param result: if set as True, use format for python client.
        :param api: if set as True, use format for backend API.
        """
        obj = {
            "id": str(self.execution_id)
        }

        if self.experiment_id is not None:
            if api:
                obj["experimentId"] = str(self.experiment_id)
            else:
                obj["experiment_id"] = str(self.experiment_id)
        if self.request is not None and not result:
            if api:
                obj["request"] = json.dumps(self.request.to_json())
            else:
                obj["request"] = self.request.to_json()
        if self.dataframe is not None and not result and not api:
            obj["dataframe"] = DataFrameToolkit.convert_to_json(self.dataframe)
        if self.script is not None and not result:
            if api:
                obj["scriptId"] = str(self.script.script_id)
            else:
                obj["script"] = self.script.to_json()
        if self.ml_model is not None and not result:
            if api:
                obj["mlModelId"] = str(self.ml_model.model_id)
            else:
                obj["ml_model"] = self.ml_model.to_json()
        if self.function is not None and not result:
            obj["function"] = self.function
        if self.isAnomalyDetection and not result:
            obj["isAnomalyDetection"] = str(True)

        if result and not api:
            if self.plots is not None:
                plots_ids = []
                for plot in self.plots:
                    plots_ids.append({"id": str(plot.plot_id), "name": plot.name})
                obj["plots"] = plots_ids
            if self.models is not None:
                models_json = []
                for ml_model in self.models:
                    models_json.append(ml_model.get_sample_payload())
                obj["models"] = models_json
            if self.anomalies is not None:
                obj["anomaliesList"] = self.anomalies
            if self.result_dataframe is not None:
                obj["dataframe"] = {
                    "id": str(self.output_ds_dataframe.df_id)
                }

        return obj

    def from_json(self, obj, api=False):
        if "id" in obj.keys():
            self.execution_id = uuid.UUID(obj["id"])
        if "request" in obj.keys():
            self.request = Request()
            self.request.from_json(obj["request"])
        if "dataframe" in obj.keys():
            self.dataframe = DataFrameToolkit.convert_from_json(obj["dataframe"])
        if "script" in obj.keys():
            self.script = Script()
            self.script.from_json(obj["script"], api=api)
        if "ml_model" in obj.keys():
            self.ml_model = MLModel()
            self.ml_model.from_json(obj["ml_model"], api=api)
        if "function" in obj.keys() and obj:
            self.function = obj["function"]
        if "isAnomalyDetection" in obj.keys() and obj["isAnomalyDetection"] == 'True':
            self.isAnomalyDetection = True

    def to_pickle(self):
        return pickle.dumps(self)
