import numpy as np

import strax
import straxen

export, __all__ = strax.exporter()
__all__ += ['ADC_TO_E', 'FIXED_TO_PE']

# Convert from ADC * samples to electrons emitted by PMT
# see pax.dsputils.adc_to_pe for calculation
ADC_TO_E  = 17142.81741

@export
def get_to_pe(run_id, gain_model, n_tpc_pmts):
    if not isinstance(gain_model, tuple):
        raise ValueError(f"gain_model must be a tuple")
    if not len(gain_model) == 2:
        raise ValueError(f"gain_model must have two elements: "
                         f"the model type and its specific configuration")
    model_type, model_conf = gain_model
    
    # Convert from ADC * samples to electrons emitted by PMT
    # see pax.dsputils.adc_to_pe for calculation
    adc_to_e  = 17142.81741
    
    if model_type == 'disabled':
        # Somebody messed up
        raise RuntimeError("Attempt to use a disabled gain model")

    if model_type == 'to_pe_per_run':
        # Load a npy file specifing a run_id -> to_pe array
        to_pe_file = model_conf
        x = straxen.get_resource(to_pe_file, fmt='npy')
        run_index = np.where(x['run_id'] == int(run_id))[0]
        if not len(run_index):
            # Gains not known: using placeholders
            run_index = [-1]
        to_pe = x[run_index[0]]['to_pe']

    elif model_type == 'to_pe_constant':
        if model_conf in FIXED_TO_PE:
            return FIXED_TO_PE[model_conf]

        # Uniform gain, specified as a to_pe factor
        to_pe = np.ones(n_tpc_pmts, dtype=np.float32) * model_conf
        
    else:
        raise NotImplementedError(f"Gain model type {model_type} not implemented")

    if len(to_pe) != n_tpc_pmts:
        raise ValueError(
            f"Gain model {gain_model} resulted in a to_pe "
            f"of length {len(to_pe)}, but n_tpc_pmts is {n_tpc_pmts}!")
    return to_pe


# Specific gain models, fixed forever. Do not remove or alter models here!
FIXED_TO_PE = {
    # First gain calibration, PMTs at 1300 V.
    # https://xe1t-wiki.lngs.infn.it/doku.php?id=xenon:giovo:first_led_run
    '1300V_20200428': np.array([0.00241, 0.0064, 0.01071, 0.01465, 0.00812, 0.03647, 0.00384, 0.0025, 0.00385, 0.00546, 0.00456, 0.00755, 0.0197, 0.00521, 0.00672, 0.04181, 0.00647, 0.00652, 0.00849, 0.01027, 0.00581, 0.01491, 0.0072, 0.00739, 0.00952, 0.00599, 0.00801, 0.00482, 0.02414, 0.01633, 0.00745, 0.00667, 0.00783, 0.01905, 0.00974, 0.00836, 0.0064, 0.00593, 0.00531, 0.00742, 0.01199, 0.00717, 0.01045, 0.01224, 0.01014, 0.0102, 0.01014, 0.00686, 0.00917, 0.01033, 0.00606, 0.00708, 0.00723, 0.00527, 0.00675, 0.01309, 0.00779, 0.01052, 0.00828, 0.00503, 0.00828, 0.02198, 0.00688, 0.00942, 0.00652, 0.01078, 0.0098, 0.00619, 0.0061, 0.01207, 0.00446, 0.00628, 0.00937, 0.00652, 0.00828, 0.00828, 0.01394, 0.01602, 0.01014, 0.00947, 0.01158, 0.00801, 0.00635, 0.01319, 0.00542, 0.01003, 0.00745, 0.00749, 0.0084, 0.00564, 0.00828, 0.00691, 0.00828, 0.00828, 0.00879, 0.00683, 0.0084, 0.01371, 0.00974, 0.00664, 0.00832, 0.01045, 0.01078, 0.00626, 0.00772, 0.00546, 0.00974, 0.00828, 0.00828, 0.00488, 0.00969, 0.00553, 0.01199, 0.01092, 0.00745, 0.00612, 0.00942, 0.00898, 0.00783, 0.01681, 0.00755, 0.01045, 0.00828, 0.00828, 0.00564, 0.00828, 0.00828, 0.01681, 0.00797, 0.00649, 0.00759, 0.00861, 0.01143, 0.00733, 0.00736, 0.01216, 0.01681, 0.00703, 0.01216, 0.00917, 0.00828, 0.00828, 0.00523, 0.00828, 0.00657, 0.01394, 0.00794, 0.00801, 0.00705, 0.01633, 0.00769, 0.00824, 0.02116, 0.00739, 0.00937, 0.00898, 0.00963, 0.01113, 0.007, 0.00783, 0.00602, 0.00717, 0.00519, 0.01843, 0.00509, 0.00521, 0.00353, 0.00487, 0.00678, 0.00879, 0.03297, 0.02721, 0.00985, 0.00549, 0.00969, 0.0056, 0.00667, 0.00861, 0.02484, 0.00697, 0.02637, 0.00759, 0.00449, 0.0011, 0.00857, 0.00828, 0.0127, 0.00688, 0.00468, 0.00606, 0.0217, 0.01843, 0.00595, 0.00581, 0.0087, 0.00599, 0.0072, 0.00423, 0.00683, 0.00644, 0.00947, 0.00755, 0.01151, 0.0084, 0.01309, 0.00717, 0.00548, 0.0112, 0.00597, 0.00912, 0.00623, 0.00922, 0.00449, 0.00888, 0.00769, 0.00927, 0.01279, 0.00765, 0.01207, 0.00809, 0.00902, 0.01602, 0.0056, 0.00765, 0.00794, 0.00551, 0.01174, 0.00509, 0.00577, 0.00597, 0.01207, 0.0112, 0.00644, 0.01027, 0.00991, 0.00575, 0.00433, 0.00828, 0.00812, 0.01948, 0.0084, 0.00591, 0.00664, 0.00691, 0.00776, 0.00548, 0.00786, 0.02348, 0.0087, 0.01092, 0.00772, 0.00786, 0.00717, 0.02198, 0.0082, 0.00902, 0.01405, 0.00688, 0.00708, 0.00985, 0.00902, 0.00694, 0.01279, 0.0098, 0.00779, 0.01033, 0.02597, 0.00729, 0.01429, 0.01039, 0.02041, 0.00542, 0.01052, 0.01617, 0.01052, 0.01106, 0.00937, 0.00828, 0.00866, 0.00884, 0.00844, 0.00902, 0.01361, 0.00776, 0.00476, 0.02317, 0.01008, 0.0135, 0.00898, 0.00647, 0.00828, 0.00861, 0.00614, 0.00958, 0.01052, 0.00617, 0.01926, 0.01453, 0.02484, 0.00652, 0.00969, 0.01039, 0.03117, 0.00828, 0.0047, 0.00805, 0.00828, 0.00614, 0.02956, 0.00907, 0.01587, 0.00947, 0.00524, 0.00717, 0.0381, 0.00902, 0.00853, 0.00917, 0.0068, 0.00828, 0.01014, 0.00922, 0.00828, 0.01884, 0.00714, 0.03499, 0.00501, 0.03571, 0.00912, 0.02065, 0.01251, 0.00568, 0.00902, 0.00783, 0.01099, 0.00521, 0.00828, 0.00686, 0.02597, 0.00912, 0.01045, 0.00893, 0.00599, 0.00812, 0.00456, 0.00714, 0.00729, 0.01174, 0.0112, 0.00451, 0.00828, 0.00942, 0.0072, 0.00932, 0.00828, 0.00937, 0.00521, 0.02597, 0.00515, 0.00927, 0.00726, 0.00659, 0.00717, 0.00917, 0.01309, 0.00801, 0.01242, 0.00664, 0.01199, 0.00739, 0.00697, 0.02956, 0.00551, 0.00591, 0.0067, 0.00474, 0.03361, 0.007, 0.01242, 0.01071, 0.00749, 0.02857, 0.00591, 0.00844, 0.00583, 0.01233, 0.00828, 0.01309, 0.00711, 0.01233, 0.00628, 0.02637, 0.00902, 0.0061, 0.01884, 0.00686, 0.00952, 0.00974, 0.00425, 0.01544, 0.0051, 0.0197, 0.0044, 0.00678, 0.01587, 0.01135, 0.00527, 0.00985, 0.01003, 0.00585, 0.00664, 0.00893, 0.00657, 0.00801, 0.01158, 0.00571, 0.00523, 0.00801, 0.00776, 0.02721, 0.00853, 0.01045, 0.0061, 0.00446, 0.00523, 0.00694, 0.00478, 0.00828, 0.00733, 0.00752, 0.0051, 0.02116, 0.01617, 0.00828, 0.0112, 0.00776, 0.00577, 0.00675, 0.03117, 0.01382, 0.0084, 0.00536, 0.00902, 0.00875, 0.03499, 0.00675, 0.0079, 0.00506, 0.02484, 0.00717, 0.01158, 0.01106, 0.00463, 0.00521, 0.00762, 0.01329, 0.00947, 0.02017, 0.00577, 0.00551, 0.0061, 0.00733, 0.00649, 0.01174, 0.00527, 0.01289, 0.00659, 0.00849, 0.00902, 0.00642, 0.01199, 0.01065, 0.00779, 0.00879, 0.00801, 0.01008, 0.01085, 0.01182, 0.00853, 0.02484, 0.00786, 0.00875, 0.02857, 0.00776, 0.00755, 0.00836, 0.01339, 0.00769, 0.01135, 0.02956, 0.00703, 0.00672, 0.01371, 0.02597]),
    # Results from HV scan
    # https://xe1t-wiki.lngs.infn.it/doku.php?id=xenon:giovo:hvscan
    '1300V_20200614': np.array([0.006659382489882943, 0.009516346729411417, 0.0126985227715348, 0.014754657947266192, 0.007015261404802661, 0.030576533060808188, 0.006137914243572641, 0.006902663892828348, 0.00800886999014765, 0.007207833741488562, 0.005825006028663779, 0.006646929721723732, 0.019063239956629764, 0.0056594142964485365, 0.00643254683593419, 0.03577618727547231, 0.007304919634212174, 0.007842123242809992, 0.009724279762958934, 0.012641602330850283, 0.006177489000587177, 0.01575490779300332, 0.008049629119972479, 0.00685071816173829, 0.009758243163289788, 0.0060944436438814895, 0.007019027934792125, 0.004867990090053548, 0.029360293825165532, 0.015921002599888774, 0.007924482873340896, 0.0076118825567819525, 0.009695316106724439, 0.021923463267142358, 0.009649232237155804, 0.009307666593732165, 0.005770306038106889, 0.005702994470173205, 0.005978776666889018, 0.00627028330669146, 0.013665235503118132, 0.00806004033777127, 0.01135165085184168, 0.013091103421048777, 0.011474643152407975, 0.010301110676135958, 0.011841250247514526, 0.006967999940117469, 0.010466580621966369, 0.009706828947054687, 0.006244064750987698, 0.0076755529396450185, 0.00785572145901305, 0.00531879749165125, 0.008081228962028302, 0.015165737099090922, 0.007933436317690747, 0.009107840093998844, 0.008997236531780087, 0.005838131254421794, 0.01431325457702854, 0.02302897764970398, 0.007665966888515642, 0.010719698564390838, 0.006532804737318578, 0.011535308549068713, 0.009363040289419498, 0.0060078652318925705, 0.006788794388875758, 0.012068771302320563, 0.005357578046406048, 0.006218503903361452, 0.009233025382481844, 0.0070272056170273085, 0.011222146028999305, 0.00971228237186992, 0.017650118914633493, 0.01849886531719377, 0.010728579756930805, 0.009359577689693916, 0.012921123996893881, 0.008830940607966284, 0.005858941603045496, 0.013710896917321014, 0.00569876055852159, 0.007897713891774882, 0.0077184366772052905, 0.008184017698575893, 0.008912902536229709, 0.005820930715549343, 0.00858218348916142, 0.009039964249357811, 0.007737394067932338, 0.008321979279133272, 0.009312282924988199, 0.007546913575779297, 0.009737758904084076, 0.014541161950369101, 0.009520191648752063, 0.007636215501054581, 0.008353253727665879, 0.006595721714325259, 0.011759231438368269, 0.006330572926386776, 0.008513475708035893, 0.005697081302492067, 0.010639679080049001, 0.007605424817728583, 0.00666124660886643, 0.00727963057919672, 0.010194403806221955, 0.005866932481988484, 0.013173902534786544, 0.01171526564593427, 0.00773262172068053, 0.005738499280879939, 0.008602344045507694, 0.008678276516015326, 0.00852866245292504, 0.017095209258635495, 0.007342379042942051, 0.010751360464228943, 0.020968917999839534, 0.013590885888342298, 0.0077815906111715115, 0.021617722010714296, 0.006125092046335478, 0.01918304717474248, 0.00814517698581393, 0.007290163745492949, 0.008547345525520644, 0.008800732821247109, 0.012508761114640683, 0.007954455561840459, 0.006169066509273942, 0.012086177231607802, 0.015667886029742666, 0.008554988062772592, 0.011637488076157903, 0.010221001496368474, 0.006071393411427476, 0.006762930998668586, 0.006734077053114836, 0.008642137425256118, 0.007021954236031411, 0.014422598615748896, 0.00991767342122608, 0.0089692482845839, 0.00932235781143847, 0.021580827133171755, 0.009627568389285444, 0.00940266761737659, 0.02030730175453069, 0.0076249787950457785, 0.009607049565134201, 0.010190159774592774, 0.011152613159493945, 0.012554991490467467, 0.008256026074709297, 0.007832299461219246, 0.0066409023341441045, 0.009137090333859312, 0.006068459525413403, 0.021412025097191298, 0.0068243863664177295, 0.008552176583648138, 0.009288553938321146, 0.008228754855112784, 0.0068558426181006835, 0.006297353866293757, 0.031526004125255004, 0.027037328208835157, 0.010299531817578257, 0.006431797083959897, 0.010333565723623659, 0.007289032282748822, 0.007675318608492869, 0.009878815101141069, 0.030738693844465387, 0.009147256709903937, 0.026658835547078066, 0.009709011953209844, 0.009467156209462873, 0.005942066133771048, 0.008177224196077892, 0.0, 0.014671165245061181, 0.006251025664425353, 0.006121340623615174, 0.007746784715107708, 0.02382933385235206, 0.021411865689168978, 0.007956327745823678, 0.007233330613731701, 0.010194067766809548, 0.0063234822073327986, 0.008148510207039236, 0.006327636389516171, 0.016466892798879595, 0.007403012941502159, 0.00943550743424886, 0.009547522299536159, 0.011439601482183225, 0.009705984450658971, 0.016168576498165322, 0.008484236869163634, 0.006993454186347751, 0.013352314182118056, 0.007795289681190731, 0.011330265981504868, 0.008241218659944913, 0.010930181092862881, 0.0056987108755424615, 0.009541178177092869, 0.008439344456115145, 0.009529302565190063, 0.015115497637109033, 0.008945146726250734, 0.012126283420758938, 0.008089175006410095, 0.01025190695943988, 0.013448565740226565, 0.006904816223777369, 0.008714296200600801, 0.009150975066715116, 0.007080699217599363, 0.009372751900509162, 0.006108631026542291, 0.0059796477908265816, 0.0067968417828469735, 0.011435235975157406, 0.012477817001294664, 0.006270662812120812, 0.010023585402266371, 0.011905110898545688, 0.008728081623664856, 0.006107030929606876, 0.006678468821886783, 0.008191839667906462, 0.019216485684474593, 0.008298544771028569, 0.006287583690842643, 0.0071158056901832155, 0.007446137034176457, 0.0091507008016136, 0.006092344525260846, 0.01016055236200614, 0.016772185974814578, 0.007345216541545783, 0.009601675470304534, 0.007960304768119425, 0.008284343339541683, 0.007312502572120277, 0.017776343475511297, 0.007550443374611413, 0.007126249735853846, 0.013134601675821464, 0.005677456757182427, 0.006470982993088532, 0.00754477067963279, 0.008892347695325934, 0.006005223397215947, 0.011962115876951564, 0.008759196440156025, 0.0066998192190332985, 0.00872795024929162, 0.023372023908976493, 0.007619930798090623, 0.014312579661781006, 0.010222115288887872, 0.019515226955620052, 0.005329448105804, 0.009157120350970344, 0.01286945722540382, 0.009063785449458376, 0.010857800608075339, 0.0112737951052484, 0.009096654989110898, 0.00873625791833905, 0.00808626570572481, 0.008106509438168027, 0.009234171427642015, 0.012202943750394993, 0.007462021144387836, 0.00439992076504142, 0.020835813898102167, 0.009368395234990372, 0.012240815864922256, 0.008056872977909867, 0.007729469601003508, 0.008608433897626845, 0.009527369407761785, 0.008225162784188741, 0.008953292895656885, 0.010119953720715552, 0.006113662989432306, 0.018271419962832488, 0.01539742296218728, 0.02159183444042997, 0.006224676027239651, 0.00846756510801204, 0.008969138079685733, 0.026792648591212456, 0.010290268366204404, 0.005766745302637647, 0.010841432458922341, 0.028252154472143742, 0.00816419191672793, 0.02924604846279972, 0.008636392865628704, 0.017405846411285564, 0.009592805156646799, 0.005586358725947815, 0.008013712412325784, 0.033215401530276835, 0.008748527705383532, 0.007118526413393173, 0.0076155032493673575, 0.00884464740311114, 0.00943857977268513, 0.014373412746406307, 0.012803439901264682, 0.009701578943501472, 0.0177584021001395, 0.007399209941321354, 0.03755435856458304, 0.005374433807979138, 0.03447689296206782, 0.010997343918719002, 0.019512694139083797, 0.011698938539253323, 0.005584012066541802, 0.008011049025135922, 0.0066717792376343515, 0.01896140921079221, 0.007586890754106359, 0.02948459406702185, 0.008907075597538198, 0.033520269667305064, 0.01089816079286445, 0.010370757770126573, 0.009493588578474358, 0.00670007518635729, 0.008186082462207391, 0.005024583383926628, 0.007062659817641318, 0.007217900609023465, 0.010262142702543404, 0.01077559364239475, 0.005849177098351076, 0.006722954992640727, 0.015052828833827971, 0.009473560138852265, 0.009530466499649197, 0.0, 0.009519488331279429, 0.006289697933628242, 0.02563920588488449, 0.005849759090491589, 0.009960070910512922, 0.007465208427276583, 0.006234140180489553, 0.0060956702190099596, 0.008613905939733888, 0.012429332604204407, 0.008239253448472815, 0.01173860810722613, 0.008388922567940027, 0.013094583025980096, 0.00881476789988069, 0.008012902405266175, 0.032145384445807335, 0.006010576719133816, 0.005636043903218614, 0.006983471447704166, 0.005305952682234555, 0.03260941801489846, 0.006603808873584932, 0.013207532834568624, 0.010114816583585711, 0.00787713027074675, 0.025705948320843882, 0.006319515094883218, 0.010237530567116666, 0.00710889418886595, 0.0137435381130423, 0.0, 0.014406380042395823, 0.007906548662841732, 0.013795556481940195, 0.006934741930416982, 0.028523484514212962, 0.009756916349886683, 0.006875638790168898, 0.016192466310408108, 0.0066998922613729665, 0.007879766652687798, 0.008549903720317172, 0.005320374889753741, 0.014536859455702617, 0.00568623990474591, 0.020899413414625624, 0.005337919387252008, 0.007523010155344642, 0.01865388793091925, 0.012535123895400164, 0.005801647082419162, 0.009667282830698398, 0.010846633217020034, 0.00555022392165494, 0.006294610066104299, 0.008536692195952494, 0.006403806717262391, 0.0086329870537438, 0.012562236682132374, 0.005836636073509619, 0.005173459990228816, 0.009476129632159738, 0.00824256018818573, 0.032171298756387724, 0.008384014091021201, 0.009818796834657766, 0.0069100631358833065, 0.004972100047926917, 0.005928860060485513, 0.006518233607487652, 0.0051475424214461555, 0.0, 0.006749468123913969, 0.00854811678656184, 0.006366680161086934, 0.019825362931620877, 0.015499285505962976, 0.010085304519050776, 0.011881061002036637, 0.00791195321878817, 0.005883180054782882, 0.008086664951094863, 0.03253733258821518, 0.012772120527864646, 0.00895140749335495, 0.005804636655003176, 0.00863998408633922, 0.008575938873194924, 0.03338335740061464, 0.009075870923084255, 0.01027237762542383, 0.006344938231483613, 0.027742550460710034, 0.007732537730580697, 0.012161153073325529, 0.012581329553463404, 0.00525983260093505, 0.0049250085887811565, 0.007227655604341545, 0.012994179031588518, 0.008306608412741718, 0.025727892568050446, 0.006378089945728218, 0.006714790531151561, 0.007025811712217171, 0.007556557396570518, 0.00755059481611, 0.014395262714284917, 0.006660255339026092, 0.013166531541276486, 0.007515671912119837, 0.008540380001636446, 0.009489799556326072, 0.006549392547826409, 0.012379591712481508, 0.009686645225488195, 0.0078011846770811724, 0.00836201726268873, 0.009163228935307514, 0.011251299793285537, 0.011540794532442271, 0.012547840051152684, 0.008223023717050276, 0.024670316092483897, 0.006929244461237489, 0.007810584330844709, 0.028671779379188064, 0.008106685835680033, 0.007570708709048857, 0.008672014266305653, 0.012345383308824006, 0.007795514491689613, 0.009551835921460444, 0.02755466053510842, 0.007122030064164409, 0.006555492165383512, 0.014012307102255723, 0.028103005197169375]),
    '1400V_20200614': np.array([0.00364657408771866, 0.005153532260280403, 0.0069737775665415285, 0.008166153940847654, 0.003832694952237084, 0.017919663819897, 0.003394468024000821, 0.0037669468250097014, 0.004361791357295889, 0.0040007492227474555, 0.003144186120282208, 0.003617410958604763, 0.010732830015137812, 0.0030828613282825705, 0.003520848921197539, 0.020426166705572468, 0.00403606465938768, 0.004280984134766605, 0.00537811849707454, 0.006978858889925133, 0.0033644087241229094, 0.008630849748683944, 0.004422937132601492, 0.003763796599833862, 0.005306037257881762, 0.003378516294999383, 0.0038593939983404234, 0.002663488401659824, 0.015532627120606834, 0.008755382316208907, 0.004502128845263302, 0.004207297559132231, 0.00530149748124421, 0.012337174012709028, 0.005228503770304003, 0.005192356907150509, 0.0030605561872978507, 0.0031589101961899246, 0.003280172365131184, 0.0034277219416390336, 0.007482398703583989, 0.004396110212824413, 0.006401864726038557, 0.007329174535376298, 0.0062105303351932435, 0.005672269775844657, 0.006570012611123943, 0.0038393949877328585, 0.005653342180747612, 0.0053696864621157634, 0.0034450494498618417, 0.004203411895544201, 0.004330161541196614, 0.0028646537713386266, 0.004380123006812338, 0.00840768295814193, 0.004349276131630464, 0.004966107530301824, 0.004983225970791793, 0.003130914817933361, 0.008136370151504722, 0.012744392246305879, 0.004224927676964288, 0.005981537092353988, 0.0036142009132077977, 0.006347956983708264, 0.005129580404234135, 0.0032956170892618204, 0.0037071229437475564, 0.006739061276925891, 0.002894829925096534, 0.003409454246233705, 0.00508291295122753, 0.0038195753911905834, 0.00621909017268975, 0.0053915605045985375, 0.009835111658442476, 0.010321456243749697, 0.005846524149116911, 0.0050431332719392515, 0.00718655624754399, 0.004758893908223607, 0.0031613985692831924, 0.007671342896160527, 0.003137290153348062, 0.004391352606413159, 0.004146282464207678, 0.0043705366440171985, 0.004853518327048975, 0.0031513229441629447, 0.004840895257391626, 0.004933434631444343, 0.00428448044870896, 0.004584359480267856, 0.005051461884195445, 0.004082915599561476, 0.005215571377935827, 0.007814624180829188, 0.005332162505586169, 0.0041665523265447075, 0.004619694125538871, 0.0036063426550400532, 0.0063856995553128095, 0.0033874641425475733, 0.004564608911628262, 0.0031035757875819033, 0.005890836141997885, 0.004226874389345578, 0.003658381329231053, 0.003786247753258174, 0.005526790951840314, 0.003240329472730026, 0.00731410780017014, 0.006373986235983629, 0.004349930516120652, 0.003093153273092414, 0.00474645430610976, 0.004759643656248287, 0.004658006180388885, 0.009560403682236115, 0.004063451274544272, 0.005841421926737908, 0.011873843694280998, 0.007637924634572783, 0.004225261853443071, 0.012220349691263246, 0.0032841132568121986, 0.011046493673090626, 0.004412840347065628, 0.004024864125280718, 0.004677162957297653, 0.004891237900409624, 0.006949807979072471, 0.004452651954582186, 0.0034113688389930943, 0.0067477680409401, 0.008565027988693241, 0.00464293763917589, 0.006426099966170372, 0.0055629047666670185, 0.003394389182229981, 0.003694007441944061, 0.0035503349195331597, 0.004877453477262727, 0.003744441712691906, 0.008014370719486644, 0.005427082171231647, 0.004842789259873711, 0.005132845656386585, 0.01157144642725187, 0.005281330276430483, 0.005240431959255412, 0.011294395843360972, 0.004184743364652423, 0.005341710909602378, 0.005685642336368778, 0.006075465394058205, 0.00698686239322182, 0.0044858087527682895, 0.004326641594134081, 0.003650863419007238, 0.0049117111011187, 0.003288779207774885, 0.012027612344953045, 0.0036779536940815785, 0.004610696848525922, 0.005121395476319635, 0.004590839068530932, 0.0037923565486404458, 0.0035162816088253825, 0.017856621194364874, 0.015247954135818746, 0.005714374908988222, 0.003445457533083168, 0.005641265281348761, 0.0038778268804126065, 0.004299636519301696, 0.005352204167660582, 0.01708451819406149, 0.0050305877140324005, 0.01502829479938095, 0.00528337735887197, 0.005231063011810303, 0.002752301649144239, 0.00448064933454268, 0.0, 0.008038319052548818, 0.0034982762714547474, 0.0033355983708151056, 0.004172137580850978, 0.013470583951940774, 0.012485647613998813, 0.00432565977187545, 0.0038895825520726996, 0.005443206646693509, 0.0035402356064045985, 0.004395587768880133, 0.003457107407672188, 0.009285212749343148, 0.004019828229745902, 0.005135292648582561, 0.0052380825708046586, 0.006351433027550193, 0.005250830169438729, 0.00900389912909989, 0.004659269377877967, 0.003910376734916024, 0.007418186050300931, 0.004234146058528923, 0.006310596465425379, 0.004558723877995352, 0.006052240541732076, 0.003090432645103755, 0.0052815091145923644, 0.004658127188278395, 0.005299962508222607, 0.008353459466282137, 0.004847540070616695, 0.006820245436581316, 0.00450862809601828, 0.005580819110975164, 0.007342115131289916, 0.0037021003279928966, 0.004873781382644835, 0.004916748786600193, 0.003937958563314117, 0.005309865168681121, 0.00329587903721807, 0.003316622729258494, 0.003711514221123819, 0.006333678445136894, 0.006881445312842375, 0.0034787873077025645, 0.0054128786888853475, 0.006373802602868175, 0.004851315750608478, 0.003384190913308295, 0.003704183383037467, 0.004431387392936705, 0.010493820716270872, 0.004581902047523992, 0.0035194795149549535, 0.0038065324446511736, 0.004048815927167978, 0.00501685865622727, 0.00334945331810228, 0.005699558766576662, 0.009049687715476065, 0.003971617395955204, 0.005349762271384774, 0.004423062832162685, 0.004616038195115698, 0.004057910730825987, 0.010141724553642257, 0.004153965499105299, 0.003967827096625373, 0.007130008447273889, 0.003140142491146394, 0.0036101078122205204, 0.004173503894595319, 0.004893880559597717, 0.0033152360374514096, 0.006707339067334821, 0.004799641110305308, 0.003746525068966135, 0.004875448628747317, 0.013447962067440205, 0.004177041283371012, 0.00830702971320099, 0.005889788544674809, 0.011275550662744238, 0.0029182739752290055, 0.005066316002249783, 0.007158229106553461, 0.00506196103746613, 0.006006179006434216, 0.006239004320407595, 0.00498051563505895, 0.004848429494822983, 0.004548458408305383, 0.0045443550439087255, 0.005090696978503141, 0.00672106279357742, 0.004051981063570291, 0.002395720366092177, 0.011736257128531593, 0.005196447585559358, 0.006716502895722163, 0.004501983390006765, 0.004221695896241581, 0.004687672920558835, 0.005299084831353634, 0.004527910423781739, 0.0049324788970459945, 0.005647704479457895, 0.0034004121971716325, 0.009822656596232897, 0.008672533072957471, 0.01193576647346375, 0.003523558125233808, 0.004751692312099732, 0.0048416632436728575, 0.015056210972791674, 0.005532359988116741, 0.0031323745320020655, 0.005852132513816381, 0.01568827091185737, 0.004402811931495148, 0.016550005372506484, 0.004836400452958348, 0.009620976512294254, 0.005241022500875595, 0.003078886518607016, 0.004387360634054866, 0.0193565033596114, 0.004779138409750302, 0.003996351767760958, 0.00422714922935811, 0.004758801308682039, 0.005126869417333482, 0.00790469287750301, 0.006901581274695548, 0.005446698266107089, 0.009898077018475579, 0.004071026505374972, 0.02187109606254742, 0.0029938675948822054, 0.019993043012328196, 0.005826895187963717, 0.01092321036705389, 0.006296253743580676, 0.0030728248978125474, 0.004319146589367147, 0.0036763834345293527, 0.010359322147201403, 0.004024080235487021, 0.01706509367978868, 0.004916020994127166, 0.01952327065560419, 0.005843010250642681, 0.005654837418225779, 0.005089462469014848, 0.003621005968791472, 0.004535801918407842, 0.002802575405348733, 0.0038678228738393996, 0.0039226308938371595, 0.005672315418965656, 0.00596845292574195, 0.003187131559111443, 0.003591376022928035, 0.008272635834344453, 0.0051012930757817244, 0.005389417033164173, 0.0, 0.005173004783846368, 0.003479282887203982, 0.01435462884601323, 0.00315353849888925, 0.00556622289189217, 0.004034691295795574, 0.003401242516362241, 0.003173990821489383, 0.004737734510966822, 0.00715789774512938, 0.0044747160503110445, 0.00639523401140713, 0.004584541584709736, 0.0074361398409001295, 0.004843231918156406, 0.004477187224182041, 0.018725202762151026, 0.0032613179336408366, 0.003028336924382972, 0.003796018470689102, 0.0028776655184133927, 0.017961768492475088, 0.003735804901002484, 0.00702603826993877, 0.005532548143138397, 0.004411565384196546, 0.01468252859071661, 0.0033904082731971794, 0.0056902120458426735, 0.0038847907212879403, 0.007662747185739387, 0.0, 0.00810288813159923, 0.004435909786459532, 0.007538055293894747, 0.003720328445143348, 0.01588684341242503, 0.005300664233847095, 0.0036176981656501884, 0.009039642046846848, 0.003686749311233475, 0.004140451428284342, 0.004640028923865917, 0.002943154841918804, 0.007939721913460913, 0.003092434095999199, 0.011805752373590524, 0.0029618449008498955, 0.004103125423201264, 0.010394378065157237, 0.006819099267616606, 0.003131188325459035, 0.0052510568008377325, 0.006058301260650985, 0.003004227566208296, 0.0034247273597906055, 0.004600331928242977, 0.0035281602997204926, 0.004772776298414735, 0.00701748316841715, 0.0032158579812073197, 0.0028001232003554365, 0.005341795438488821, 0.004680119344368232, 0.01871418463222438, 0.0045972399985625195, 0.00531631207667304, 0.003695937007011035, 0.002679768229308341, 0.0031494360300685074, 0.003528475065239343, 0.002753187116076178, 0.0, 0.003750437372524337, 0.004660762252348776, 0.0034936053497511294, 0.011217504573619443, 0.008522248516394802, 0.005630784557935166, 0.00681753005510913, 0.004316639163651508, 0.003219112997285344, 0.004347733174486304, 0.01803745987253181, 0.006996283633181049, 0.004964593317990102, 0.003159771988557551, 0.0047082388576154205, 0.004676757738568041, 0.019043463242817138, 0.00494038939433542, 0.005545680725972902, 0.003459783294521938, 0.015725326731538203, 0.004314006780289235, 0.006847878680808874, 0.006849105005979437, 0.0029026923234024725, 0.0027043159633544115, 0.004022365753067892, 0.007110833175041574, 0.004529883575832916, 0.014155089425077191, 0.0035028897312170374, 0.0036917807523732128, 0.003865540957472244, 0.004192430878657677, 0.004177988351383208, 0.008111004941705462, 0.0036927537926396037, 0.007296424542607478, 0.004195373184670943, 0.004655973825679769, 0.005321400621576756, 0.003530213693006696, 0.006905691592741904, 0.00530322499786678, 0.004294590722178729, 0.004629003421693797, 0.005092750054295805, 0.006156463359374414, 0.006395742572164722, 0.006992491145244544, 0.004540080759588459, 0.01415990082582352, 0.003955053974454174, 0.004345689069417848, 0.01642566940360353, 0.004522746500974208, 0.0041345996803331285, 0.004777043116525109, 0.00681838925011447, 0.004264326674152678, 0.005268614619666487, 0.015697540760477406, 0.003871762151044611, 0.0035937235185151293, 0.00783069838515839, 0.01565736395624899]),
    '1500V_20200614': np.array([0.002085906119267193, 0.0029496084017914597, 0.003952448116271957, 0.004824403113401857, 0.00217780505384671, 0.010232598126577432, 0.0019554455127201095, 0.0021539430268551502, 0.002417991555717992, 0.0022305023927096553, 0.0017361841649091428, 0.0021111605641224653, 0.006304445776362535, 0.0017994422440659598, 0.002041309197617747, 0.011836973311412393, 0.0023108150818581326, 0.0024323091074616096, 0.0029485299571002906, 0.004008726893016484, 0.0018674388598860186, 0.004943297845755464, 0.0025372432404134415, 0.0021918245164027133, 0.003010778307598621, 0.0019297792433476272, 0.0021834161123340246, 0.0014942692290827032, 0.008911392486830982, 0.005009112627211388, 0.0025437706353522356, 0.002434524955867615, 0.0030056850937882104, 0.007081327526960166, 0.0029693538612311295, 0.0029417693499565406, 0.0018115528604149328, 0.0017784691651530115, 0.001832194277379499, 0.0019639576155885685, 0.004370307125157561, 0.002464799014812319, 0.003734420702276067, 0.004218233948117498, 0.003673461998658046, 0.0032007967100049404, 0.003687956135586555, 0.0021489186176972797, 0.0032373322476253095, 0.0030342046665441788, 0.0019524761958859395, 0.0023635643586272627, 0.002526540702566303, 0.0016175112330621434, 0.002440857185545587, 0.004983091627196609, 0.0024566971652106094, 0.0027790735077032644, 0.002839823092932199, 0.0017484774887587104, 0.004710058536327132, 0.007454654493597317, 0.0024156354109320264, 0.003327235214779573, 0.0020074136689291235, 0.0035557963735973756, 0.0029861574075790514, 0.0019098670338939135, 0.002142158020260367, 0.0038627524205708673, 0.0016114971327446305, 0.0018996608288017814, 0.0029695012647492744, 0.002155902617992124, 0.0035914413192844833, 0.0031206881956971134, 0.005429766978405164, 0.005875158327524598, 0.00326146764151243, 0.0028299635290610463, 0.0042191987994413096, 0.002807655410565436, 0.001834096994255787, 0.004538861298449682, 0.0018506432930421165, 0.0025173150358710826, 0.002360043348841007, 0.0025982038886333573, 0.0027042549374500166, 0.0017781642811156525, 0.0027875819041915657, 0.0027441181544758525, 0.0024949850300616566, 0.0026589618961289935, 0.002878020976255783, 0.002282330970787667, 0.002913634836048282, 0.004671581564672208, 0.003065113181772031, 0.002481217860874142, 0.002663121618737516, 0.0021026204687735373, 0.003771913161451095, 0.001930716949441709, 0.0026115738371577154, 0.0017934462025910388, 0.0033376183445005094, 0.002446787399968141, 0.0021074772887702893, 0.0020373408915177384, 0.0031582269824773092, 0.0018228767570471025, 0.004143280136121194, 0.0035464008610378585, 0.0025084051753655927, 0.0017793758140849172, 0.0028261173560890592, 0.002804218866494978, 0.002678011000967532, 0.005618013504706242, 0.0023311588777551306, 0.003348260743915169, 0.0070896650877023465, 0.004480754042874426, 0.0022573690866521893, 0.007324667923613205, 0.0019070400891162025, 0.006334615885154824, 0.0024960810346332826, 0.0022852524007158687, 0.002563574178600082, 0.0028194093836847573, 0.004183536658250434, 0.0025801036906804957, 0.0019104633980705857, 0.003925826367673606, 0.004981595342833323, 0.002631697646897424, 0.0037463025360539334, 0.0031345031155350315, 0.0019524227544583236, 0.00214411885077283, 0.0019496486716782245, 0.002840787104296206, 0.0020691457319821555, 0.004505932055747864, 0.00309295714909413, 0.0027168849063559943, 0.002895611938824199, 0.007055893357983793, 0.00311311931509635, 0.003000607580200689, 0.0065605225088498096, 0.0024061233867442152, 0.003038507032306309, 0.003196698633840876, 0.003481203243805577, 0.003868392213032792, 0.0025614955846030713, 0.002484745205108528, 0.001979008536563463, 0.002744645666938453, 0.0018310190660099587, 0.006758939381307689, 0.002061008384525737, 0.0026090603112981494, 0.002894261749644281, 0.0026547459458047084, 0.002188238855600996, 0.001988109782742354, 0.010800064462871193, 0.00897070917490104, 0.0032582761626448213, 0.0019237703137838203, 0.0031583013390912547, 0.0021539508989034894, 0.0024163548296819836, 0.003056860605443962, 0.00989706541191255, 0.0028859342141818833, 0.009070473757145052, 0.002949605255785661, 0.0029189739053318807, 0.0013670707489532319, 0.002585716417529998, 0.0, 0.004635737832691273, 0.0019517395134533845, 0.0018719853273635057, 0.0023472280166440293, 0.007428210200397244, 0.007542826021778212, 0.0023831787418589867, 0.0021680588709320917, 0.003037176323132788, 0.0020065440137004946, 0.0025235844495933844, 0.0019074482547054544, 0.0051307810007371535, 0.0023139663081965718, 0.0029486233040324286, 0.002966903859064441, 0.003537497224751838, 0.002984604523114847, 0.005024117488252195, 0.0026722576414425856, 0.00213704113759646, 0.0042319051920824795, 0.002409611973982524, 0.003631266138370235, 0.0025726795731613257, 0.0034386384843055734, 0.001775104527175836, 0.003034320419650108, 0.0026352171729016934, 0.003005025100436563, 0.004827441672384273, 0.002727294922988591, 0.003957262487870031, 0.002545943145756636, 0.003184029128069157, 0.004353319254562286, 0.0020916661003556533, 0.002752916209370159, 0.00276692406229051, 0.002195890747954383, 0.003054805243861106, 0.0018763398895043548, 0.0018276167312407678, 0.0020736848440358765, 0.003644368363286227, 0.003973323132651654, 0.0019646831520363163, 0.0031094485340531952, 0.003641907773189352, 0.0027941043611759138, 0.0018450911895410316, 0.0021457435818873646, 0.002490763675629057, 0.006008785527312767, 0.0025973364768485875, 0.0020126253882395205, 0.0022197411905526426, 0.002278284796021994, 0.0028384379120278912, 0.0019159342681869763, 0.00325315603852368, 0.005240725685508535, 0.0022909384687946504, 0.0030276123823173517, 0.0025624993911514482, 0.0026460858170606104, 0.00232158859917268, 0.005881147916605516, 0.0024370603439857782, 0.0022798471808779396, 0.004100058526190502, 0.001784016264922319, 0.0020463514737801575, 0.002361407031540062, 0.0028341798627856143, 0.0019007905559032053, 0.0038862067400276634, 0.0027584526420330667, 0.002157416178259612, 0.0027343211664965926, 0.007927104732952599, 0.0023859241115383096, 0.004882428061990786, 0.003587168701665862, 0.0069076964931178144, 0.0016727947527414011, 0.002973650992572772, 0.004074127671125862, 0.002937433232254822, 0.0034880823351646118, 0.0036209392282888675, 0.0027974676009824913, 0.0028363866166256505, 0.002621323301714016, 0.0026083680566854573, 0.0029279474460432078, 0.003975481763215634, 0.0023005834014553253, 0.0013414055227751306, 0.006613273251509428, 0.0029915382881996752, 0.0038135016324938537, 0.002724865853362915, 0.0024117638294880914, 0.0026309361452079437, 0.002988109479594258, 0.0025090346988631313, 0.0028865934521716194, 0.0032695535947252466, 0.001964610574269569, 0.005612488943465638, 0.0050476135668763465, 0.007003659926059319, 0.0020622270332848057, 0.0027512083010835975, 0.002759542860349935, 0.008779958658040266, 0.0031112232025844685, 0.0017575056450166813, 0.0032883676895303273, 0.00880329622198586, 0.002406795739949619, 0.009546215411183083, 0.002694708314600647, 0.0055146749681920584, 0.003026368641665498, 0.0016825496673117113, 0.002502773335479273, 0.011557919195091327, 0.0026348227960020713, 0.002266982991021235, 0.0024454534268902512, 0.0026796903298646015, 0.0028954288715047458, 0.00438351781343348, 0.0038475138627264496, 0.0030254543288970905, 0.0056666451907664535, 0.0023324450361296555, 0.013122841055481902, 0.0016846045869633789, 0.011797291474415196, 0.0032887160294227027, 0.006562357123945686, 0.003607526002050326, 0.0017592804278752515, 0.002461414483853556, 0.0020930372839849548, 0.005951501550991758, 0.00221726368761244, 0.009932590999395136, 0.002714287430078169, 0.01165326436119117, 0.0033646656623551297, 0.0033057697752621813, 0.0028287822468848886, 0.0020337245536485708, 0.0026204774418035663, 0.0016347534558883118, 0.0022428980795722794, 0.0022396276825914635, 0.0032461796771344597, 0.003425931289727082, 0.0018142907354703546, 0.0019832321621294975, 0.004494396125040512, 0.002901514656242194, 0.0030447737165028906, 0.0, 0.002881709044647395, 0.001963243979033229, 0.008101006586073382, 0.0017632716903311636, 0.003243073032059233, 0.0023058090050963364, 0.0019132673143853792, 0.0017329269250351107, 0.0027130756221028186, 0.004266916484328926, 0.002607173329335936, 0.003687535113400176, 0.0026566239173599062, 0.004238490000183762, 0.0027897575772819825, 0.002598125401670171, 0.011236425597874517, 0.0018149993135542295, 0.00172805651621121, 0.0020997002328370456, 0.0016250378485821145, 0.010124118079685987, 0.002160704180822827, 0.004088585594018028, 0.003160998576410285, 0.0025847054922235936, 0.00896694964694206, 0.0019336376671625318, 0.0033168748684051378, 0.0022408668957437467, 0.0045720221455967, 0.0, 0.004587154021939543, 0.0025252485631659036, 0.004237265087392108, 0.002072993002032722, 0.008939766599122385, 0.002948855971591338, 0.0020046142410292027, 0.0052340137343973355, 0.0021207043817929754, 0.002341164017620041, 0.0026063450780860224, 0.0017007543258655388, 0.004633778014240759, 0.001761680300026289, 0.006880180387878943, 0.0016694675680854446, 0.0022754591989043887, 0.0061085225628778415, 0.0038449885261370195, 0.0017678810984899994, 0.0029730108914774586, 0.003435465906023674, 0.001659978591453715, 0.0019126531210804406, 0.0026293390790658594, 0.002038804534834862, 0.0027617941992744647, 0.0041119031096982, 0.0018562950438913268, 0.0015945598280641578, 0.003046201686912237, 0.002640574367554922, 0.011530284464813394, 0.0024787113149587017, 0.0029632183532159947, 0.002053141604273163, 0.0014994108717638812, 0.0017654709029791837, 0.0020002419458651772, 0.00155983621406319, 0.0, 0.0021286179385490932, 0.0026574671510111517, 0.002031938076720185, 0.006493034167319175, 0.00499631195593466, 0.0032554926808007825, 0.003979617514814762, 0.0025170126035865318, 0.0017730834183535284, 0.0024299080747439104, 0.010286056724816988, 0.0039920331193252605, 0.0028824966956959085, 0.0018031024146613647, 0.0026737913479067785, 0.0027149675488574478, 0.011376721019487362, 0.0028435487915631896, 0.003209415790081367, 0.0019981454540675993, 0.009230097826574092, 0.0024548344725744387, 0.003994051941210087, 0.003948238094149349, 0.0016716608477140788, 0.0014899792546085664, 0.0023249224946804018, 0.00410745513961709, 0.0026350396693744097, 0.008510887884246606, 0.0020248197955596062, 0.0021364897062976745, 0.0022311998011927156, 0.0024147856818938413, 0.002379585076717854, 0.0047515701622408615, 0.00208626756447756, 0.004215966874535314, 0.0024158633185076646, 0.0027295594164158145, 0.0030731469301724184, 0.0020171161033036553, 0.004090789390605681, 0.003005066043245614, 0.002430713425029995, 0.0026568353628035795, 0.0029074287912296405, 0.003504020835087437, 0.0035702577650784874, 0.003987666293723648, 0.002616027105658719, 0.008363169808604076, 0.002193039071389865, 0.0024890330156933566, 0.009725825300955757, 0.0025897809114122797, 0.002332478782097611, 0.002698072065306421, 0.003857047113341564, 0.0024181122323124293, 0.0030290094953856136, 0.009342733366308332, 0.002201298266240293, 0.002048725041248995, 0.004631457057589959, 0.009385150997279351]),
    '1600V_20200614': np.array([0.001239354540820455, 0.0017397184557254885, 0.0023337411970948702, 0.0028594956120849124, 0.0012895948272446604, 0.006351326821534053, 0.0011833222539406843, 0.0012761330000108694, 0.0014119566105477448, 0.0012999437993988484, 0.0010022432883786052, 0.0012396279192575584, 0.00372172184078193, 0.0010544004719588522, 0.0012102414830093388, 0.007181930266122879, 0.0013424852614068266, 0.001435450150608101, 0.001715226692794068, 0.0023600204640369603, 0.001099119808234524, 0.0029437069515715238, 0.0014986014148721805, 0.0012954254702402894, 0.001790214717921091, 0.0011421354013289066, 0.001292092657899603, 0.0008681968145785883, 0.005190128379665493, 0.0029825776805162013, 0.0014934232188077473, 0.0014256338711574838, 0.0017847683039447025, 0.004246955707915927, 0.0017407020957113052, 0.0017442692069675897, 0.0010691285385749589, 0.0010509490422961672, 0.0010730198485308024, 0.0011587450550627518, 0.0025629060491934325, 0.0014406085310406367, 0.0021953447081417762, 0.002525525948476215, 0.0021125792997597577, 0.0018931579656875372, 0.0021547370113011157, 0.0012485790330926585, 0.0018815835152094095, 0.0017927724372524888, 0.001156255381579775, 0.0013954380916347733, 0.0014916473623707268, 0.0009501724810906326, 0.0014428812992105433, 0.0028908314264421923, 0.001442962181189386, 0.0016071218496363142, 0.0016984175783055526, 0.0010112536246740852, 0.0028760245369771425, 0.004462716992987055, 0.0014042756857136583, 0.0019499020505427383, 0.0011742056235610838, 0.0020980396029556627, 0.00176616718787525, 0.001112720283158474, 0.0012594027149935446, 0.002321480638860444, 0.0009320187284835941, 0.0010997502207981207, 0.0017834131929296121, 0.0012634649353482489, 0.002147161716834225, 0.0018561552459364324, 0.0031896112856949032, 0.0034555052312634648, 0.0019106806029327277, 0.001657859876914693, 0.002530559859442032, 0.0016373599680188122, 0.0010849151995200935, 0.0027019321733125976, 0.0010894625253033813, 0.0014857196909550468, 0.0013654557490581736, 0.0014407299763986355, 0.001574685452664738, 0.0010409358670841688, 0.0016811260486946091, 0.0015842115364832782, 0.0014738018359881748, 0.001579684655776039, 0.0016864227863483915, 0.0013240998138381295, 0.001709379760138391, 0.0027445488901053573, 0.0018306903444012792, 0.0014309460702920907, 0.0015685595174167021, 0.001258165541747813, 0.002218796695066617, 0.0011258712687794662, 0.0014963637425072307, 0.0010199115919327635, 0.001971531015170858, 0.001456026080473714, 0.0012552834265425676, 0.001138398223924738, 0.001854839089931887, 0.0010650562636075943, 0.002485024458924165, 0.0021231518468683695, 0.0014862806015939257, 0.0010465273844626486, 0.001660661462099063, 0.0016409227289926253, 0.001580146899643837, 0.003320880789283337, 0.0013953432386322507, 0.0019763364036646637, 0.004371284353290738, 0.002746897341729723, 0.001284646120118233, 0.0043990497470811196, 0.0011229213112713054, 0.0038695533274877526, 0.0014693122190567694, 0.0013118438959853164, 0.0014903640864632889, 0.0016640954903039322, 0.0024774591065140975, 0.0015511451596924538, 0.0011318949680329904, 0.0023141551409467873, 0.0029500779684354748, 0.0015515102274350995, 0.002203949062088458, 0.0017993054785256746, 0.0011751972415490392, 0.001255928068008058, 0.0011177592631997197, 0.0017135699960470878, 0.0011818226752777507, 0.0026906658756173946, 0.0018334326222043291, 0.0015955808307629798, 0.0016881113306051736, 0.004054355299560853, 0.0018109790707393383, 0.001763448915717071, 0.003954146829581249, 0.0014383101753585696, 0.001779464896415589, 0.001927496514820073, 0.0020785849086265035, 0.002254711071037253, 0.0015011400622126853, 0.0014678777554736142, 0.0011393651982281228, 0.001575766481714573, 0.00106208483110987, 0.004071049210271777, 0.0011875722729437126, 0.0015329276500062239, 0.0017224518478789082, 0.0015749581441936887, 0.0012908508559359745, 0.0011809138728184725, 0.006568005566502225, 0.005376784439283682, 0.001943434230392855, 0.001112152442327452, 0.0018603608291786616, 0.0012400807376438969, 0.0014278800044544538, 0.0017995105085244466, 0.0057080162516273125, 0.0017124000216028378, 0.005363602635578621, 0.0017236400826922385, 0.0016892276715007623, 0.000703301068722735, 0.0015308697721149777, 0.0, 0.002759400209533371, 0.0011635034455877733, 0.0010844473643956225, 0.001332562079094796, 0.0044067814723745685, 0.0047749552543189695, 0.0013743653403893957, 0.0012631054716686566, 0.0017435445639339606, 0.001194566851640788, 0.0014783981734718685, 0.0011026886286763565, 0.003000406018237192, 0.0013691977940100688, 0.0017366237876455153, 0.0017626422179545965, 0.0021012158455890064, 0.0017158823211902345, 0.0029174586079658395, 0.0015746504087816994, 0.0012864889378652767, 0.002453945533658887, 0.0014012911193902224, 0.002174432829975095, 0.0015152821234212454, 0.002029314036170733, 0.0010447755700528098, 0.0017972782019948675, 0.0015619803057008847, 0.0017842397100636604, 0.0028574862649268025, 0.0016017040421993677, 0.0023626752082914607, 0.0015109931901524502, 0.0018845326997980925, 0.002581975596631953, 0.0012209116329179443, 0.0016394302886441668, 0.0016269394332353176, 0.0013125484381299375, 0.0018241170766885389, 0.0010856089147407105, 0.0010918137155730412, 0.001214914320282963, 0.0021730168966866545, 0.0023487993358961763, 0.0011386015442826512, 0.0018244576521472102, 0.0021360418248214845, 0.0016632875426622294, 0.0010815359736614095, 0.0012727364217100929, 0.0014663810935306608, 0.0035289436055640465, 0.0015392353235656948, 0.0011841099387468262, 0.001301675127913039, 0.001340223197007035, 0.0016883147400785072, 0.0011133665489487178, 0.0019281630300663662, 0.003106985361067406, 0.0013470491633269998, 0.0017737961414011196, 0.0015285747760648938, 0.001564833005700938, 0.001376870514334193, 0.003601285305329272, 0.0014498929165399394, 0.0013738255941824113, 0.002455979880377718, 0.0010554881616861047, 0.0012194807356496532, 0.0014093310767889459, 0.001709928658246646, 0.0011240581039222003, 0.002360339546286057, 0.0016388383830734333, 0.001278680657172045, 0.001617820289600705, 0.004799190141904779, 0.0014172813442930648, 0.0030532483775526347, 0.0022018954843767307, 0.004223585970355811, 0.0009854639090458132, 0.0017369570201697594, 0.0024349169407696606, 0.0017522472196398037, 0.0020968147759610396, 0.0021452769622330556, 0.0016038807827231782, 0.0017183043540355003, 0.0016065407260633401, 0.0015725053667832289, 0.0017589090135086641, 0.002359005344081325, 0.001345751543806763, 0.0007895536069668588, 0.003950261687012942, 0.001777779607355439, 0.0022456384632341853, 0.0016404502114742176, 0.0014156258571909172, 0.0015456873422045505, 0.001791249658350707, 0.0014759347317886793, 0.0017592358150178805, 0.0019877376043309102, 0.001167195508680865, 0.003364802587837583, 0.003026341649337585, 0.004184515725419672, 0.0012594583877806413, 0.0016603527925871098, 0.0016254381962066751, 0.0052833808274512645, 0.0018217260921064781, 0.001027954375120093, 0.0019424670127589443, 0.005242297925685775, 0.0013964646131691399, 0.005774693957395408, 0.001595403670089354, 0.003277002347925819, 0.0017612267043566909, 0.0009889331063594354, 0.0014848183914951934, 0.007068471229987462, 0.0015286732319321932, 0.0013343218725831744, 0.0014735751475009437, 0.001578763622952162, 0.0016993364981560668, 0.0025381056882407423, 0.002237309084453527, 0.0017457546580014838, 0.003363620728210915, 0.0013736970324810974, 0.00812080966638843, 0.0010031225297076158, 0.007273777277920778, 0.0019169872413187447, 0.003966076312733891, 0.0021349889275680666, 0.001034992384671622, 0.001441064092564878, 0.0012482640524423597, 0.003546869980074541, 0.0012495655884269463, 0.006060688279988607, 0.0015756586798699914, 0.007276370544076687, 0.0019562191087476332, 0.0019321807154244388, 0.0016376484561623815, 0.0011846079113230035, 0.0015604445527682344, 0.0009624314232660683, 0.0013234143805418584, 0.0013159841357081978, 0.0019422003005910475, 0.002023531238249539, 0.0010680422294587287, 0.0011566933652743879, 0.0026490396230998085, 0.0017048974684465, 0.0018130423700065037, 0.0, 0.001679371372794242, 0.001138139788380305, 0.004819947292732349, 0.0010190123628978096, 0.0019422905501347795, 0.001347812983170646, 0.0011303315598077549, 0.0009831538718373285, 0.001598530261900193, 0.0026432154701243075, 0.0015621308466087056, 0.002197150930741777, 0.0015761962259535052, 0.0024801231352940254, 0.0016573534081273302, 0.001525225086904769, 0.006893964599510044, 0.0010461914536647261, 0.0010021992624364142, 0.0012086057279914854, 0.0009457630792535332, 0.005946656004011982, 0.0012671039253720907, 0.0024294113574653917, 0.0018700589395118632, 0.0015252648184605716, 0.0054782878946968835, 0.001137579651702027, 0.0019947010011054484, 0.0013287500158804266, 0.002734463970423483, 0.0, 0.002748453568644689, 0.001508808780172441, 0.002479573155561052, 0.0011990550423943052, 0.005156926963905293, 0.0016904674325170456, 0.0011492318796626942, 0.003094427193334416, 0.0012507095121777267, 0.0013744264214628958, 0.0015448044624673263, 0.0010147553624963938, 0.002764422482652398, 0.001036541343233817, 0.004212117678146097, 0.0009811972638076251, 0.0013253230284341454, 0.0036194658791660534, 0.0022489225856014397, 0.0010280375397491318, 0.0017302468517593427, 0.0020083141726447407, 0.0009586896759229953, 0.0011181079338212622, 0.0015488845279353234, 0.00121713734323611, 0.0016505885077326536, 0.0024671767350362447, 0.0011081405625717857, 0.0009332782376474866, 0.0018114902561934165, 0.001525109267860306, 0.007042484030809334, 0.001428773860716029, 0.0017221458545617582, 0.0011822374563630895, 0.0008714684583455757, 0.0010280194452999168, 0.001174018830332361, 0.0009153934002120843, 0.0, 0.0012580819743998144, 0.0015533586794826948, 0.00121451058406528, 0.003886729355588732, 0.003000974511516621, 0.0019646388748106713, 0.002415600396184562, 0.0014725736109169026, 0.0010328574970559145, 0.0014024906958009162, 0.006156992624010227, 0.0023540693649257804, 0.0017295905805726436, 0.0010602864205069378, 0.0015723693985628911, 0.0016042190350140748, 0.0070204599430125575, 0.001694629309348261, 0.0019045840650107882, 0.0011845874630904954, 0.005586318663655415, 0.001463831716407311, 0.0023950271003248974, 0.0023273500861544748, 0.0009953726406598032, 0.0008763467660858313, 0.001374918647346116, 0.0024268742546825445, 0.0015509146672747246, 0.005163907560092538, 0.0012084886489513414, 0.0012667804023045242, 0.001332724325892694, 0.001444791794102396, 0.0014134353172656044, 0.0029277678653917057, 0.0012318505403023522, 0.0025400506312091126, 0.0014418704556673878, 0.001576799564106098, 0.0018370340615534576, 0.0011801376688332708, 0.0024169033554530287, 0.0017434964688693914, 0.0014418110306445803, 0.0015896707407767568, 0.001722089185673184, 0.002089913513469916, 0.0021292774607771154, 0.002406861038103731, 0.0015264567729016566, 0.005111634091211782, 0.0013145617958035336, 0.0015007959444449303, 0.00609341317884589, 0.0015420753103009021, 0.001382425030647105, 0.001597515531209546, 0.0023132620804278067, 0.0014277775195234651, 0.001794622182300158, 0.00564729636546117, 0.0013010414917588443, 0.001211186824736858, 0.0028045358596449064, 0.005701706307024931]),
    # Vacuum data:
    # PMT 362 on at 1.4kV, all PMTs at 1.5kV
    '1500V_362_1400V_20200614': np.array([0.002085906119267193, 0.0029496084017914597, 0.003952448116271957, 0.004824403113401857, 0.00217780505384671, 0.010232598126577432, 0.0019554455127201095, 0.0021539430268551502, 0.002417991555717992, 0.0022305023927096553, 0.0017361841649091428, 0.0021111605641224653, 0.006304445776362535, 0.0017994422440659598, 0.002041309197617747, 0.011836973311412393, 0.0023108150818581326, 0.0024323091074616096, 0.0029485299571002906, 0.004008726893016484, 0.0018674388598860186, 0.004943297845755464, 0.0025372432404134415, 0.0021918245164027133, 0.003010778307598621, 0.0019297792433476272, 0.0021834161123340246, 0.0014942692290827032, 0.008911392486830982, 0.005009112627211388, 0.0025437706353522356, 0.002434524955867615, 0.0030056850937882104, 0.007081327526960166, 0.0029693538612311295, 0.0029417693499565406, 0.0018115528604149328, 0.0017784691651530115, 0.001832194277379499, 0.0019639576155885685, 0.004370307125157561, 0.002464799014812319, 0.003734420702276067, 0.004218233948117498, 0.003673461998658046, 0.0032007967100049404, 0.003687956135586555, 0.0021489186176972797, 0.0032373322476253095, 0.0030342046665441788, 0.0019524761958859395, 0.0023635643586272627, 0.002526540702566303, 0.0016175112330621434, 0.002440857185545587, 0.004983091627196609, 0.0024566971652106094, 0.0027790735077032644, 0.002839823092932199, 0.0017484774887587104, 0.004710058536327132, 0.007454654493597317, 0.0024156354109320264, 0.003327235214779573, 0.0020074136689291235, 0.0035557963735973756, 0.0029861574075790514, 0.0019098670338939135, 0.002142158020260367, 0.0038627524205708673, 0.0016114971327446305, 0.0018996608288017814, 0.0029695012647492744, 0.002155902617992124, 0.0035914413192844833, 0.0031206881956971134, 0.005429766978405164, 0.005875158327524598, 0.00326146764151243, 0.0028299635290610463, 0.0042191987994413096, 0.002807655410565436, 0.001834096994255787, 0.004538861298449682, 0.0018506432930421165, 0.0025173150358710826, 0.002360043348841007, 0.0025982038886333573, 0.0027042549374500166, 0.0017781642811156525, 0.0027875819041915657, 0.0027441181544758525, 0.0024949850300616566, 0.0026589618961289935, 0.002878020976255783, 0.002282330970787667, 0.002913634836048282, 0.004671581564672208, 0.003065113181772031, 0.002481217860874142, 0.002663121618737516, 0.0021026204687735373, 0.003771913161451095, 0.001930716949441709, 0.0026115738371577154, 0.0017934462025910388, 0.0033376183445005094, 0.002446787399968141, 0.0021074772887702893, 0.0020373408915177384, 0.0031582269824773092, 0.0018228767570471025, 0.004143280136121194, 0.0035464008610378585, 0.0025084051753655927, 0.0017793758140849172, 0.0028261173560890592, 0.002804218866494978, 0.002678011000967532, 0.005618013504706242, 0.0023311588777551306, 0.003348260743915169, 0.0070896650877023465, 0.004480754042874426, 0.0022573690866521893, 0.007324667923613205, 0.0019070400891162025, 0.006334615885154824, 0.0024960810346332826, 0.0022852524007158687, 0.002563574178600082, 0.0028194093836847573, 0.004183536658250434, 0.0025801036906804957, 0.0019104633980705857, 0.003925826367673606, 0.004981595342833323, 0.002631697646897424, 0.0037463025360539334, 0.0031345031155350315, 0.0019524227544583236, 0.00214411885077283, 0.0019496486716782245, 0.002840787104296206, 0.0020691457319821555, 0.004505932055747864, 0.00309295714909413, 0.0027168849063559943, 0.002895611938824199, 0.007055893357983793, 0.00311311931509635, 0.003000607580200689, 0.0065605225088498096, 0.0024061233867442152, 0.003038507032306309, 0.003196698633840876, 0.003481203243805577, 0.003868392213032792, 0.0025614955846030713, 0.002484745205108528, 0.001979008536563463, 0.002744645666938453, 0.0018310190660099587, 0.006758939381307689, 0.002061008384525737, 0.0026090603112981494, 0.002894261749644281, 0.0026547459458047084, 0.002188238855600996, 0.001988109782742354, 0.010800064462871193, 0.00897070917490104, 0.0032582761626448213, 0.0019237703137838203, 0.0031583013390912547, 0.0021539508989034894, 0.0024163548296819836, 0.003056860605443962, 0.00989706541191255, 0.0028859342141818833, 0.009070473757145052, 0.002949605255785661, 0.0029189739053318807, 0.0013670707489532319, 0.002585716417529998, 0.0, 0.004635737832691273, 0.0019517395134533845, 0.0018719853273635057, 0.0023472280166440293, 0.007428210200397244, 0.007542826021778212, 0.0023831787418589867, 0.0021680588709320917, 0.003037176323132788, 0.0020065440137004946, 0.0025235844495933844, 0.0019074482547054544, 0.0051307810007371535, 0.0023139663081965718, 0.0029486233040324286, 0.002966903859064441, 0.003537497224751838, 0.002984604523114847, 0.005024117488252195, 0.0026722576414425856, 0.00213704113759646, 0.0042319051920824795, 0.002409611973982524, 0.003631266138370235, 0.0025726795731613257, 0.0034386384843055734, 0.001775104527175836, 0.003034320419650108, 0.0026352171729016934, 0.003005025100436563, 0.004827441672384273, 0.002727294922988591, 0.003957262487870031, 0.002545943145756636, 0.003184029128069157, 0.004353319254562286, 0.0020916661003556533, 0.002752916209370159, 0.00276692406229051, 0.002195890747954383, 0.003054805243861106, 0.0018763398895043548, 0.0018276167312407678, 0.0020736848440358765, 0.003644368363286227, 0.003973323132651654, 0.0019646831520363163, 0.0031094485340531952, 0.003641907773189352, 0.0027941043611759138, 0.0018450911895410316, 0.0021457435818873646, 0.002490763675629057, 0.006008785527312767, 0.0025973364768485875, 0.0020126253882395205, 0.0022197411905526426, 0.002278284796021994, 0.0028384379120278912, 0.0019159342681869763, 0.00325315603852368, 0.005240725685508535, 0.0022909384687946504, 0.0030276123823173517, 0.0025624993911514482, 0.0026460858170606104, 0.00232158859917268, 0.005881147916605516, 0.0024370603439857782, 0.0022798471808779396, 0.004100058526190502, 0.001784016264922319, 0.0020463514737801575, 0.002361407031540062, 0.0028341798627856143, 0.0019007905559032053, 0.0038862067400276634, 0.0027584526420330667, 0.002157416178259612, 0.0027343211664965926, 0.007927104732952599, 0.0023859241115383096, 0.004882428061990786, 0.003587168701665862, 0.0069076964931178144, 0.0016727947527414011, 0.002973650992572772, 0.004074127671125862, 0.002937433232254822, 0.0034880823351646118, 0.0036209392282888675, 0.0027974676009824913, 0.0028363866166256505, 0.002621323301714016, 0.0026083680566854573, 0.0029279474460432078, 0.003975481763215634, 0.0023005834014553253, 0.0013414055227751306, 0.006613273251509428, 0.0029915382881996752, 0.0038135016324938537, 0.002724865853362915, 0.0024117638294880914, 0.0026309361452079437, 0.002988109479594258, 0.0025090346988631313, 0.0028865934521716194, 0.0032695535947252466, 0.001964610574269569, 0.005612488943465638, 0.0050476135668763465, 0.007003659926059319, 0.0020622270332848057, 0.0027512083010835975, 0.002759542860349935, 0.008779958658040266, 0.0031112232025844685, 0.0017575056450166813, 0.0032883676895303273, 0.00880329622198586, 0.002406795739949619, 0.009546215411183083, 0.002694708314600647, 0.0055146749681920584, 0.003026368641665498, 0.0016825496673117113, 0.002502773335479273, 0.011557919195091327, 0.0026348227960020713, 0.002266982991021235, 0.0024454534268902512, 0.0026796903298646015, 0.0028954288715047458, 0.00438351781343348, 0.0038475138627264496, 0.0030254543288970905, 0.0056666451907664535, 0.0023324450361296555, 0.013122841055481902, 0.0016846045869633789, 0.011797291474415196, 0.0032887160294227027, 0.006562357123945686, 0.003607526002050326, 0.0017592804278752515, 0.002461414483853556, 0.0020930372839849548, 0.005951501550991758, 0.00221726368761244, 0.009932590999395136, 0.002714287430078169, 0.01165326436119117, 0.0033646656623551297, 0.0033057697752621813, 0.0028287822468848886, 0.0020337245536485708, 0.0026204774418035663, 0.0016347534558883118, 0.0022428980795722794, 0.0022396276825914635, 0.0032461796771344597, 0.003425931289727082, 0.0018142907354703546, 0.0019832321621294975, 0.004494396125040512, 0.002901514656242194, 0.0030447737165028906, 0.0, 0.002881709044647395, 0.001963243979033229, 0.008101006586073382, 0.0017632716903311636, 0.003243073032059233, 0.0023058090050963364, 0.0019132673143853792, 0.003173990821489383, 0.0027130756221028186, 0.004266916484328926, 0.002607173329335936, 0.003687535113400176, 0.0026566239173599062, 0.004238490000183762, 0.0027897575772819825, 0.002598125401670171, 0.011236425597874517, 0.0018149993135542295, 0.00172805651621121, 0.0020997002328370456, 0.0016250378485821145, 0.010124118079685987, 0.002160704180822827, 0.004088585594018028, 0.003160998576410285, 0.0025847054922235936, 0.00896694964694206, 0.0019336376671625318, 0.0033168748684051378, 0.0022408668957437467, 0.0045720221455967, 0.0, 0.004587154021939543, 0.0025252485631659036, 0.004237265087392108, 0.002072993002032722, 0.008939766599122385, 0.002948855971591338, 0.0020046142410292027, 0.0052340137343973355, 0.0021207043817929754, 0.002341164017620041, 0.0026063450780860224, 0.0017007543258655388, 0.004633778014240759, 0.001761680300026289, 0.006880180387878943, 0.0016694675680854446, 0.0022754591989043887, 0.0061085225628778415, 0.0038449885261370195, 0.0017678810984899994, 0.0029730108914774586, 0.003435465906023674, 0.001659978591453715, 0.0019126531210804406, 0.0026293390790658594, 0.002038804534834862, 0.0027617941992744647, 0.0041119031096982, 0.0018562950438913268, 0.0015945598280641578, 0.003046201686912237, 0.002640574367554922, 0.011530284464813394, 0.0024787113149587017, 0.0029632183532159947, 0.002053141604273163, 0.0014994108717638812, 0.0017654709029791837, 0.0020002419458651772, 0.00155983621406319, 0.0, 0.0021286179385490932, 0.0026574671510111517, 0.002031938076720185, 0.006493034167319175, 0.00499631195593466, 0.0032554926808007825, 0.003979617514814762, 0.0025170126035865318, 0.0017730834183535284, 0.0024299080747439104, 0.010286056724816988, 0.0039920331193252605, 0.0028824966956959085, 0.0018031024146613647, 0.0026737913479067785, 0.0027149675488574478, 0.011376721019487362, 0.0028435487915631896, 0.003209415790081367, 0.0019981454540675993, 0.009230097826574092, 0.0024548344725744387, 0.003994051941210087, 0.003948238094149349, 0.0016716608477140788, 0.0014899792546085664, 0.0023249224946804018, 0.00410745513961709, 0.0026350396693744097, 0.008510887884246606, 0.0020248197955596062, 0.0021364897062976745, 0.0022311998011927156, 0.0024147856818938413, 0.002379585076717854, 0.0047515701622408615, 0.00208626756447756, 0.004215966874535314, 0.0024158633185076646, 0.0027295594164158145, 0.0030731469301724184, 0.0020171161033036553, 0.004090789390605681, 0.003005066043245614, 0.002430713425029995, 0.0026568353628035795, 0.0029074287912296405, 0.003504020835087437, 0.0035702577650784874, 0.003987666293723648, 0.002616027105658719, 0.008363169808604076, 0.002193039071389865, 0.0024890330156933566, 0.009725825300955757, 0.0025897809114122797, 0.002332478782097611, 0.002698072065306421, 0.003857047113341564, 0.0024181122323124293, 0.0030290094953856136, 0.009342733366308332, 0.002201298266240293, 0.002048725041248995, 0.004631457057589959, 0.009385150997279351]),
    # PMT 362 on at 1.3kV, all PMTs at 1.5kV
    '1500V_362_1300V_20200614': np.array([0.002085906119267193, 0.0029496084017914597, 0.003952448116271957, 0.004824403113401857, 0.00217780505384671, 0.010232598126577432, 0.0019554455127201095, 0.0021539430268551502, 0.002417991555717992, 0.0022305023927096553, 0.0017361841649091428, 0.0021111605641224653, 0.006304445776362535, 0.0017994422440659598, 0.002041309197617747, 0.011836973311412393, 0.0023108150818581326, 0.0024323091074616096, 0.0029485299571002906, 0.004008726893016484, 0.0018674388598860186, 0.004943297845755464, 0.0025372432404134415, 0.0021918245164027133, 0.003010778307598621, 0.0019297792433476272, 0.0021834161123340246, 0.0014942692290827032, 0.008911392486830982, 0.005009112627211388, 0.0025437706353522356, 0.002434524955867615, 0.0030056850937882104, 0.007081327526960166, 0.0029693538612311295, 0.0029417693499565406, 0.0018115528604149328, 0.0017784691651530115, 0.001832194277379499, 0.0019639576155885685, 0.004370307125157561, 0.002464799014812319, 0.003734420702276067, 0.004218233948117498, 0.003673461998658046, 0.0032007967100049404, 0.003687956135586555, 0.0021489186176972797, 0.0032373322476253095, 0.0030342046665441788, 0.0019524761958859395, 0.0023635643586272627, 0.002526540702566303, 0.0016175112330621434, 0.002440857185545587, 0.004983091627196609, 0.0024566971652106094, 0.0027790735077032644, 0.002839823092932199, 0.0017484774887587104, 0.004710058536327132, 0.007454654493597317, 0.0024156354109320264, 0.003327235214779573, 0.0020074136689291235, 0.0035557963735973756, 0.0029861574075790514, 0.0019098670338939135, 0.002142158020260367, 0.0038627524205708673, 0.0016114971327446305, 0.0018996608288017814, 0.0029695012647492744, 0.002155902617992124, 0.0035914413192844833, 0.0031206881956971134, 0.005429766978405164, 0.005875158327524598, 0.00326146764151243, 0.0028299635290610463, 0.0042191987994413096, 0.002807655410565436, 0.001834096994255787, 0.004538861298449682, 0.0018506432930421165, 0.0025173150358710826, 0.002360043348841007, 0.0025982038886333573, 0.0027042549374500166, 0.0017781642811156525, 0.0027875819041915657, 0.0027441181544758525, 0.0024949850300616566, 0.0026589618961289935, 0.002878020976255783, 0.002282330970787667, 0.002913634836048282, 0.004671581564672208, 0.003065113181772031, 0.002481217860874142, 0.002663121618737516, 0.0021026204687735373, 0.003771913161451095, 0.001930716949441709, 0.0026115738371577154, 0.0017934462025910388, 0.0033376183445005094, 0.002446787399968141, 0.0021074772887702893, 0.0020373408915177384, 0.0031582269824773092, 0.0018228767570471025, 0.004143280136121194, 0.0035464008610378585, 0.0025084051753655927, 0.0017793758140849172, 0.0028261173560890592, 0.002804218866494978, 0.002678011000967532, 0.005618013504706242, 0.0023311588777551306, 0.003348260743915169, 0.0070896650877023465, 0.004480754042874426, 0.0022573690866521893, 0.007324667923613205, 0.0019070400891162025, 0.006334615885154824, 0.0024960810346332826, 0.0022852524007158687, 0.002563574178600082, 0.0028194093836847573, 0.004183536658250434, 0.0025801036906804957, 0.0019104633980705857, 0.003925826367673606, 0.004981595342833323, 0.002631697646897424, 0.0037463025360539334, 0.0031345031155350315, 0.0019524227544583236, 0.00214411885077283, 0.0019496486716782245, 0.002840787104296206, 0.0020691457319821555, 0.004505932055747864, 0.00309295714909413, 0.0027168849063559943, 0.002895611938824199, 0.007055893357983793, 0.00311311931509635, 0.003000607580200689, 0.0065605225088498096, 0.0024061233867442152, 0.003038507032306309, 0.003196698633840876, 0.003481203243805577, 0.003868392213032792, 0.0025614955846030713, 0.002484745205108528, 0.001979008536563463, 0.002744645666938453, 0.0018310190660099587, 0.006758939381307689, 0.002061008384525737, 0.0026090603112981494, 0.002894261749644281, 0.0026547459458047084, 0.002188238855600996, 0.001988109782742354, 0.010800064462871193, 0.00897070917490104, 0.0032582761626448213, 0.0019237703137838203, 0.0031583013390912547, 0.0021539508989034894, 0.0024163548296819836, 0.003056860605443962, 0.00989706541191255, 0.0028859342141818833, 0.009070473757145052, 0.002949605255785661, 0.0029189739053318807, 0.0013670707489532319, 0.002585716417529998, 0.0, 0.004635737832691273, 0.0019517395134533845, 0.0018719853273635057, 0.0023472280166440293, 0.007428210200397244, 0.007542826021778212, 0.0023831787418589867, 0.0021680588709320917, 0.003037176323132788, 0.0020065440137004946, 0.0025235844495933844, 0.0019074482547054544, 0.0051307810007371535, 0.0023139663081965718, 0.0029486233040324286, 0.002966903859064441, 0.003537497224751838, 0.002984604523114847, 0.005024117488252195, 0.0026722576414425856, 0.00213704113759646, 0.0042319051920824795, 0.002409611973982524, 0.003631266138370235, 0.0025726795731613257, 0.0034386384843055734, 0.001775104527175836, 0.003034320419650108, 0.0026352171729016934, 0.003005025100436563, 0.004827441672384273, 0.002727294922988591, 0.003957262487870031, 0.002545943145756636, 0.003184029128069157, 0.004353319254562286, 0.0020916661003556533, 0.002752916209370159, 0.00276692406229051, 0.002195890747954383, 0.003054805243861106, 0.0018763398895043548, 0.0018276167312407678, 0.0020736848440358765, 0.003644368363286227, 0.003973323132651654, 0.0019646831520363163, 0.0031094485340531952, 0.003641907773189352, 0.0027941043611759138, 0.0018450911895410316, 0.0021457435818873646, 0.002490763675629057, 0.006008785527312767, 0.0025973364768485875, 0.0020126253882395205, 0.0022197411905526426, 0.002278284796021994, 0.0028384379120278912, 0.0019159342681869763, 0.00325315603852368, 0.005240725685508535, 0.0022909384687946504, 0.0030276123823173517, 0.0025624993911514482, 0.0026460858170606104, 0.00232158859917268, 0.005881147916605516, 0.0024370603439857782, 0.0022798471808779396, 0.004100058526190502, 0.001784016264922319, 0.0020463514737801575, 0.002361407031540062, 0.0028341798627856143, 0.0019007905559032053, 0.0038862067400276634, 0.0027584526420330667, 0.002157416178259612, 0.0027343211664965926, 0.007927104732952599, 0.0023859241115383096, 0.004882428061990786, 0.003587168701665862, 0.0069076964931178144, 0.0016727947527414011, 0.002973650992572772, 0.004074127671125862, 0.002937433232254822, 0.0034880823351646118, 0.0036209392282888675, 0.0027974676009824913, 0.0028363866166256505, 0.002621323301714016, 0.0026083680566854573, 0.0029279474460432078, 0.003975481763215634, 0.0023005834014553253, 0.0013414055227751306, 0.006613273251509428, 0.0029915382881996752, 0.0038135016324938537, 0.002724865853362915, 0.0024117638294880914, 0.0026309361452079437, 0.002988109479594258, 0.0025090346988631313, 0.0028865934521716194, 0.0032695535947252466, 0.001964610574269569, 0.005612488943465638, 0.0050476135668763465, 0.007003659926059319, 0.0020622270332848057, 0.0027512083010835975, 0.002759542860349935, 0.008779958658040266, 0.0031112232025844685, 0.0017575056450166813, 0.0032883676895303273, 0.00880329622198586, 0.002406795739949619, 0.009546215411183083, 0.002694708314600647, 0.0055146749681920584, 0.003026368641665498, 0.0016825496673117113, 0.002502773335479273, 0.011557919195091327, 0.0026348227960020713, 0.002266982991021235, 0.0024454534268902512, 0.0026796903298646015, 0.0028954288715047458, 0.00438351781343348, 0.0038475138627264496, 0.0030254543288970905, 0.0056666451907664535, 0.0023324450361296555, 0.013122841055481902, 0.0016846045869633789, 0.011797291474415196, 0.0032887160294227027, 0.006562357123945686, 0.003607526002050326, 0.0017592804278752515, 0.002461414483853556, 0.0020930372839849548, 0.005951501550991758, 0.00221726368761244, 0.009932590999395136, 0.002714287430078169, 0.01165326436119117, 0.0033646656623551297, 0.0033057697752621813, 0.0028287822468848886, 0.0020337245536485708, 0.0026204774418035663, 0.0016347534558883118, 0.0022428980795722794, 0.0022396276825914635, 0.0032461796771344597, 0.003425931289727082, 0.0018142907354703546, 0.0019832321621294975, 0.004494396125040512, 0.002901514656242194, 0.0030447737165028906, 0.0, 0.002881709044647395, 0.001963243979033229, 0.008101006586073382, 0.0017632716903311636, 0.003243073032059233, 0.0023058090050963364, 0.0019132673143853792, 0.0060956702190099596, 0.0027130756221028186, 0.004266916484328926, 0.002607173329335936, 0.003687535113400176, 0.0026566239173599062, 0.004238490000183762, 0.0027897575772819825, 0.002598125401670171, 0.011236425597874517, 0.0018149993135542295, 0.00172805651621121, 0.0020997002328370456, 0.0016250378485821145, 0.010124118079685987, 0.002160704180822827, 0.004088585594018028, 0.003160998576410285, 0.0025847054922235936, 0.00896694964694206, 0.0019336376671625318, 0.0033168748684051378, 0.0022408668957437467, 0.0045720221455967, 0.0, 0.004587154021939543, 0.0025252485631659036, 0.004237265087392108, 0.002072993002032722, 0.008939766599122385, 0.002948855971591338, 0.0020046142410292027, 0.0052340137343973355, 0.0021207043817929754, 0.002341164017620041, 0.0026063450780860224, 0.0017007543258655388, 0.004633778014240759, 0.001761680300026289, 0.006880180387878943, 0.0016694675680854446, 0.0022754591989043887, 0.0061085225628778415, 0.0038449885261370195, 0.0017678810984899994, 0.0029730108914774586, 0.003435465906023674, 0.001659978591453715, 0.0019126531210804406, 0.0026293390790658594, 0.002038804534834862, 0.0027617941992744647, 0.0041119031096982, 0.0018562950438913268, 0.0015945598280641578, 0.003046201686912237, 0.002640574367554922, 0.011530284464813394, 0.0024787113149587017, 0.0029632183532159947, 0.002053141604273163, 0.0014994108717638812, 0.0017654709029791837, 0.0020002419458651772, 0.00155983621406319, 0.0, 0.0021286179385490932, 0.0026574671510111517, 0.002031938076720185, 0.006493034167319175, 0.00499631195593466, 0.0032554926808007825, 0.003979617514814762, 0.0025170126035865318, 0.0017730834183535284, 0.0024299080747439104, 0.010286056724816988, 0.0039920331193252605, 0.0028824966956959085, 0.0018031024146613647, 0.0026737913479067785, 0.0027149675488574478, 0.011376721019487362, 0.0028435487915631896, 0.003209415790081367, 0.0019981454540675993, 0.009230097826574092, 0.0024548344725744387, 0.003994051941210087, 0.003948238094149349, 0.0016716608477140788, 0.0014899792546085664, 0.0023249224946804018, 0.00410745513961709, 0.0026350396693744097, 0.008510887884246606, 0.0020248197955596062, 0.0021364897062976745, 0.0022311998011927156, 0.0024147856818938413, 0.002379585076717854, 0.0047515701622408615, 0.00208626756447756, 0.004215966874535314, 0.0024158633185076646, 0.0027295594164158145, 0.0030731469301724184, 0.0020171161033036553, 0.004090789390605681, 0.003005066043245614, 0.002430713425029995, 0.0026568353628035795, 0.0029074287912296405, 0.003504020835087437, 0.0035702577650784874, 0.003987666293723648, 0.002616027105658719, 0.008363169808604076, 0.002193039071389865, 0.0024890330156933566, 0.009725825300955757, 0.0025897809114122797, 0.002332478782097611, 0.002698072065306421, 0.003857047113341564, 0.0024181122323124293, 0.0030290094953856136, 0.009342733366308332, 0.002201298266240293, 0.002048725041248995, 0.004631457057589959, 0.009385150997279351]),
    # Nitrogen data 
    # PMTs at 1.5kV,  1.25bar nitrogen, (PMTs 116, 427 at 1kV, PMT 362 at 1.3kV, PMT 141 at 1.4kV, off: 185, 354, 386)
    '1500V_116_427_100V_362_1300V_141_1400V_20200614': np.array([0.002085906119267193, 0.0029496084017914597, 0.003952448116271957, 0.004824403113401857, 0.00217780505384671, 0.010232598126577432, 0.0019554455127201095, 0.0021539430268551502, 0.002417991555717992, 0.0022305023927096553, 0.0017361841649091428, 0.0021111605641224653, 0.006304445776362535, 0.0017994422440659598, 0.002041309197617747, 0.011836973311412393, 0.0023108150818581326, 0.0024323091074616096, 0.0029485299571002906, 0.004008726893016484, 0.0018674388598860186, 0.004943297845755464, 0.0025372432404134415, 0.0021918245164027133, 0.003010778307598621, 0.0019297792433476272, 0.0021834161123340246, 0.0014942692290827032, 0.008911392486830982, 0.005009112627211388, 0.0025437706353522356, 0.002434524955867615, 0.0030056850937882104, 0.007081327526960166, 0.0029693538612311295, 0.0029417693499565406, 0.0018115528604149328, 0.0017784691651530115, 0.001832194277379499, 0.0019639576155885685, 0.004370307125157561, 0.002464799014812319, 0.003734420702276067, 0.004218233948117498, 0.003673461998658046, 0.0032007967100049404, 0.003687956135586555, 0.0021489186176972797, 0.0032373322476253095, 0.0030342046665441788, 0.0019524761958859395, 0.0023635643586272627, 0.002526540702566303, 0.0016175112330621434, 0.002440857185545587, 0.004983091627196609, 0.0024566971652106094, 0.0027790735077032644, 0.002839823092932199, 0.0017484774887587104, 0.004710058536327132, 0.007454654493597317, 0.0024156354109320264, 0.003327235214779573, 0.0020074136689291235, 0.0035557963735973756, 0.0029861574075790514, 0.0019098670338939135, 0.002142158020260367, 0.0038627524205708673, 0.0016114971327446305, 0.0018996608288017814, 0.0029695012647492744, 0.002155902617992124, 0.0035914413192844833, 0.0031206881956971134, 0.005429766978405164, 0.005875158327524598, 0.00326146764151243, 0.0028299635290610463, 0.0042191987994413096, 0.002807655410565436, 0.001834096994255787, 0.004538861298449682, 0.0018506432930421165, 0.0025173150358710826, 0.002360043348841007, 0.0025982038886333573, 0.0027042549374500166, 0.0017781642811156525, 0.0027875819041915657, 0.0027441181544758525, 0.0024949850300616566, 0.0026589618961289935, 0.002878020976255783, 0.002282330970787667, 0.002913634836048282, 0.004671581564672208, 0.003065113181772031, 0.002481217860874142, 0.002663121618737516, 0.0021026204687735373, 0.003771913161451095, 0.001930716949441709, 0.0026115738371577154, 0.0017934462025910388, 0.0033376183445005094, 0.002446787399968141, 0.0021074772887702893, 0.0020373408915177384, 0.0031582269824773092, 0.0018228767570471025, 0.004143280136121194, 0.0035464008610378585, 0.0025084051753655927, 0.0017793758140849172, 0.06808781688210443, 0.002804218866494978, 0.002678011000967532, 0.005618013504706242, 0.0023311588777551306, 0.003348260743915169, 0.0070896650877023465, 0.004480754042874426, 0.0022573690866521893, 0.007324667923613205, 0.0019070400891162025, 0.006334615885154824, 0.0024960810346332826, 0.0022852524007158687, 0.002563574178600082, 0.0028194093836847573, 0.004183536658250434, 0.0025801036906804957, 0.0019104633980705857, 0.003925826367673606, 0.004981595342833323, 0.002631697646897424, 0.0037463025360539334, 0.0031345031155350315, 0.0019524227544583236, 0.003694007441944061, 0.0019496486716782245, 0.002840787104296206, 0.0020691457319821555, 0.004505932055747864, 0.00309295714909413, 0.0027168849063559943, 0.002895611938824199, 0.007055893357983793, 0.00311311931509635, 0.003000607580200689, 0.0065605225088498096, 0.0024061233867442152, 0.003038507032306309, 0.003196698633840876, 0.003481203243805577, 0.003868392213032792, 0.0025614955846030713, 0.002484745205108528, 0.001979008536563463, 0.002744645666938453, 0.0018310190660099587, 0.006758939381307689, 0.002061008384525737, 0.0026090603112981494, 0.002894261749644281, 0.0026547459458047084, 0.002188238855600996, 0.001988109782742354, 0.010800064462871193, 0.00897070917490104, 0.0032582761626448213, 0.0019237703137838203, 0.0031583013390912547, 0.0021539508989034894, 0.0024163548296819836, 0.003056860605443962, 0.00989706541191255, 0.0028859342141818833, 0.009070473757145052, 0.002949605255785661, 0.0029189739053318807, 0.0013670707489532319, 0.002585716417529998, 0.0, 0.004635737832691273, 0.0019517395134533845, 0.0018719853273635057, 0.0023472280166440293, 0.007428210200397244, 0.007542826021778212, 0.0023831787418589867, 0.0021680588709320917, 0.003037176323132788, 0.0020065440137004946, 0.0025235844495933844, 0.0019074482547054544, 0.0051307810007371535, 0.0023139663081965718, 0.0029486233040324286, 0.002966903859064441, 0.003537497224751838, 0.002984604523114847, 0.005024117488252195, 0.0026722576414425856, 0.00213704113759646, 0.0042319051920824795, 0.002409611973982524, 0.003631266138370235, 0.0025726795731613257, 0.0034386384843055734, 0.001775104527175836, 0.003034320419650108, 0.0026352171729016934, 0.003005025100436563, 0.004827441672384273, 0.002727294922988591, 0.003957262487870031, 0.002545943145756636, 0.003184029128069157, 0.004353319254562286, 0.0020916661003556533, 0.002752916209370159, 0.00276692406229051, 0.002195890747954383, 0.003054805243861106, 0.0018763398895043548, 0.0018276167312407678, 0.0020736848440358765, 0.003644368363286227, 0.003973323132651654, 0.0019646831520363163, 0.0031094485340531952, 0.003641907773189352, 0.0027941043611759138, 0.0018450911895410316, 0.0021457435818873646, 0.002490763675629057, 0.006008785527312767, 0.0025973364768485875, 0.0020126253882395205, 0.0022197411905526426, 0.002278284796021994, 0.0028384379120278912, 0.0019159342681869763, 0.00325315603852368, 0.005240725685508535, 0.0022909384687946504, 0.0030276123823173517, 0.0025624993911514482, 0.0026460858170606104, 0.00232158859917268, 0.005881147916605516, 0.0024370603439857782, 0.0022798471808779396, 0.004100058526190502, 0.001784016264922319, 0.0020463514737801575, 0.002361407031540062, 0.0028341798627856143, 0.0019007905559032053, 0.0038862067400276634, 0.0027584526420330667, 0.002157416178259612, 0.0027343211664965926, 0.007927104732952599, 0.0023859241115383096, 0.004882428061990786, 0.003587168701665862, 0.0069076964931178144, 0.0016727947527414011, 0.002973650992572772, 0.004074127671125862, 0.002937433232254822, 0.0034880823351646118, 0.0036209392282888675, 0.0027974676009824913, 0.0028363866166256505, 0.002621323301714016, 0.0026083680566854573, 0.0029279474460432078, 0.003975481763215634, 0.0023005834014553253, 0.0013414055227751306, 0.006613273251509428, 0.0029915382881996752, 0.0038135016324938537, 0.002724865853362915, 0.0024117638294880914, 0.0026309361452079437, 0.002988109479594258, 0.0025090346988631313, 0.0028865934521716194, 0.0032695535947252466, 0.001964610574269569, 0.005612488943465638, 0.0050476135668763465, 0.007003659926059319, 0.0020622270332848057, 0.0027512083010835975, 0.002759542860349935, 0.008779958658040266, 0.0031112232025844685, 0.0017575056450166813, 0.0032883676895303273, 0.00880329622198586, 0.002406795739949619, 0.009546215411183083, 0.002694708314600647, 0.0055146749681920584, 0.003026368641665498, 0.0016825496673117113, 0.002502773335479273, 0.011557919195091327, 0.0026348227960020713, 0.002266982991021235, 0.0024454534268902512, 0.0026796903298646015, 0.0028954288715047458, 0.00438351781343348, 0.0038475138627264496, 0.0030254543288970905, 0.0056666451907664535, 0.0023324450361296555, 0.013122841055481902, 0.0016846045869633789, 0.011797291474415196, 0.0032887160294227027, 0.006562357123945686, 0.003607526002050326, 0.0017592804278752515, 0.002461414483853556, 0.0020930372839849548, 0.005951501550991758, 0.00221726368761244, 0.009932590999395136, 0.002714287430078169, 0.01165326436119117, 0.0033646656623551297, 0.0033057697752621813, 0.0028287822468848886, 0.0020337245536485708, 0.0026204774418035663, 0.0016347534558883118, 0.0022428980795722794, 0.0022396276825914635, 0.0032461796771344597, 0.003425931289727082, 0.0018142907354703546, 0.0019832321621294975, 0.004494396125040512, 0.002901514656242194, 0.0030447737165028906, 0.0, 0.002881709044647395, 0.001963243979033229, 0.008101006586073382, 0.0017632716903311636, 0.003243073032059233, 0.0023058090050963364, 0.0019132673143853792, 0.0060956702190099596, 0.0027130756221028186, 0.004266916484328926, 0.002607173329335936, 0.003687535113400176, 0.0026566239173599062, 0.004238490000183762, 0.0027897575772819825, 0.002598125401670171, 0.011236425597874517, 0.0018149993135542295, 0.00172805651621121, 0.0020997002328370456, 0.0016250378485821145, 0.010124118079685987, 0.002160704180822827, 0.004088585594018028, 0.003160998576410285, 0.0025847054922235936, 0.00896694964694206, 0.0019336376671625318, 0.0033168748684051378, 0.0022408668957437467, 0.0045720221455967, 0.0, 0.004587154021939543, 0.0025252485631659036, 0.004237265087392108, 0.002072993002032722, 0.008939766599122385, 0.002948855971591338, 0.0020046142410292027, 0.0052340137343973355, 0.0021207043817929754, 0.002341164017620041, 0.0026063450780860224, 0.0017007543258655388, 0.004633778014240759, 0.001761680300026289, 0.006880180387878943, 0.0016694675680854446, 0.0022754591989043887, 0.0061085225628778415, 0.0038449885261370195, 0.0017678810984899994, 0.0029730108914774586, 0.003435465906023674, 0.001659978591453715, 0.0019126531210804406, 0.0026293390790658594, 0.002038804534834862, 0.0027617941992744647, 0.0041119031096982, 0.0018562950438913268, 0.0015945598280641578, 0.003046201686912237, 0.002640574367554922, 0.011530284464813394, 0.0024787113149587017, 0.0029632183532159947, 0.002053141604273163, 0.0014994108717638812, 0.0017654709029791837, 0.0020002419458651772, 0.00155983621406319, 0.07379546835190369, 0.0021286179385490932, 0.0026574671510111517, 0.002031938076720185, 0.006493034167319175, 0.00499631195593466, 0.0032554926808007825, 0.003979617514814762, 0.0025170126035865318, 0.0017730834183535284, 0.0024299080747439104, 0.010286056724816988, 0.0039920331193252605, 0.0028824966956959085, 0.0018031024146613647, 0.0026737913479067785, 0.0027149675488574478, 0.011376721019487362, 0.0028435487915631896, 0.003209415790081367, 0.0019981454540675993, 0.009230097826574092, 0.0024548344725744387, 0.003994051941210087, 0.003948238094149349, 0.0016716608477140788, 0.0014899792546085664, 0.0023249224946804018, 0.00410745513961709, 0.0026350396693744097, 0.008510887884246606, 0.0020248197955596062, 0.0021364897062976745, 0.0022311998011927156, 0.0024147856818938413, 0.002379585076717854, 0.0047515701622408615, 0.00208626756447756, 0.004215966874535314, 0.0024158633185076646, 0.0027295594164158145, 0.0030731469301724184, 0.0020171161033036553, 0.004090789390605681, 0.003005066043245614, 0.002430713425029995, 0.0026568353628035795, 0.0029074287912296405, 0.003504020835087437, 0.0035702577650784874, 0.003987666293723648, 0.002616027105658719, 0.008363169808604076, 0.002193039071389865, 0.0024890330156933566, 0.009725825300955757, 0.0025897809114122797, 0.002332478782097611, 0.002698072065306421, 0.003857047113341564, 0.0024181122323124293, 0.0030290094953856136, 0.009342733366308332, 0.002201298266240293, 0.002048725041248995, 0.004631457057589959, 0.009385150997279351]),
    # PMTs at 4e6 gain, cutoff 1550V, 1.25bar nitrogen (PMTs off: 185, 354, 386, 427, PMT 362 at 1300V, 
    # PMTs : 58, 60, 74, 75, 90, 92, 93, 107, 108, 122, 123, 125, 126, 140, 141, 143 at 1500V)
    '4e6_gain_362_1300V_16PMTs_1500V': np.array([0.0034779779272330787, 0.0034924697512544564, 0.003294083621684023, 0.0036781141342799796, 0.0031303549224138707, 0.00808774057352306, 0.0031217866503397765, 0.003447752937893365, 0.003182001875149702, 0.00346925020135177, 0.0034234573989969604, 0.0032224209627734917, 0.00478033709268571, 0.003310846910624227, 0.0032373288790770558, 0.009162493576613999, 0.0034375554169642697, 0.0033425711311855324, 0.0032283866273525343, 0.0034818185206547834, 0.0034079099296911485, 0.003794017713098041, 0.003341594864670753, 0.00320191116625019, 0.003283293002653994, 0.0033254367415584338, 0.003101002549329652, 0.003203558296714975, 0.0067614624205102215, 0.003845014924295724, 0.0033037686477596227, 0.003385714134106216, 0.0034115667800124656, 0.005455579464868477, 0.0032720608637545963, 0.0033448881521853198, 0.003379092592942211, 0.003392270277759635, 0.003402952833023225, 0.0032415486649537335, 0.003375933465478925, 0.0031826034596059687, 0.003313801251363, 0.003284295169242346, 0.0032942546905785014, 0.0033459402079906646, 0.0031971571754188847, 0.0032899381077443335, 0.003274169984095398, 0.0031848067702128333, 0.0033772446927605223, 0.0033352321213733207, 0.0033561995319625042, 0.0033159759312760624, 0.0033475316861760122, 0.0037198970654135754, 0.0031755008659873985, 0.003105080077362275, 0.002851664582782315, 0.003388602223020197, 0.004736062922760161, 0.005728189303146344, 0.0033811565739181055, 0.003199096081577878, 0.003272977557868792, 0.0032646171295932468, 0.0032860010811618474, 0.0033695487125019784, 0.003366363226601939, 0.003273328667923594, 0.0033209312008847104, 0.003262653421605488, 0.003311437727893917, 0.00325493038109889, 0.0035900403856509327, 0.00310458099110049, 0.0041395530490333276, 0.004462058665094094, 0.003224211845211976, 0.0031506782855009146, 0.003448265751988282, 0.003431622577986645, 0.0032195384901443375, 0.0034623038857710285, 0.0031651257576079173, 0.002874380137395766, 0.0032395632261026353, 0.0032723050850357953, 0.0032268618709462367, 0.0032164938156786153, 0.002790360075799194, 0.003371393179733254, 0.002467618687042964, 0.0026477426803678426, 0.0033254013288870152, 0.0032388720130508623, 0.003267193921400832, 0.003531608469535183, 0.0033366448619774488, 0.003363158450047758, 0.0033050983186787577, 0.0027706456595012778, 0.0032638319493248813, 0.0031899814866994925, 0.003167247065884535, 0.0033004277559355545, 0.0032514815215470656, 0.0024340186574865036, 0.0021087669295950413, 0.003350891855344198, 0.0033631970267147516, 0.0032855510466442447, 0.0032056187887277332, 0.0033190655545987185, 0.003365261881119648, 0.0032976107074504137, 0.0032613730843612564, 0.0033021718175644085, 0.0032760488211643654, 0.004269916019891947, 0.003307593819489653, 0.0032748996351186326, 0.007116515280208218, 0.004515207108109338, 0.0033438888991117856, 0.007215656809083287, 0.0019026122228462498, 0.004947167253037719, 0.0032880462250904135, 0.003364087439980464, 0.0032256176098232284, 0.003292798996904871, 0.003408069492835456, 0.00337527113919113, 0.003044498853243413, 0.003267523089323985, 0.003805922106142375, 0.003406961227593592, 0.003219473249496295, 0.003194474159074381, 0.0019578563020166214, 0.0021194700021901986, 0.003337031859231501, 0.002833487166148402, 0.0032974044662239512, 0.0034757848646348234, 0.003390469259535312, 0.0032574792305031675, 0.0033945883646674013, 0.005215659281667577, 0.0034675584991766886, 0.003372567473738048, 0.0050793232337810355, 0.0032749961425830236, 0.00327808631268153, 0.003391072598502862, 0.0053909756925050924, 0.004899025634244467, 0.0019495546671135253, 0.0027690358586471954, 0.0021170734594043094, 0.004399560966345003, 0.003298417908147712, 0.005248884560118597, 0.0034661677499064777, 0.0030764118672347005, 0.0034557780984821856, 0.003090441899909081, 0.003286424953189313, 0.002746472929850629, 0.008345375076280128, 0.006873189553397335, 0.0033781988509052918, 0.002664038359846981, 0.0024186120663409923, 0.001626302343718935, 0.003446677909275655, 0.005568745364305584, 0.011650536427414397, 0.002211362819613625, 0.006852419193463066, 0.0034591501315148912, 0.0032157212885234095, 0.0027888254006041163, 0.003217655393760068, 0.0, 0.0035564380221490563, 0.0032881692118584137, 0.0020576011782769727, 0.00322597857904773, 0.011500951650699635, 0.007897931489146645, 0.004113065239496338, 0.002599254384812595, 0.0051286144513092375, 0.0033759653489821182, 0.003389418020308829, 0.0035988028596904875, 0.0039002435911469147, 0.0032646466490524463, 0.003191538676541214, 0.0032818089603540005, 0.003200340400540261, 0.0039433225626967355, 0.00651290617434353, 0.0022416851212559382, 0.002078342490228238, 0.003385159620106715, 0.003468051337379611, 0.003510663275476698, 0.003523489933364409, 0.003473418151503927, 0.003557953464133263, 0.0031546375015353728, 0.003240729019123058, 0.0032096251946304544, 0.0036837914711751003, 0.00323774528011621, 0.003300264362259205, 0.0032870160860989203, 0.0034102773740321266, 0.0033212487116993693, 0.003477808842863569, 0.003450324980924435, 0.0034371695874146554, 0.0035104977101090037, 0.0027966638039228823, 0.0033370309301663436, 0.0032982772795009333, 0.003221270888751101, 0.003311273288870849, 0.0032923454989023187, 0.003287840886901921, 0.003246096829359806, 0.003374404273583486, 0.0036040544937206978, 0.003543302922823548, 0.003188717328056627, 0.003117878242092108, 0.004572122495905632, 0.0032336464169770546, 0.0033546953686605843, 0.0033507166924112613, 0.003403101047713169, 0.0034591358111690372, 0.003512386909196687, 0.0035266779797162627, 0.004012665714268995, 0.0031147196770046947, 0.0031675219462327427, 0.0033040097128163447, 0.0033329627492371112, 0.003150034452476307, 0.004598635675840584, 0.0032155259432483955, 0.0032120159654414966, 0.0031678659028331678, 0.0031182719073252103, 0.0031945332717748624, 0.0031662392927552237, 0.003247765592933999, 0.0032205737703012056, 0.0032646445239312345, 0.003235074326791271, 0.0032091208014509978, 0.0030170491212051045, 0.006115205555523562, 0.0033560099928429494, 0.003866972023507183, 0.003626409193558729, 0.005341512714283517, 0.00339940041940229, 0.0032802256611207444, 0.0031923623519256555, 0.003208613951475116, 0.003315923688274725, 0.0030535473678946945, 0.0031352359131960913, 0.003398766057189667, 0.003336474585399162, 0.0033933813678125083, 0.0032993637515340364, 0.0032539076315800876, 0.0032730330380345684, 0.003277096622656353, 0.005101101415008993, 0.0032080683899901935, 0.0031835912651309493, 0.0033942612068446993, 0.0031584958055428376, 0.00304583731654068, 0.003226812733233765, 0.0031385425434146265, 0.0033681568018283483, 0.003228922023848682, 0.00334183488426459, 0.004347541065903674, 0.0038780498624063236, 0.005370415477870831, 0.0035269054911476316, 0.003177123138544852, 0.003112289252712477, 0.006769407579956567, 0.003026348320603768, 0.003106379118246414, 0.003122159423730182, 0.00676764062966777, 0.0031715000525345607, 0.0073900056404691, 0.0032826014904579607, 0.004229972467610041, 0.0032498109154475087, 0.003262574046237446, 0.0033069303578660285, 0.008970275616912673, 0.0031736603456925735, 0.0031648552846949224, 0.0032141504039705547, 0.0031016270427209262, 0.0031126382156558686, 0.0033126716821740856, 0.003111049797814406, 0.003120011622344615, 0.004341746347528227, 0.003325625393674117, 0.010245312908945856, 0.003377919630422754, 0.009225684379561784, 0.0031943878103887617, 0.00506029327812724, 0.0032072382525877236, 0.003208336915003685, 0.0031424285197139052, 0.003130962591524964, 0.004577374651805015, 0.0033015629309815966, 0.007713802483892269, 0.0031791413693673075, 0.009176808222127953, 0.003292021071265262, 0.003288899767869543, 0.003378421820386868, 0.0032888664561398695, 0.003313142145642897, 0.0033782986737514037, 0.0033109878074149618, 0.0032618327378083085, 0.0031236904076260875, 0.0032110534682957916, 0.003154138389743653, 0.0031563855484199785, 0.0034540705807774843, 0.0031188131738102256, 0.003303047685935157, 0.0, 0.00318278059002305, 0.003321625864863083, 0.006227278521638617, 0.0032883917348478756, 0.0033784666189637467, 0.003249555855765056, 0.0032247197214197248, 0.006089567647219057, 0.0032420634038721834, 0.0035642430521488485, 0.0029540993225082687, 0.0030163293144861024, 0.0031145789343638573, 0.0033203800143466024, 0.0033935529112612186, 0.0033336790232755113, 0.00871983970022402, 0.0032587309428044426, 0.0033067103836360045, 0.003273458908815452, 0.003335262453894298, 0.007702338982360332, 0.00347316437974167, 0.003449278416920205, 0.0032123351148175177, 0.0032166453210927836, 0.006928770453872968, 0.003071125143477558, 0.0030850460582325734, 0.0031238025278714234, 0.0034995416164528763, 0.0, 0.0035353863183298423, 0.0033473903848834554, 0.0032258103726115106, 0.003246142013810767, 0.006699919879319455, 0.0032431950229057753, 0.0032445243213661836, 0.003983660326481792, 0.0032714817819975697, 0.0031349334704405167, 0.003156672864966794, 0.0031608379284327123, 0.003557197649727964, 0.0031615600084857376, 0.0053819008000107185, 0.0033359619040445494, 0.0032083108837832317, 0.004649874749956333, 0.0031203857155158434, 0.0032863743145954537, 0.0031263290828448774, 0.0032772264948144423, 0.003259300887007177, 0.0032203597102818554, 0.0031645250844414595, 0.0031574079698339238, 0.003000235896192363, 0.0031657567136553176, 0.003192487162181502, 0.003148863821120023, 0.0033067610758858073, 0.003222303299884881, 0.008902212511697268, 0.003198143831077773, 0.00320239852003023, 0.0032675880995741502, 0.003221992850200131, 0.00324532008999203, 0.0032183438333475136, 0.0032854578357058095, 0.0, 0.0031017783515728922, 0.0032478249155043527, 0.003052114886865085, 0.00500253928105761, 0.003855435222417484, 0.003354872142819278, 0.003443995052435725, 0.003260342881019143, 0.0032450236068004997, 0.0032209760472520894, 0.007934346459713202, 0.003048858990178299, 0.0032925443453964676, 0.0032741642564398134, 0.0031676738684879367, 0.003133091791870366, 0.00889805088198051, 0.0031018978588321293, 0.0030524140290793802, 0.0031562560293703743, 0.007131640429817776, 0.0033788389008408523, 0.0033251300757210873, 0.003222858350898242, 0.0032869993442316858, 0.0031720511535276367, 0.0033032704137343945, 0.0031795907093427273, 0.003203003714503943, 0.006579440077764407, 0.0030046416699204344, 0.003143117228482202, 0.0032758601419667533, 0.0031630527383565833, 0.0032614702604873455, 0.003731195050476675, 0.0032079556061464644, 0.0032619335373145217, 0.0033558767933430946, 0.0032696896318085895, 0.0033908607884549015, 0.003209273914465072, 0.0032828445955527187, 0.0030873034705511306, 0.0032241337826795706, 0.003145553081453288, 0.003213444526718206, 0.0032496507305146494, 0.0031416022606278343, 0.003292151368139176, 0.0032814755652573555, 0.006500123015774869, 0.0031737749097418156, 0.003177566957140131, 0.007700965166630181, 0.0032105872642421026, 0.003147668653291248, 0.003152490892154785, 0.0031096933138113616, 0.003086823457458204, 0.0030664234536024446, 0.007198677990874883, 0.003061076872983808, 0.003146656335122689, 0.0035807358175736505, 0.007257668046712308]),
    # Gas xenon gain during cooling phase: 2020-08-26. PTMs at 1.5 kV a part PMT 116 and 195 which were at 1,3 kV. PMTs 354, 362, 386, 427 were off.
    # Note that during led calibration of August 26th PMTs 264, 272, 285, 300, 308, 311, 313, 316, 325, 339, 356, 359 didn't acquire data. For them I have used 
    # the results from 2020-08-23. For PMT 137, which were always off, I have used the result from 2020-07-24.
    'TemporaryGXe_1500V_PMT116_1300_PMT195_1300': np.array([0.0018913855076313215, 0.0028329303195210854, 0.003696779568714482, 0.00437507432021853, 0.001996949208878799, 0.009961701975866035, 0.0017772253043406426, 0.001984670560387615, 0.0024106259969455175, 0.002184836974062895, 0.001585764938423848, 0.002037984644835631, 0.005594197364915706, 0.0016298129740158446, 0.0018872198407832604, 0.011225649089548274, 0.002052100452660691, 0.002318167535101193, 0.002838329674647198, 0.0035642224196595833, 0.0017781634016245385, 0.004633144116886573, 0.002289274985597026, 0.002069210756382541, 0.0027709248243609754, 0.001766697490778006, 0.0020700697953945704, 0.0013131365764095769, 0.006587680894929626, 0.004716262165968249, 0.002411249925336603, 0.00217795954855625, 0.002720593928321278, 0.006686006072092426, 0.0028557936798272537, 0.002729777242750512, 0.001651236878142821, 0.0015772316176516598, 0.0016964665955370311, 0.0019469849448772883, 0.004419908356758589, 0.0023144903864681485, 0.0033107615250404015, 0.00388891356717293, 0.00336419684258327, 0.002945543318083504, 0.003534182427186866, 0.002148928893598179, 0.0032226429533104488, 0.0029766678216702025, 0.0018257660002343557, 0.0022443642313925193, 0.002324447484183, 0.001442800327441055, 0.0023705379495792137, 0.00467149415285596, 0.002366387743065802, 0.002693262298023729, 0.002570474446044821, 0.001715385927823272, 0.004168023706469527, 0.007101648180826111, 0.002169853150554484, 0.0032049775919625605, 0.0018786509168759959, 0.0032775091198171406, 0.002756448923253263, 0.0017568888024852105, 0.00187658983737892, 0.003417774605141497, 0.0015293945629160906, 0.0017769059162614268, 0.0026578398446241284, 0.0019953923978381087, 0.00320486045178322, 0.0027950134574866252, 0.005149023180933853, 0.005720877494360243, 0.003065457646812941, 0.00283682663381561, 0.003683681539070039, 0.0024885681686736016, 0.001844258067167463, 0.004049870423071719, 0.0016506221845253518, 0.0022715375069700715, 0.002178143493128193, 0.00253108529449206, 0.0025282094297333635, 0.0016999276161629353, 0.0026022787584169716, 0.002692058905927995, 0.00217067905550609, 0.002365864344937322, 0.0025548421658688637, 0.0022993549263406097, 0.002771012603834756, 0.004210623212213024, 0.0028767564251173205, 0.002197613725837754, 0.002540410382383294, 0.0019383332535220083, 0.003258263760085124, 0.001804827542700756, 0.0025293588486904506, 0.0016086208262161828, 0.0031364695839779418, 0.0021862923926718853, 0.001877879615175329, 0.0020814122154190202, 0.002855499937414323, 0.0017303275953518074, 0.003903266275885624, 0.0033862886869877396, 0.0022682478039384103, 0.0017055725878190754, 0.0071381790498803955, 0.002513168046912445, 0.002311656670134927, 0.00332235028050492, 0.004448337267388257, 0.0018350142429037224, 0.006575110967955619, 0.003972444183666391, 0.002277479734723085, 0.0016980515265769668, 0.0063179086588284845, 0.00574765531695736, 0.0022819305588718405, 0.0021116756775623096, 0.0025914580341831897, 0.002739416806404835, 0.003900428988912452, 0.0023163668270681234, 0.0018240767044344103, 0.0021748059253728913, 0.0029514024755758293, 0.003121944534851669, 0.001934350337542683, 0.003206764326222533, 0.0018074649595250104, 0.0018848610143651242, 0.002067843722875183, 0.0025224804471985942, 0.0020268220510544447, 0.004192759933948683, 0.002920116218152507, 0.0026530526951125313, 0.0027466610046452448, 0.006602727621067973, 0.0028924231182333284, 0.0017969804938841386, 0.009781780438907988, 0.00794829159660931, 0.003030893058002169, 0.002496114655805185, 0.002114738555261509, 0.002450957295261546, 0.004718501704421784, 0.0028979619187164277, 0.003087420871295964, 0.002051059186294451, 0.0018634219360359656, 0.006358789044438128, 0.001969407274251086, 0.0025220845874474648, 0.0028948655891575955, 0.00237505823228227, 0.0033831937811954808, 0.002736782964332214, 0.005818693653639101, 0.002308505945878109, 0.0029053026653979097, 0.002330543373046142, 0.00654413980679969, 0.00698234435002539, 0.0022231808182639036, 0.0017446420774420263, 0.0025669984134388877, 0.009366903847886285, 0.008581205292840828, 0.0027559464412873374, 0.00277763255418063, 0.0020695090158795704, 0.002501384944697325, 0.005222627917887776, 0.0020846779483866063, 0.003196159798069177, 0.002872181469618931, 0.002301601429487202, 0.0021034939396079025, 0.0030224331354734165, 0.0019893005213877718, 0.002726953173609682, 0.0019325057577266537, 0.005804197604245136, 0.0022692359770237016, 0.0017825024653336707, 0.005069388953867662, 0.0034466043813436028, 0.004689991227965938, 0.00273211707659124, 0.0032841906591539113, 0.0024387912580258854, 0.0023848961114464693, 0.004069684531835648, 0.003171117806051746, 0.0038229778532466343, 0.002241515177248813, 0.0031441731961773865, 0.0022526855226299475, 0.00316619384364768, 0.0016989889731659903, 0.002803478436528385, 0.002416521797375037, 0.0027449635712057945, 0.004446999632257134, 0.0025765943820510134, 0.003694654329914762, 0.0024075435981871372, 0.0027733460296294424, 0.004269443020645269, 0.0020574472510042955, 0.0025914120822620166, 0.0027485868398633042, 0.0020181329252415728, 0.002974431634666397, 0.0017365481304363868, 0.0016434639429044006, 0.001935120195196219, 0.0035341396939595564, 0.003604085232575598, 0.0018292338497317437, 0.003073994097686485, 0.0032912553027458847, 0.002599761840346838, 0.0016891441736839753, 0.0019764396087702373, 0.002330534157302106, 0.00539818879873659, 0.002310725520891357, 0.0018722075965177624, 0.002055090093842948, 0.002106257156108381, 0.0028469059375260284, 0.001862593024914504, 0.00300044279873789, 0.004893136544445233, 0.0021357857814955374, 0.0028847622328487393, 0.0023119527981777794, 0.002474488086169222, 0.002144840180437844, 0.005133238484302647, 0.0022858365089094487, 0.0018585400178827308, 0.0037841442944615486, 0.0015829158326944779, 0.0018211344964991555, 0.002262940436688199, 0.002539090750845904, 0.0017464998962838279, 0.0035483571687554972, 0.0024663678655986775, 0.002008281610231964, 0.002623037329687509, 0.0069075577977451095, 0.002206993937430013, 0.004130113885783556, 0.003072898701877684, 0.005710786232711787, 0.0015025742173054467, 0.0027172510405197225, 0.0037580901102031595, 0.0025534599846298942, 0.0031427906558571957, 0.0034611236249829777, 0.0025832954593037836, 0.0027046707818321383, 0.0023291758591620733, 0.0023274916632390038, 0.002728427321044224, 0.003396074501190261, 0.0021588517457397107, 0.0012381232489625457, 0.0063829473749597194, 0.002667078493335525, 0.0034301603914327302, 0.002372438929902116, 0.0023425572495472664, 0.0024726282581416076, 0.002848764338123287, 0.002428856928395265, 0.002517859632438592, 0.0030315698326429255, 0.0017466207235832343, 0.005217964227484815, 0.004347367517159481, 0.006442894318783412, 0.0018544717037903263, 0.002525774416701802, 0.002602160660191792, 0.007742516198873608, 0.00294950216730862, 0.001700879367320429, 0.003111400107665771, 0.00865683531684949, 0.0022807771905763753, 0.009218236549309608, 0.0024456612079878574, 0.005410614819507854, 0.0028129578807012455, 0.0016687270641962161, 0.0023628348729000203, 0.010478536909317196, 0.002475453650659083, 0.0021589339395923232, 0.0021277642832587085, 0.002547499146350737, 0.0027190836038749645, 0.004718814158406194, 0.003740313463391579, 0.002771188820295577, 0.0053299944786254355, 0.0020487357822649662, 0.012342936641473543, 0.0014972673341446246, 0.010766301566782786, 0.0031118555221109605, 0.0060550182462789405, 0.0034282070004852446, 0.001586184308178554, 0.0023931725616052587, 0.0020824353212372344, 0.0054331609030498435, 0.002107026704860344, 0.008972450443514761, 0.0026797718247766247, 0.010470386404649552, 0.0032648732098947888, 0.003055804916461226, 0.0027792140113769358, 0.0019935943287507757, 0.002380983742450868, 0.0014599612252190258, 0.0020506272551530608, 0.0021531884807401308, 0.003043216988324111, 0.0031996358651849425, 0.0017124711833935017, 0.001891016214539326, 0.0045496340924712775, 0.002766168288028441, 0.0027179653168710044, 0.0, 0.002754043405242351, 0.0017585425927124242, 0.008331080900688466, 0.0016391235074188577, 0.002999750063167117, 0.0021292661962072318, 0.0016492236215519176, 0.0, 0.002527473937441381, 0.0036574192470481288, 0.002395678699007992, 0.0033990683321537354, 0.0025049096885944016, 0.0037939577948518655, 0.0024547112762143383, 0.0023596804100716818, 0.010287069941805765, 0.0017153928031385676, 0.0015717637006210656, 0.002100617427375109, 0.0015754257434814495, 0.009798186009716317, 0.0020861172093601136, 0.004006851293832444, 0.0030821083503513144, 0.00239727969255997, 0.007905763650587892, 0.0018327472246340805, 0.0030789487480697153, 0.002071639829308905, 0.0041372470827009625, 0.0, 0.004282237008723668, 0.0023747568562688033, 0.004292216768033768, 0.002042670376147806, 0.008534367788272138, 0.0028775676274306453, 0.0019266611565852932, 0.004834291713144765, 0.0019681755727347532, 0.002208790977927611, 0.002444541703704623, 0.001572392160052091, 0.004483713866812388, 0.0016433066375977996, 0.0060434553398506004, 0.0014532686103293129, 0.002174025782317918, 0.004625556508048827, 0.0024305564860750716, 0.0016486018072638389, 0.002763296284231834, 0.0030762143453881064, 0.0016107746745276064, 0.0018433723871843566, 0.0025182550488981013, 0.0019148637799870107, 0.0026550121881693364, 0.0038759355149143716, 0.001660879064007765, 0.0014834466653395975, 0.002786726452705704, 0.0023605268209442613, 0.010350303981071953, 0.002498362221007697, 0.0027900634992199073, 0.0020519533421701837, 0.0014063444738655032, 0.0016209430013973812, 0.0019352324869225694, 0.0013698008809376562, 0.0, 0.001948838900430677, 0.0024901281080400633, 0.0019082089974197067, 0.0060865761777528764, 0.004663705800719834, 0.0028794090929727487, 0.0035284355866156206, 0.0023118210772410724, 0.0016733983450029509, 0.0023574837856956093, 0.01046771505621165, 0.004030533879059483, 0.00267563284668105, 0.0016200247075792722, 0.002544053968185267, 0.0025661826650413017, 0.01106218438209768, 0.002686885124319307, 0.002956749180054523, 0.0018572212033063158, 0.008405035242337903, 0.0023134762610468314, 0.0034249815211941973, 0.003679627731558352, 0.0015820842471901228, 0.0013771869152846507, 0.002045104508947996, 0.003917741542734526, 0.0023574261245130827, 0.007578232348928459, 0.001968257465821998, 0.001984078657361999, 0.0019311948583023433, 0.0022378029801674235, 0.0022192332466123275, 0.004338343142743706, 0.00199774284607738, 0.003870161646522202, 0.002180721057496781, 0.0023883338441824413, 0.0028055268356319244, 0.0019023410933635704, 0.0036657310329459864, 0.002881433041221169, 0.0022461807080895223, 0.002503374235835245, 0.0026723239854667667, 0.0033311042560432856, 0.0035594207263907125, 0.0036843418062460645, 0.0024330151120525866, 0.00771616464465055, 0.001993136311912788, 0.002274426748510561, 0.008629839236751409, 0.002412838932025963, 0.002240835748554997, 0.0024637134196302643, 0.0035695519800371983, 0.0023546273100460463, 0.002939102468962238, 0.008229204084591513, 0.002066044557108603, 0.0019630113602779544, 0.004233394685236203, 0.008508536224326645]),
}
