# -*- coding: utf-8 -*-
from setuptools import setup

package_dir = \
{'': 'src'}

packages = \
['dsl']

package_data = \
{'': ['*']}

install_requires = \
['astor>=0.8.1,<0.9.0', 'nr.util>=0.4.0,<0.5.0']

setup_kwargs = {
    'name': 'craftr-dsl',
    'version': '0.7.1',
    'description': 'Domain specific language for the Craftr build system.',
    'long_description': '# craftr-dsl\n\nThe Craftr DSL is a transpiler for the Python language that introduces the concept of\n**closures**, **function calls without parentheses** and a few other syntactical sugar\ninto the language. The language is a full superset of Python 3 code. The added syntax\nfeatures should only be used where it makes the code more readable or where it is\nsemantically more relevant.\n\n## Installation\n\nFrom Pip:\n\n    $ pip install craftr-dsl[codegen,colors]\n\nLatest from GitHub:\n\n    $ pip install git+https://github.com/craftr-build/craftr-dsl\n\nRequirements: Python 3.8 or newer\n\n## Hello, World!\n\nA convoluted Hello, World! example in Craftr DSL might look like this:\n\n```py\n# hello.craftr\nworld = { self(\'World!\') }\nworld { print(\'Hello,\', self) }\n```\n\nThis is transpiled to\n\n```py\n# $ python -m craftr.dsl hello.craftr -E | grep -v -e \'^$\'\ndef _closure_1(self, *arguments, **kwarguments):\n    self(\'World!\')\nworld = _closure_1\ndef _closure_2(self, *arguments, **kwarguments):\n    print(\'Hello,\', self)\nworld(_closure_2)\n```\n\nAnd evaluates to\n\n```py\n# $ python -m craftr.dsl hello.craftr\nHello, World!\n```\n\n## Language features\n\nThe Craftr DSL grammar and code generator can be configured to an extend to turn some\nlanguage features and semantics on and off. What will be shown below in most examples\nis compatible with the default configuration unless otherwise noted.\n\n### Closures\n\nClosures are formed with the following syntax: `[ arg -> | (arg1, arg2, ...) -> ] { body }`. A closure without\nan argument list automatically has the signature `(self, *argnames, **kwargnames)`.\n\n<table align="center"><tr><th>Craftr DSL</th><th>Python</th></tr>\n\n<tr><td>\n\n```py\nfilter({ self % 2 }, range(5))\n```\n</td><td>\n\n```py\ndef _closure_1(self, *argnames, **kwargnames):\n    self % 2\nfilter(_closure_1, range(5))\n```\n</td></tr>\n\n\n<tr><td>\n\n```py\nfilter(x -> x % 2, range(5))\n```\n</td><td>\n\n```py\ndef _closure_1(x):\n    return x % 2\nfilter(_closure_1, range(5))\n```\n</td></tr>\n\n\n<tr><td>\n\n```py\nreduce((a, b) -> {\n  a.append(b * 2)\n  return a\n}, [1, 2, 3], [])\n```\n</td><td>\n\n```py\ndef _closure_1(a, b):\n    a.append(b * 2)\n    return a\nreduce(_closure_1, [1, 2, 3], [])\n```\n</td></tr>\n\n</table>\n\n\n### Function calls without parentheses\n\nSuch function calls are only supported at the statement level. A function can be called without parentheses by\nsimply omitting them. Variadic and keyword arguments are supported as expected. Applying a closure on an object\nis basically the same as calling that object with the function, and arguments following the closure are still\nsupported.\n\n\n<table align="center"><tr><th>Craftr DSL</th><th>Python</th></tr>\n\n<tr><td>\n\n```py\nprint \'Hello, World!\', file=sys.stderr\n```\n</td><td>\n\n```py\nprint(\'Hello, World!\', file=sys.stderr)\n```\n</td></tr>\n\n\n<tr><td>\n\n```py\nmap {\n  print(\'Hello,\', self)\n}, [\'John\', \'World\']\n```\n</td><td>\n\n```py\ndef _closure_1(self, *arguments, **kwarguments):\n    print(\'Hello,\', self)\nmap(_closure_1, [\'John\', \'World\'])\n```\n</td></tr>\n\n</table>\n\n\n### Unseparated arguments & colon keyword arguments\n\nThe Craftr DSL allows passing arguments to function calls without separation by commas.\nKeyword arguments may be specified using colons (`:`) instead of equal signs (`=`).\n\n<table>\n\n<tr><th>Craftr DSL</th><th>Python</th></tr>\n\n<tr><td>\n\n```py\nprint \'Hello, World!\' 42 * 1 + 10 file: sys.stdout\n```\n</td><td>\n\n```py\nprint(\'Hello, World!\', 42 * 1 + 10, file=sys.stdout)\n```\n</td></tr>\n\n\n<tr><td>\n\n```py\ntask "hello_world" do: {\n  print "Hello, World!"\n}\n```\n</td><td>\n\n```py\ndef _closure_1(self, *arguments, **kwarguments):\n    print(\'Hello, World!\')\ntask(\'hello_world\', do=_closure_1)\n```\n</td></tr>\n\n\n<tr><td>\n\n```py\nlist(map {\n  print(\'Hello,\', self)\n}, [\'John\', \'World\'])\n```\n\n> **Note**: Pitfall, this actually passes three arguments to `list()`.\n</td><td>\n\n```py\ndef _closure_1(self, *arguments, **kwarguments):\n    print(\'Hello,\', self)\nlist(map, _closure_1, [\'John\', \'World\'])\n```\n</td></tr>\n\n</table>\n\n\n### Dynamic name resolution <sup>(non-default)</sup>\n\nFor some purposes and applications, dynamic name resolution may be desirable, for\nexample when writing `self` in front of every name to access a property of the closure\ntarget object is too cumbersome. For this, the Craftr DSL transpiler can generate code that\nlooks up, sets and deletes keys using subscript syntax on a particular variable name.\n\nUsing the `craftr.dsl.runtime` package, you can configure the transpiler and runtime\nto use dynamic name resolution. Example usage:\n\n```py\nfrom craftr.dsl.transpiler import transpile_to_ast\nfrom craftr.dsl.runtime import Closure\n\nclass Project:\n  def task(self, name: str, *, do: callable): ...\n\ncode = ...\nfilename = ...\n\n# Long form:\nmodule = transpile_to_ast(code, filename, Closure.get_options())\ncode = compile(module, filename, \'exec\')\nscope = {\'__closure__\': Closure(None, None, Project())}\nexec(code, scope)\n\n# Shorthand form:\nClosure(None, None, Project()).run_code(code, filename)\n```\n\nThe `Closure.get_options()` function returns `TranspileOptions` that instruct the transpiler\nto convert name lookups into subscripts on the `__closure__` variable, add a\n`@__closure__.child` decoration before every closure function definition and to add a\n`__closure__,` argument to their arglist. The `Closure` object passed into the `scope`\non execution deals with the rest.\n\n<table>\n\n<tr><th>Craftr DSL</th><th>Python</th></tr>\n\n<tr><td>\n\n```py\ntask "foobar" do: {\n  return n_times\n}\n\ntask "belzebub" do: {\n  def n_times = 1\n  return n_times\n}\n\ntask "cheeky" do: {\n  def n_times = 1\n  return (() -> n_times )()\n}\n```\n</td><td>\n\n```py\n@__closure__.child\ndef _closure_1(__closure__, self, *arguments, **kwarguments):\n    return __closure__[\'n_times\']\n__closure__[\'task\'](\'foobar\', do=_closure_1)\n\n@__closure__.child\ndef _closure_2(__closure__, self, *arguments, **kwarguments):\n    n_times = 1\n    return n_times\n__closure__[\'task\'](\'belzebub\', do=_closure_2)\n\n@__closure__.child\ndef _closure_3(__closure__, self, *arguments, **kwarguments):\n    n_times = 1\n    @__closure__.child\n    def _closure_3_closure_3(__closure__):\n        return n_times\n    return _closure_3_closure_3()\n__closure__[\'task\'](\'cheeky\', do=_closure_3)\n```\n\n</td></tr>\n\n</table>\n\n\n### Limitations\n\nCraftr DSL is intended to behave as a complete syntactic superset of standard Python. However there are currently\nsome limitations, namely:\n\n* Literal sets cannot be expressed due to the grammar conflict with parameter-less closures\n* Type annotations are not currently supported\n* The walrus operator is not currently supported\n* Function calls without parenthesis do not support passing `*args` as the first argument as that is\n  interpreted as a multiplication expression.\n\n---\n\n<p align="center">Copyright &copy; 2021 Niklas Rosenstein</p>\n',
    'author': 'Niklas Rosenstein',
    'author_email': 'rosensteinniklas@gmail.com',
    'maintainer': 'None',
    'maintainer_email': 'None',
    'url': 'None',
    'package_dir': package_dir,
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'python_requires': '>=3.10,<4.0',
}


setup(**setup_kwargs)
