/****************************************************************************
**
** Copyright (C) 2013 Digia Plc and/or its subsidiary(-ies).
** Contact: http://www.qt-project.org/legal
**
** This file is part of the Qt Solutions component.
**
** $QT_BEGIN_LICENSE:BSD$
** You may use this file under the terms of the BSD license as follows:
**
** "Redistribution and use in source and binary forms, with or without
** modification, are permitted provided that the following conditions are
** met:
**   * Redistributions of source code must retain the above copyright
**     notice, this list of conditions and the following disclaimer.
**   * Redistributions in binary form must reproduce the above copyright
**     notice, this list of conditions and the following disclaimer in
**     the documentation and/or other materials provided with the
**     distribution.
**   * Neither the name of Digia Plc and its Subsidiary(-ies) nor the names
**     of its contributors may be used to endorse or promote products derived
**     from this software without specific prior written permission.
**
**
** THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
** "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
** LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
** A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
** OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
** SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
** LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
** DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
** THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
** (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
** OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE."
**
** $QT_END_LICENSE$
**
****************************************************************************/

#include "qtpropertybrowserutils_p.h"
#include "qtgroupboxpropertybrowser.h"

#include <QSet>
#include <QGridLayout>
#include <QLabel>
#include <QCheckBox>
#include <QComboBox>
#include <QGroupBox>
#include <QTimer>
#include <QMap>
#include <QCoreApplication>

#if QT_VERSION >= 0x040400
QT_BEGIN_NAMESPACE
#endif

class QtGroupBoxPropertyBrowserPrivate
{
    QtGroupBoxPropertyBrowser *q_ptr;
    Q_DECLARE_PUBLIC(QtGroupBoxPropertyBrowser)
public:

    void init(QWidget *parent);
    void updateAttributes();

    void propertyInserted(QtBrowserItem *index, QtBrowserItem *afterIndex);
    void propertyRemoved(QtBrowserItem *index);
    void propertyChanged(QtBrowserItem *index);
    QWidget *createEditor(QtProperty *property, QWidget *parent) const
        { return q_ptr->createEditor(property, parent); }
    QWidget *createAttributeEditor(QtProperty *property, QWidget *parent, BrowserCol attribute) const
    { return q_ptr->createAttributeEditor(property, parent,attribute); }
    BrowserCol columnToAttribute(const int col) const;

    void slotEditorDestroyed();
    void slotUpdate();

    struct WidgetItem
    {
        WidgetItem() : widget(nullptr), label(nullptr), widgetLabel(nullptr),
                unit(nullptr), pkAvg(nullptr), format(nullptr), minimum(nullptr), maximum(nullptr), check(nullptr),
                groupBox(nullptr), layout(nullptr), line(nullptr), parent(nullptr) { }
        QWidget *widget; // can be null
        QLabel *label;
        QLabel *widgetLabel;
        QComboBox *unit;
        QComboBox *pkAvg;
        QComboBox *format;
        QDoubleEdit *minimum;
        QDoubleEdit *maximum;
        QtBoolEdit *check;
        QGroupBox *groupBox;
        QGridLayout *layout;
        QFrame *line;
        WidgetItem *parent;
        QList<WidgetItem *> children;
    };
private:
    void updateLater();
    void updateItem(WidgetItem *item);
    void insertRow(QGridLayout *layout, int row) const;
    void removeRow(QGridLayout *layout, int row) const;

    bool hasHeader(WidgetItem *item) const;

    QMap<QtBrowserItem *, WidgetItem *> m_indexToItem;
    QMap<WidgetItem *, QtBrowserItem *> m_itemToIndex;
    QMap<QWidget *, WidgetItem *> m_widgetToItem;
    QGridLayout *m_mainLayout;
    QList<WidgetItem *> m_children;
    QList<WidgetItem *> m_recreateQueue;
    QList<BrowserCol> m_attributes;
};

void QtGroupBoxPropertyBrowserPrivate::init(QWidget *parent)
{
    m_mainLayout = new QGridLayout();
    parent->setLayout(m_mainLayout);
    QLayoutItem *item = new QSpacerItem(0, 0,
                QSizePolicy::Fixed, QSizePolicy::Expanding);
    m_mainLayout->addItem(item, 0, 0);
}

void QtGroupBoxPropertyBrowserPrivate::updateAttributes()
{
    return;
}

void QtGroupBoxPropertyBrowserPrivate::slotEditorDestroyed()
{
    QWidget *editor = qobject_cast<QWidget *>(q_ptr->sender());
    if (!editor)
        return;
    if (!m_widgetToItem.contains(editor))
        return;
    m_widgetToItem[editor]->widget = nullptr;
    m_widgetToItem.remove(editor);
}

void QtGroupBoxPropertyBrowserPrivate::slotUpdate()
{
    QListIterator<WidgetItem *> itItem(m_recreateQueue);
    while (itItem.hasNext()) {
        WidgetItem *item = itItem.next();

        WidgetItem *par = item->parent;
        QWidget *w = nullptr;
        QGridLayout *l = nullptr;
        int oldRow = -1;
        if (!par) {
            w = q_ptr;
            l = m_mainLayout;
            oldRow = m_children.indexOf(item);
        } else {
            w = par->groupBox;
            l = par->layout;
            oldRow = par->children.indexOf(item);
            if (hasHeader(par))
                oldRow += 2;
        }

        if (item->widget) {
            item->widget->setParent(w);
        } else if (item->widgetLabel) {
            item->widgetLabel->setParent(w);
        } else {
            item->widgetLabel = new QLabel(w);
            item->widgetLabel->setSizePolicy(QSizePolicy(QSizePolicy::Ignored, QSizePolicy::Fixed));
            item->widgetLabel->setTextFormat(Qt::PlainText);
        }
        int span = 1;
        if (item->widget)
            l->addWidget(item->widget, oldRow, 1, 1, 1);
        else if (item->widgetLabel)
            l->addWidget(item->widgetLabel, oldRow, 1, 1, 1);
        else
            span = 2;
        item->label = new QLabel(w);
        item->label->setSizePolicy(QSizePolicy(QSizePolicy::Fixed, QSizePolicy::Fixed));
        l->addWidget(item->label, oldRow, 0, 1, span);

        updateItem(item);
    }
    m_recreateQueue.clear();
}

void QtGroupBoxPropertyBrowserPrivate::updateLater()
{
    QTimer::singleShot(0, q_ptr, SLOT(slotUpdate()));
}

void QtGroupBoxPropertyBrowserPrivate::propertyInserted(QtBrowserItem *index, QtBrowserItem *afterIndex)
{
    WidgetItem *afterItem = m_indexToItem.value(afterIndex);
    WidgetItem *parentItem = m_indexToItem.value(index->parent());

    WidgetItem *newItem = new WidgetItem();
    newItem->parent = parentItem;

    QGridLayout *layout = nullptr;
    QWidget *parentWidget = nullptr;
    int row = -1;
    if (!afterItem) {
        row = 0;
        if (parentItem)
            parentItem->children.insert(0, newItem);
        else
            m_children.insert(0, newItem);
    } else {
        if (parentItem) {
            row = parentItem->children.indexOf(afterItem) + 1;
            parentItem->children.insert(row, newItem);
        } else {
            row = m_children.indexOf(afterItem) + 1;
            m_children.insert(row, newItem);
        }
    }
    if (parentItem && hasHeader(parentItem))
        row += 2;

    if (!parentItem) {
        layout = m_mainLayout;
        parentWidget = q_ptr;;
    } else {
        if (!parentItem->groupBox) {
            m_recreateQueue.removeAll(parentItem);
            WidgetItem *par = parentItem->parent;
            QWidget *w = nullptr;
            QGridLayout *l = nullptr;
            int oldRow = -1;
            if (!par) {
                w = q_ptr;
                l = m_mainLayout;
                oldRow = m_children.indexOf(parentItem);
            } else {
                w = par->groupBox;
                l = par->layout;
                oldRow = par->children.indexOf(parentItem);
                if (hasHeader(par))
                    oldRow += 2;
            }
            parentItem->groupBox = new QGroupBox(w);
            parentItem->layout = new QGridLayout();
            parentItem->groupBox->setLayout(parentItem->layout);
            if (parentItem->label) {
                l->removeWidget(parentItem->label);
                delete parentItem->label;
                parentItem->label = nullptr;
            }
            if (parentItem->widget) {
                l->removeWidget(parentItem->widget);
                parentItem->widget->setParent(parentItem->groupBox);
                parentItem->layout->addWidget(parentItem->widget, 0, 0, 1, 2);
                for (unsigned short ind = 0; ind < m_attributes.count(); ind++) {
                    switch (m_attributes.at(ind)) {
                    case BrowserCol::UNIT:
                        if (parentItem->unit){
                            l->removeWidget(parentItem->unit);
                            parentItem->unit->setParent(parentItem->groupBox);
                            parentItem->layout->addWidget(parentItem->unit, 0, ind + 2, 1, 1);
                        }
                        break;
                    case BrowserCol::PKAVG:
                        if (parentItem->pkAvg){
                            l->removeWidget(parentItem->pkAvg);
                            parentItem->pkAvg->setParent(parentItem->groupBox);
                            parentItem->layout->addWidget(parentItem->pkAvg, 0, ind + 2, 1, 1);
                        }
                        break;
                    case BrowserCol::FORMAT:
                        if (parentItem->format){
                            l->removeWidget(parentItem->format);
                            parentItem->format->setParent(parentItem->groupBox);
                            parentItem->layout->addWidget(parentItem->format, 0, ind + 2, 1, 1);
                        }
                        break;
                    case BrowserCol::CHECK:
                        if (parentItem->check){
                            l->removeWidget(parentItem->check);
                            parentItem->check->setParent(parentItem->groupBox);
                            parentItem->layout->addWidget(parentItem->check, 0, ind + 2, 1, 1, Qt::AlignRight);
                        }
                        break;
                    default:
                        break;
                    }
                }
                parentItem->line = new QFrame(parentItem->groupBox);
            } else if (parentItem->widgetLabel) {
                l->removeWidget(parentItem->widgetLabel);
                delete parentItem->widgetLabel;
                parentItem->widgetLabel = nullptr;
            }
            if (parentItem->line) {
                parentItem->line->setFrameShape(QFrame::HLine);
                parentItem->line->setFrameShadow(QFrame::Sunken);
                parentItem->layout->addWidget(parentItem->line, 1, 0, 1, m_attributes.count() + 2);
            }
            l->addWidget(parentItem->groupBox, oldRow, 0, 1, m_attributes.count() + 2);
            updateItem(parentItem);
        }
        layout = parentItem->layout;
        parentWidget = parentItem->groupBox;
    }

    newItem->label = new QLabel(parentWidget);
    newItem->label->setSizePolicy(QSizePolicy(QSizePolicy::Fixed, QSizePolicy::Fixed));
    newItem->widget = createEditor(index->property(), parentWidget);
    if (!newItem->widget) {
        newItem->widgetLabel = new QLabel(parentWidget);
        newItem->widgetLabel->setSizePolicy(QSizePolicy(QSizePolicy::Ignored, QSizePolicy::Fixed));
        newItem->widgetLabel->setTextFormat(Qt::PlainText);
    } else {
        QObject::connect(newItem->widget, SIGNAL(destroyed()), q_ptr, SLOT(slotEditorDestroyed()));
        m_widgetToItem[newItem->widget] = newItem;
    }

    insertRow(layout, row);
    int span = 1;
    if (newItem->widget)
        layout->addWidget(newItem->widget, row, 1);
    else if (newItem->widgetLabel)
        layout->addWidget(newItem->widgetLabel, row, 1);
    else
        span = 2;
    layout->addWidget(newItem->label, row, 0, 1, span);
    for (unsigned short ind = 0; ind < m_attributes.count(); ind++) {
        switch (m_attributes.at(ind)) {
        case BrowserCol::UNIT:
            newItem->unit = dynamic_cast<QComboBox*>(createAttributeEditor(index->property(), parentWidget, BrowserCol::UNIT));
            if (newItem->unit)
                layout->addWidget(newItem->unit, row, ind+2, 1, 1);
            break;
        case BrowserCol::PKAVG:
            newItem->pkAvg = dynamic_cast<QComboBox*>(createAttributeEditor(index->property(), parentWidget, BrowserCol::PKAVG));
            if (newItem->pkAvg)
                layout->addWidget(newItem->pkAvg, row, ind+2, 1, 1);
            break;
        case BrowserCol::FORMAT:
            newItem->format = dynamic_cast<QComboBox*>(createAttributeEditor(index->property(), parentWidget, BrowserCol::FORMAT));
            if (newItem->format)
                layout->addWidget(newItem->format, row, ind+2, 1, 1);
            break;
        case BrowserCol::MINIMUM:
            newItem->minimum = dynamic_cast<QDoubleEdit*>(createAttributeEditor(index->property(), parentWidget, BrowserCol::MINIMUM));
            if (newItem->minimum)
                layout->addWidget(newItem->minimum, row, ind+2, 1, 1);
            break;
        case BrowserCol::MAXIMUM:
           newItem->maximum = dynamic_cast<QDoubleEdit*>(createAttributeEditor(index->property(), parentWidget, BrowserCol::MAXIMUM));
            if (newItem->maximum)
                layout->addWidget(newItem->maximum, row, ind+2, 1, 1);
            break;
        case BrowserCol::CHECK:
           newItem->check = dynamic_cast<QtBoolEdit*>(createAttributeEditor(index->property(), parentWidget, BrowserCol::CHECK));
            if (newItem->check)
                layout->addWidget(newItem->check, row, ind+2, 1, 1, Qt::AlignRight);
            break;
        default:
            break;
        }
    }

    m_itemToIndex[newItem] = index;
    m_indexToItem[index] = newItem;

    updateItem(newItem);
}

void QtGroupBoxPropertyBrowserPrivate::propertyRemoved(QtBrowserItem *index)
{
    WidgetItem *item = m_indexToItem.value(index);

    m_indexToItem.remove(index);
    m_itemToIndex.remove(item);

    WidgetItem *parentItem = item->parent;

    int row = -1;

    if (parentItem) {
        row = parentItem->children.indexOf(item);
        parentItem->children.removeAt(row);
        if (hasHeader(parentItem))
            row += 2;
    } else {
        row = m_children.indexOf(item);
        m_children.removeAt(row);
    }

    if (item->widget)
        delete item->widget;
    if (item->label)
        delete item->label;
    if (item->widgetLabel)
        delete item->widgetLabel;
    if (item->groupBox)
        delete item->groupBox;

    if (!parentItem) {
        removeRow(m_mainLayout, row);
    } else if (parentItem->children.count() != 0) {
        removeRow(parentItem->layout, row);
    } else {
        WidgetItem *par = parentItem->parent;
        QGridLayout *l = nullptr;
        int oldRow = -1;
        if (!par) {
            l = m_mainLayout;
            oldRow = m_children.indexOf(parentItem);
        } else {
            l = par->layout;
            oldRow = par->children.indexOf(parentItem);
            if (hasHeader(par))
                oldRow += 2;
        }

        if (parentItem->widget) {
            parentItem->widget->hide();
            parentItem->widget->setParent(nullptr);
        } else if (parentItem->widgetLabel) {
            parentItem->widgetLabel->hide();
            parentItem->widgetLabel->setParent(nullptr);
        } else {
            //parentItem->widgetLabel = new QLabel(w);
        }
        l->removeWidget(parentItem->groupBox);
        delete parentItem->groupBox;
        parentItem->groupBox = nullptr;
        parentItem->line = nullptr;
        parentItem->layout = nullptr;
        if (!m_recreateQueue.contains(parentItem))
            m_recreateQueue.append(parentItem);
        updateLater();
    }
    m_recreateQueue.removeAll(item);

    delete item;
}

void QtGroupBoxPropertyBrowserPrivate::insertRow(QGridLayout *layout, int row) const
{
    QMap<QLayoutItem *, QRect> itemToPos;
    int idx = 0;
    while (idx < layout->count()) {
        int r, c, rs, cs;
        layout->getItemPosition(idx, &r, &c, &rs, &cs);
        if (r >= row) {
            itemToPos[layout->takeAt(idx)] = QRect(r + 1, c, rs, cs);
        } else {
            idx++;
        }
    }

    const QMap<QLayoutItem *, QRect>::ConstIterator icend = itemToPos.constEnd();
    for (QMap<QLayoutItem *, QRect>::ConstIterator it = itemToPos.constBegin(); it != icend; ++it) {
        const QRect r = it.value();
        layout->addItem(it.key(), r.x(), r.y(), r.width(), r.height());
    }
}

void QtGroupBoxPropertyBrowserPrivate::removeRow(QGridLayout *layout, int row) const
{
    QMap<QLayoutItem *, QRect> itemToPos;
    int idx = 0;
    while (idx < layout->count()) {
        int r, c, rs, cs;
        layout->getItemPosition(idx, &r, &c, &rs, &cs);
        if (r > row) {
            itemToPos[layout->takeAt(idx)] = QRect(r - 1, c, rs, cs);
        } else {
            idx++;
        }
    }

    const QMap<QLayoutItem *, QRect>::ConstIterator icend = itemToPos.constEnd();
    for (QMap<QLayoutItem *, QRect>::ConstIterator it = itemToPos.constBegin(); it != icend; ++it) {
        const QRect r = it.value();
        layout->addItem(it.key(), r.x(), r.y(), r.width(), r.height());
    }
}

bool QtGroupBoxPropertyBrowserPrivate::hasHeader(WidgetItem *item) const
{
    if (item->widget)
        return true;
    return false;
}

void QtGroupBoxPropertyBrowserPrivate::propertyChanged(QtBrowserItem *index)
{
    WidgetItem *item = m_indexToItem.value(index);
    WidgetItem *parentItem = item->parent;
    updateItem(item);
    if (parentItem)
        updateItem(item);
}

void QtGroupBoxPropertyBrowserPrivate::updateItem(WidgetItem *item)
{
    QtProperty *property = m_itemToIndex[item]->property();
    if (item->groupBox) {
        QFont font = item->groupBox->font();
        font.setUnderline(property->isModified());
        item->groupBox->setFont(font);
        item->groupBox->setTitle(property->propertyName());
        item->groupBox->setToolTip(property->toolTip());
        item->groupBox->setStatusTip(property->statusTip());
        item->groupBox->setWhatsThis(property->whatsThis());
        item->groupBox->setEnabled(property->isEnabled());
    }
    if (item->label) {
        QFont font = item->label->font();
        font.setUnderline(property->isModified());
        item->label->setFont(font);
        item->label->setText(property->propertyName());
        item->label->setToolTip(property->toolTip());
        item->label->setStatusTip(property->statusTip());
        item->label->setWhatsThis(property->whatsThis());
        item->label->setEnabled(property->isEnabled());
        QPalette palette = item->label->palette();
        palette.setColor(QPalette::Foreground, property->foreground().color());
        item->label->setPalette(palette);
    }
    if (item->widgetLabel) {
        QFont font = item->widgetLabel->font();
        font.setUnderline(false);
        item->widgetLabel->setFont(font);
        item->widgetLabel->setText(property->valueText());
        item->widgetLabel->setToolTip(property->valueText());
        item->widgetLabel->setEnabled(property->isEnabled());
    }
    if (item->widget) {
        QFont font = item->widget->font();
        font.setUnderline(false);
        item->widget->setFont(font);
        item->widget->setEnabled(property->isEnabled());
        item->widget->setToolTip(property->valueText());
        QLabel *label = dynamic_cast<QLabel*>(item->widget);
        if (label){
            label->setText(property->valueText());
            QFontMetrics metrics(item->widget->fontMetrics());
            item->widget->setMaximumWidth(metrics.boundingRect("12.34<-180.00").width());
        }
    }
    if (item->unit){
        QFont font = item->unit->font();
        QFontMetrics metrics(item->unit->fontMetrics());
        font.setUnderline(false);
        item->unit->setFont(font);
        item->unit->setEnabled(property->isEnabled());
        item->unit->setCurrentText(property->unitText());
        item->unit->setMaximumWidth(40+metrics.boundingRect("dBmrW").width());
    }
    if (item->pkAvg){
        QFont font = item->pkAvg->font();
        QFontMetrics metrics(item->pkAvg->fontMetrics());
        font.setUnderline(false);
        item->pkAvg->setFont(font);
        item->pkAvg->setEnabled(property->isEnabled());
        item->pkAvg->setCurrentText(property->pkAvgText());
        item->pkAvg->setMaximumWidth(40+metrics.boundingRect("avg").width());
    }
    if (item->format){
        QFont font = item->format->font();
        QFontMetrics metrics(item->format->fontMetrics());
        font.setUnderline(false);
        item->format->setFont(font);
        item->format->setEnabled(property->isEnabled());
        item->format->setCurrentText(property->formatText());
        item->format->setMaximumWidth(45+metrics.boundingRect("Log<Deg").width());
    }
    if (item->minimum){
        QFont font = item->minimum->font();
        QFontMetrics metrics(item->minimum->fontMetrics());
        font.setUnderline(false);
        item->minimum->setFont(font);
        item->minimum->setEnabled(property->isEnabled());
        item->minimum->setMaximumWidth(metrics.boundingRect("-12.3e10").width());
    }
    if (item->maximum){
        QFont font = item->maximum->font();
        QFontMetrics metrics(item->maximum->fontMetrics());
        font.setUnderline(false);
        item->maximum->setFont(font);
        item->maximum->setEnabled(property->isEnabled());
        item->maximum->setMaximumWidth(metrics.boundingRect("+12.3e10").width());
    }
    if (item->check){
        item->check->setEnabled(property->isEnabled());
        item->check->setChecked(property->check());
    }
}

/*!
    \class QtGroupBoxPropertyBrowser

    \brief The QtGroupBoxPropertyBrowser class provides a QGroupBox
    based property browser.

    A property browser is a widget that enables the user to edit a
    given set of properties. Each property is represented by a label
    specifying the property's name, and an editing widget (e.g. a line
    edit or a combobox) holding its value. A property can have zero or
    more subproperties.

    QtGroupBoxPropertyBrowser provides group boxes for all nested
    properties, i.e. subproperties are enclosed by a group box with
    the parent property's name as its title. For example:

    \image qtgroupboxpropertybrowser.png

    Use the QtAbstractPropertyBrowser API to add, insert and remove
    properties from an instance of the QtGroupBoxPropertyBrowser
    class. The properties themselves are created and managed by
    implementations of the QtAbstractPropertyManager class.

    \sa QtTreePropertyBrowser, QtAbstractPropertyBrowser
*/

/*!
    Creates a property browser with the given \a parent.
*/
QtGroupBoxPropertyBrowser::QtGroupBoxPropertyBrowser(QWidget *parent)
    : QtAbstractPropertyBrowser(parent)
{
    d_ptr = new QtGroupBoxPropertyBrowserPrivate;
    d_ptr->q_ptr = this;

    d_ptr->init(this);
}

/*!
    Destroys this property browser.

    Note that the properties that were inserted into this browser are
    \e not destroyed since they may still be used in other
    browsers. The properties are owned by the manager that created
    them.

    \sa QtProperty, QtAbstractPropertyManager
*/
QtGroupBoxPropertyBrowser::~QtGroupBoxPropertyBrowser()
{
    const QMap<QtGroupBoxPropertyBrowserPrivate::WidgetItem *, QtBrowserItem *>::ConstIterator icend = d_ptr->m_itemToIndex.constEnd();
    for (QMap<QtGroupBoxPropertyBrowserPrivate::WidgetItem *, QtBrowserItem *>::ConstIterator it = d_ptr->m_itemToIndex.constBegin(); it != icend; ++it)
        delete it.key();
    delete d_ptr;
}

/*!
 \property QtGroupBoxPropertyBrowser::attributes
 \brief optional attribute column setting.

 Optional setting to include an attribute column

 \sa setAttribute1()
 */

QList<BrowserCol> QtGroupBoxPropertyBrowser::attributes() const
{
    return d_ptr->m_attributes;
}

void QtGroupBoxPropertyBrowser::setAttributes(const QList<BrowserCol> &attributeList)
{
    if (d_ptr->m_attributes == attributeList)
        return;

    d_ptr->m_attributes = attributeList;
    d_ptr->updateAttributes();

    QMapIterator<QtGroupBoxPropertyBrowserPrivate::WidgetItem *, QtBrowserItem *> it(d_ptr->m_itemToIndex);
    while (it.hasNext()) {
        d_ptr->updateItem(it.key());
    }
}

/*!
 \property QtGroupBoxPropertyBrowser::attribute1
 \brief optional attribute column setting.

 Optional setting to include an attribute column

 \sa setAttribute1()
 */

BrowserCol QtGroupBoxPropertyBrowser::attribute1() const
{
    return d_ptr->m_attributes[0];
}
void QtGroupBoxPropertyBrowser::setAttribute1(BrowserCol attribute)
{
    QList<BrowserCol> &attributeList = d_ptr->m_attributes;
    attributeList[0] = attribute;
    setAttributes(attributeList);
}

/*!
 \property QtGroupBoxPropertyBrowser::attribute2
 \brief optional attribute column setting.

 Optional setting to include an attribute column

 \sa setAttribute2()
 */
void QtGroupBoxPropertyBrowser::setAttribute2(BrowserCol attribute)
{
    QList<BrowserCol> &attributeList = d_ptr->m_attributes;
    attributeList[1] = attribute;
    setAttributes(attributeList);
}

BrowserCol QtGroupBoxPropertyBrowser::attribute2() const
{
    return d_ptr->m_attributes[1];
}

/*!
 \property QtGroupBoxPropertyBrowser::attribute3
 \brief optional attribute column setting.

 Optional setting to include an attribute column

 \sa setAttribute3()
 */

BrowserCol QtGroupBoxPropertyBrowser::attribute3() const
{
    return d_ptr->m_attributes[2];
}
void QtGroupBoxPropertyBrowser::setAttribute3(BrowserCol attribute)
{
    QList<BrowserCol> &attributeList = d_ptr->m_attributes;
    attributeList[2] = attribute;
    setAttributes(attributeList);
}

/*!
    \reimp
*/
void QtGroupBoxPropertyBrowser::itemInserted(QtBrowserItem *item, QtBrowserItem *afterItem)
{
    d_ptr->propertyInserted(item, afterItem);
}

/*!
    \reimp
*/
void QtGroupBoxPropertyBrowser::itemRemoved(QtBrowserItem *item)
{
    d_ptr->propertyRemoved(item);
}

/*!
    \reimp
*/
void QtGroupBoxPropertyBrowser::itemChanged(QtBrowserItem *item)
{
    d_ptr->propertyChanged(item);
}

#if QT_VERSION >= 0x040400
QT_END_NAMESPACE
#endif

#include "moc_qtgroupboxpropertybrowser.cpp"
