"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.RateLimitedApiKey = exports.ApiKey = void 0;
const iam = require("@aws-cdk/aws-iam");
const core_1 = require("@aws-cdk/core");
const apigateway_generated_1 = require("./apigateway.generated");
const usage_plan_1 = require("./usage-plan");
/**
 * Base implementation that is common to the various implementations of IApiKey
 */
class ApiKeyBase extends core_1.Resource {
    /**
     * Permits the IAM principal all read operations through this key.
     *
     * @param grantee The principal to grant access to.
     */
    grantRead(grantee) {
        return iam.Grant.addToPrincipal({
            grantee,
            actions: readPermissions,
            resourceArns: [this.keyArn],
        });
    }
    /**
     * Permits the IAM principal all write operations through this key.
     *
     * @param grantee The principal to grant access to.
     */
    grantWrite(grantee) {
        return iam.Grant.addToPrincipal({
            grantee,
            actions: writePermissions,
            resourceArns: [this.keyArn],
        });
    }
    /**
     * Permits the IAM principal all read and write operations through this key.
     *
     * @param grantee The principal to grant access to.
     */
    grantReadWrite(grantee) {
        return iam.Grant.addToPrincipal({
            grantee,
            actions: [...readPermissions, ...writePermissions],
            resourceArns: [this.keyArn],
        });
    }
}
/**
 * An API Gateway ApiKey.
 *
 * An ApiKey can be distributed to API clients that are executing requests
 * for Method resources that require an Api Key.
 */
class ApiKey extends ApiKeyBase {
    /**
     *
     */
    constructor(scope, id, props = {}) {
        super(scope, id, {
            physicalName: props.apiKeyName,
        });
        const resource = new apigateway_generated_1.CfnApiKey(this, 'Resource', {
            customerId: props.customerId,
            description: props.description,
            enabled: props.enabled || true,
            generateDistinctId: props.generateDistinctId,
            name: this.physicalName,
            stageKeys: this.renderStageKeys(props.resources),
            value: props.value,
        });
        this.keyId = resource.ref;
        this.keyArn = core_1.Stack.of(this).formatArn({
            service: 'apigateway',
            account: '',
            resource: '/apikeys',
            sep: '/',
            resourceName: this.keyId,
        });
    }
    /**
     * Import an ApiKey by its Id.
     */
    static fromApiKeyId(scope, id, apiKeyId) {
        class Import extends ApiKeyBase {
            constructor() {
                super(...arguments);
                this.keyId = apiKeyId;
                this.keyArn = core_1.Stack.of(this).formatArn({
                    service: 'apigateway',
                    account: '',
                    resource: '/apikeys',
                    sep: '/',
                    resourceName: apiKeyId,
                });
            }
        }
        return new Import(scope, id);
    }
    renderStageKeys(resources) {
        if (!resources) {
            return undefined;
        }
        return resources.map((resource) => {
            const restApi = resource;
            const restApiId = restApi.restApiId;
            const stageName = restApi.deploymentStage.stageName.toString();
            return { restApiId, stageName };
        });
    }
}
exports.ApiKey = ApiKey;
/**
 * An API Gateway ApiKey, for which a rate limiting configuration can be specified.
 *
 * @resource AWS::ApiGateway::ApiKey
 */
class RateLimitedApiKey extends ApiKeyBase {
    /**
     *
     */
    constructor(scope, id, props = {}) {
        super(scope, id, {
            physicalName: props.apiKeyName,
        });
        const resource = new ApiKey(this, 'Resource', props);
        if (props.apiStages || props.quota || props.throttle) {
            new usage_plan_1.UsagePlan(this, 'UsagePlanResource', {
                apiKey: resource,
                apiStages: props.apiStages,
                quota: props.quota,
                throttle: props.throttle,
            });
        }
        this.keyId = resource.keyId;
        this.keyArn = resource.keyArn;
    }
}
exports.RateLimitedApiKey = RateLimitedApiKey;
const readPermissions = [
    'apigateway:GET',
];
const writePermissions = [
    'apigateway:POST',
    'apigateway:PUT',
    'apigateway:PATCH',
    'apigateway:DELETE',
];
//# sourceMappingURL=data:application/json;base64,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