"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.DomainName = exports.SecurityPolicy = void 0;
const core_1 = require("@aws-cdk/core");
const apigateway_generated_1 = require("./apigateway.generated");
const base_path_mapping_1 = require("./base-path-mapping");
const restapi_1 = require("./restapi");
/**
 * The minimum version of the SSL protocol that you want API Gateway to use for HTTPS connections.
 */
var SecurityPolicy;
(function (SecurityPolicy) {
    SecurityPolicy["TLS_1_0"] = "TLS_1_0";
    SecurityPolicy["TLS_1_2"] = "TLS_1_2";
})(SecurityPolicy = exports.SecurityPolicy || (exports.SecurityPolicy = {}));
/**
 *
 */
class DomainName extends core_1.Resource {
    /**
     *
     */
    constructor(scope, id, props) {
        super(scope, id);
        const endpointType = props.endpointType || restapi_1.EndpointType.REGIONAL;
        const edge = endpointType === restapi_1.EndpointType.EDGE;
        if (!core_1.Token.isUnresolved(props.domainName) && /[A-Z]/.test(props.domainName)) {
            throw new Error('domainName does not support uppercase letters. ' +
                `got: '${props.domainName}'`);
        }
        const mtlsConfig = this.configureMTLS(props.mtls);
        const resource = new apigateway_generated_1.CfnDomainName(this, 'Resource', {
            domainName: props.domainName,
            certificateArn: edge ? props.certificate.certificateArn : undefined,
            regionalCertificateArn: edge ? undefined : props.certificate.certificateArn,
            endpointConfiguration: { types: [endpointType] },
            mutualTlsAuthentication: mtlsConfig,
            securityPolicy: props.securityPolicy,
        });
        this.domainName = resource.ref;
        this.domainNameAliasDomainName = edge
            ? resource.attrDistributionDomainName
            : resource.attrRegionalDomainName;
        this.domainNameAliasHostedZoneId = edge
            ? resource.attrDistributionHostedZoneId
            : resource.attrRegionalHostedZoneId;
        if (props.mapping) {
            this.addBasePathMapping(props.mapping);
        }
    }
    /**
     * Imports an existing domain name.
     */
    static fromDomainNameAttributes(scope, id, attrs) {
        class Import extends core_1.Resource {
            constructor() {
                super(...arguments);
                this.domainName = attrs.domainName;
                this.domainNameAliasDomainName = attrs.domainNameAliasTarget;
                this.domainNameAliasHostedZoneId = attrs.domainNameAliasHostedZoneId;
            }
        }
        return new Import(scope, id);
    }
    /**
     * Maps this domain to an API endpoint.
     *
     * @param targetApi That target API endpoint, requests will be mapped to the deployment stage.
     * @param options Options for mapping to base path with or without a stage.
     */
    addBasePathMapping(targetApi, options = {}) {
        const basePath = options.basePath || '/';
        const id = `Map:${basePath}=>${targetApi.node.uniqueId}`;
        return new base_path_mapping_1.BasePathMapping(this, id, {
            domainName: this,
            restApi: targetApi,
            ...options,
        });
    }
    configureMTLS(mtlsConfig) {
        if (!mtlsConfig)
            return undefined;
        return {
            truststoreUri: mtlsConfig.bucket.s3UrlForObject(mtlsConfig.key),
            truststoreVersion: mtlsConfig.version,
        };
    }
}
exports.DomainName = DomainName;
//# sourceMappingURL=data:application/json;base64,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