"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.PolicyStatementWithActions = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const common_substrings_1 = require("@udondan/common-substrings");
const RegexParser = require("regex-parser");
const access_level_1 = require("../access-level");
const _2_conditions_1 = require("./2-conditions");
/**
 * Adds "action" functionality to the Policy Statement
 */
class PolicyStatementWithActions extends _2_conditions_1.PolicyStatementWithCondition {
    constructor() {
        super(...arguments);
        this.accessLevelList = {};
        this.useNotAction = false;
        this.floydActions = [];
        this.cdkActionsApplied = false;
        this.isCompact = false;
    }
    /**
     * Injects actions into the statement.
     *
     * Only relevant for the main package. In CDK mode this only calls super.
     */
    toJSON() {
        // @ts-ignore only available after swapping 1-base
        if (typeof this.addResources == 'function') {
            this.cdkApplyActions();
            return super.toJSON();
        }
        const mode = this.useNotAction ? 'NotAction' : 'Action';
        const statement = super.toJSON();
        const self = this;
        if (this.hasActions()) {
            if (this.isCompact) {
                this.compactActions();
            }
            const actions = this.floydActions
                .filter((elem, pos) => {
                return self.floydActions.indexOf(elem) == pos;
            })
                .sort();
            statement[mode] = actions.length > 1 ? actions : actions[0];
        }
        return statement;
    }
    toStatementJson() {
        this.cdkApplyActions();
        // @ts-ignore only available after swapping 1-base
        return super.toStatementJson();
    }
    freeze() {
        // @ts-ignore only available after swapping 1-base
        if (!this.frozen) {
            this.cdkApplyActions();
        }
        return super.freeze();
    }
    cdkApplyActions() {
        if (!this.cdkActionsApplied) {
            const mode = this.useNotAction ? 'addNotActions' : 'addActions';
            const self = this;
            if (this.isCompact) {
                this.compactActions();
            }
            const uniqueActions = this.floydActions
                .filter((elem, pos) => {
                return self.floydActions.indexOf(elem) == pos;
            })
                .sort();
            // @ts-ignore only available after swapping 1-base
            this[mode](...uniqueActions);
        }
        this.cdkActionsApplied = true;
    }
    /**
     * Switches the statement to use [`NotAction`](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_notaction.html).
     */
    notAction() {
        this.useNotAction = true;
        return this;
    }
    /**
     * Checks weather actions have been applied to the policy.
     */
    hasActions() {
        return this.floydActions.length > 0;
    }
    /**
     * Adds actions by name.
     *
     * Depending on the "mode", actions will be either added to the list of [`Actions`](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_action.html) or [`NotAction`](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_notaction.html).
     *
     * The mode can be switched by calling `notAction()`.
     *
     * If the action does not contain a colon, the action will be prefixed with the service prefix of the class, e.g. `ec2:`
     *
     * @param action Actions that will be added to the statement.
     */
    to(action) {
        if (this.servicePrefix.length && action.indexOf(':') < 0) {
            action = this.servicePrefix + ':' + action;
        }
        this.floydActions.push(action);
        return this;
    }
    /**
     * Adds all actions of the statement provider to the statement, e.g. `actions: 'ec2:*'`
     */
    allActions() {
        if (this.servicePrefix.length) {
            this.to(`${this.servicePrefix}:*`);
        }
        else {
            this.to('*');
        }
        return this;
    }
    /**
     * Adds all actions that match one of the given regular expressions.
     *
     * @param expressions One or more regular expressions. The regular expressions need to be in [Perl/JavaScript literal style](https://developer.mozilla.org/en-US/docs/Web/JavaScript/Guide/Regular_Expressions) and need to be passed as strings,
     * For example:
     * ```typescript
     * allMatchingActions('/vpn/i')
     * ```
     */
    allMatchingActions(...expressions) {
        expressions.forEach((expression) => {
            for (const [_, actions] of Object.entries(this.accessLevelList)) {
                actions.forEach((action) => {
                    if (action.match(RegexParser(expression))) {
                        this.to(`${this.servicePrefix}:${action}`);
                    }
                });
            }
        });
        return this;
    }
    /**
     * Adds all actions with [access level](https://docs.aws.amazon.com/IAM/latest/UserGuide/access_policies_understand-policy-summary-access-level-summaries.html#access_policies_access-level) LIST to the statement
     *
     * Permission to list resources within the service to determine whether an object exists.
     *
     * Actions with this level of access can list objects but cannot see the contents of a resource.
     *
     * For example, the Amazon S3 action `ListBucket` has the List access level.
     */
    allListActions() {
        return this.addAccessLevel(access_level_1.AccessLevel.LIST);
    }
    /**
     * Adds all actions with [access level](https://docs.aws.amazon.com/IAM/latest/UserGuide/access_policies_understand-policy-summary-access-level-summaries.html#access_policies_access-level) READ to the statement
     *
     * Permission to read but not edit the contents and attributes of resources in the service.
     *
     * For example, the Amazon S3 actions `GetObject` and `GetBucketLocation` have the Read access level.
     */
    allReadActions() {
        return this.addAccessLevel(access_level_1.AccessLevel.READ);
    }
    /**
     * Adds all actions with [access level](https://docs.aws.amazon.com/IAM/latest/UserGuide/access_policies_understand-policy-summary-access-level-summaries.html#access_policies_access-level) WRITE to the statement
     *
     * Permission to create, delete, or modify resources in the service.
     *
     * For example, the Amazon S3 actions `CreateBucket`, `DeleteBucket` and `PutObject` have the Write access level.
     *
     * Write actions might also allow modifying a resource tag. However, an action that allows only changes to tags has the Tagging access level.
     */
    allWriteActions() {
        return this.addAccessLevel(access_level_1.AccessLevel.WRITE);
    }
    /**
     * Adds all actions with [access level](https://docs.aws.amazon.com/IAM/latest/UserGuide/access_policies_understand-policy-summary-access-level-summaries.html#access_policies_access-level) PERMISSION MANAGEMENT to the statement
     *
     * Permission to grant or modify resource permissions in the service.
     *
     * For example, most IAM and AWS Organizations actions, as well as actions like the Amazon S3 actions `PutBucketPolicy` and `DeleteBucketPolicy` have the Permissions management access level.
     */
    allPermissionManagementActions() {
        return this.addAccessLevel(access_level_1.AccessLevel.PERMISSION_MANAGEMENT);
    }
    /**
     * Adds all actions with [access level](https://docs.aws.amazon.com/IAM/latest/UserGuide/access_policies_understand-policy-summary-access-level-summaries.html#access_policies_access-level) TAGGING to the statement
     *
     * Permission to perform actions that only change the state of resource tags.
     *
     * For example, the IAM actions `TagRole` and `UntagRole` have the Tagging access level because they allow only tagging or untagging a role. However, the `CreateRole` action allows tagging a role resource when you create that role. Because the action does not only add a tag, it has the Write access level.
     */
    allTaggingActions() {
        return this.addAccessLevel(access_level_1.AccessLevel.TAGGING);
    }
    addAccessLevel(accessLevel) {
        if (accessLevel in this.accessLevelList) {
            this.accessLevelList[accessLevel].forEach((action) => {
                this.to(`${this.servicePrefix}:${action}`);
            });
        }
        return this;
    }
    /**
     * Condense action list down to a list of patterns.
     *
     * Using this method can help to reduce the policy size.
     *
     * For example, all actions with access level `list` could be reduced to a small pattern `List*`.
     */
    compact() {
        this.isCompact = true;
        return this;
    }
    compactActions() {
        // actions that will be included, service prefix is removed
        const includeActions = this.floydActions.map((elem) => {
            return elem.substr(elem.indexOf(':') + 1);
        });
        // actions that will not be included, the opposite of includeActions
        const excludeActions = [];
        for (const [_, actions] of Object.entries(this.accessLevelList)) {
            actions.forEach((action) => {
                if (!includeActions.includes(action)) {
                    excludeActions.push(`^${action}$`);
                }
            });
        }
        // will contain the index of elements that are covered by substrings
        let covered = [];
        const subs = common_substrings_1.default(includeActions.map((action) => {
            return `^${action}$`;
        }), {
            minLength: 3,
            minOccurrence: 2,
        })
            .filter((sub) => {
            // remove all substrings, that match an action we have not selected
            for (let action of excludeActions) {
                if (action.includes(sub.name)) {
                    return false;
                }
            }
            return true;
        })
            .sort((a, b) => {
            // sort list by the number of matches
            if (a.source.length < b.source.length)
                return 1;
            if (a.source.length > b.source.length)
                return -1;
            return 0;
        })
            .filter((sub) => {
            // removes substrings, that have already been covered by other substrings
            const sources = sub.source.filter((source) => {
                return !covered.includes(source);
            });
            if (sources.length > 1) {
                //add list of sources to the global list of covered actions
                covered = covered.concat(sources);
                return true;
            }
            return false;
        });
        // stores the list of patterns
        const compactActionList = [];
        subs.forEach((sub) => {
            compactActionList.push(`${this.servicePrefix}:*${sub.name}*`
                .replace('*^', '')
                .replace('$*', ''));
            sub.source.forEach((source) => {
                includeActions[source] = ''; // invalidate, will be filtered later
            });
        });
        includeActions
            .filter((action) => {
            // remove elements that have been covered by patterns, we invalidated them above
            return action.length > 0;
        })
            .forEach((action) => {
            // add actions that have not been covered by patterns to the new action list
            compactActionList.push(`${this.servicePrefix}:${action}`);
        });
        // we're done. override action list
        this.floydActions = compactActionList;
    }
}
exports.PolicyStatementWithActions = PolicyStatementWithActions;
_a = JSII_RTTI_SYMBOL_1;
PolicyStatementWithActions[_a] = { fqn: "iam-floyd.PolicyStatementWithActions", version: "0.502.0" };
//# sourceMappingURL=data:application/json;base64,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