"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.Metaflow = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const path = require("path");
const batch = require("@aws-cdk/aws-batch");
const ec2 = require("@aws-cdk/aws-ec2");
const ecs = require("@aws-cdk/aws-ecs");
const events = require("@aws-cdk/aws-events");
const iam = require("@aws-cdk/aws-iam");
const lambda = require("@aws-cdk/aws-lambda");
const logs = require("@aws-cdk/aws-logs");
const cdk = require("@aws-cdk/core");
const constructs_1 = require("./constructs");
const constants_1 = require("./constructs/constants");
/**
 * @experimental
 */
class Metaflow extends cdk.Construct {
    /**
     * @experimental
     */
    constructor(scope, id) {
        super(scope, id);
        // Network
        const vpc = new constructs_1.MetaflowVpc(this, 'vpc');
        this.vpc = vpc;
        const serviceSecurityGroup = new ec2.SecurityGroup(this, 'fargate-sg', {
            allowAllOutbound: true,
            vpc: this.vpc,
        });
        const databaseSecurityGroup = new ec2.SecurityGroup(this, 'rds-sg', {
            allowAllOutbound: true,
            vpc: this.vpc,
        });
        databaseSecurityGroup.connections.allowFrom(serviceSecurityGroup, ec2.Port.tcp(5432));
        serviceSecurityGroup.connections.allowInternally(ec2.Port.allTraffic(), 'Internal Communication');
        serviceSecurityGroup.connections.allowFrom(ec2.Peer.ipv4(this.vpc.vpcCidrBlock), ec2.Port.tcp(8080), 'Allow API Calls Internally');
        serviceSecurityGroup.connections.allowFrom(ec2.Peer.ipv4(this.vpc.vpcCidrBlock), ec2.Port.tcp(8082), 'Allow API Calls Internally');
        // Persistence
        this.bucket = new constructs_1.MetaflowBucket(this, 'bucket');
        this.table = new constructs_1.MetaflowTable(this, 'table');
        this.eventBus = new events.EventBus(this, 'event-bus');
        this.database = new constructs_1.MetaflowDatabaseInstance(this, 'database', {
            vpc: this.vpc,
            dbSecurityGroups: [databaseSecurityGroup],
            dbSubnets: {
                subnetType: ec2.SubnetType.PUBLIC,
            },
        });
        const logGroup = new logs.LogGroup(this, 'ecs-log-group', {
            logGroupName: `/ecs/${cdk.Aws.STACK_NAME}-${constants_1.ServiceInfo.SERVICE_NAME}`,
            retention: logs.RetentionDays.ONE_MONTH,
            removalPolicy: cdk.RemovalPolicy.DESTROY,
        });
        // iam
        this.ecsExecutionRole = new constructs_1.EcsExecutionRole(this, 'ecs-execution-role');
        this.ecsTaskRole = new constructs_1.EcsTaskRole(this, 'ecs-task-role');
        this.lambdaECSExecuteRole = new constructs_1.LambdaECSExecuteRole(this, 'lambda-ecs-execution-role');
        this.batchExecutionRole = new constructs_1.BatchExecutionRole(this, 'batch-execution-role');
        this.batchExecutionRole.grantPassRole(new iam.ServicePrincipal('ecs-tasks.amazonaws.com'));
        this.batchExecutionRole.grantPassRole(new iam.ServicePrincipal('ec2.amazonaws.com.cn'));
        this.batchExecutionRole.grantPassRole(new iam.ServicePrincipal('ec2.amazonaws.com'));
        this.ecsRole = new constructs_1.EcsRole(this, 'ecs-role');
        this.stepFunctionsRole = new constructs_1.StepFunctionsRole(this, 'step-functions-role');
        this.eventBridgeRole = new constructs_1.EventBridgeRole(this, 'event-bridge-role');
        this.batchS3TaskRole = new constructs_1.BatchS3TaskRole(this, 'batch-s3-task-role');
        // Ecs
        this.cluster = new ecs.Cluster(this, 'metaflow-cluster', {
            enableFargateCapacityProviders: true,
            containerInsights: true,
            vpc: this.vpc,
        });
        const metaflowFargate = new constructs_1.MetaflowFargateService(this, 'fargate-service', {
            logGroup: logGroup,
            secret: this.database.credentials,
            securityGroup: serviceSecurityGroup,
            executionRole: this.ecsExecutionRole,
            taskRole: this.ecsTaskRole,
            cluster: this.cluster,
            database: this.database.database,
        });
        // Nlb
        const metaflowNlb = new constructs_1.MetaflowNlb(this, 'nlb', {
            vpc: this.vpc,
        });
        const nlbTarget = metaflowFargate.fargateService.loadBalancerTarget({
            containerName: constants_1.ServiceInfo.SERVICE_NAME,
            protocol: ecs.Protocol.TCP,
            containerPort: 8080,
        });
        const dbMigrateTarget = metaflowFargate.fargateService.loadBalancerTarget({
            containerName: constants_1.ServiceInfo.SERVICE_NAME,
            protocol: ecs.Protocol.TCP,
            containerPort: 8082,
        });
        metaflowNlb.nlbTargetGroup.addTarget(nlbTarget);
        metaflowNlb.dbMigrateTargetGroup.addTarget(dbMigrateTarget);
        // Lambda
        this.dbMigrateLambda = new lambda.Function(this, 'db-migrate-handler', {
            runtime: lambda.Runtime.PYTHON_3_8,
            description: 'Trigger DB Migration',
            functionName: 'migrate-db',
            vpc: vpc,
            securityGroups: [serviceSecurityGroup],
            allowPublicSubnet: true,
            code: lambda.Code.fromAsset(path.join(__dirname, '../assets/lambda/db-migrate')),
            role: this.lambdaECSExecuteRole,
            handler: 'index.handler',
            environment: {
                MD_LB_ADDRESS: `http://${metaflowNlb.nlb.loadBalancerDnsName}:8082`,
            },
        });
        // API Gateway
        const api = new constructs_1.MetaflowApi(this, 'api', {
            nlb: metaflowNlb.nlb,
        });
        this.api = api.api;
        // Batch
        const computeEnvironment = new batch.CfnComputeEnvironment(this, 'ComputeEnvironment', {
            type: 'MANAGED',
            serviceRole: this.batchExecutionRole.roleArn,
            computeResources: {
                maxvCpus: 90,
                type: 'FARGATE',
                securityGroupIds: [vpc.vpcDefaultSecurityGroup],
                subnets: vpc.publicSubnets.map((subnet) => subnet.subnetId),
            },
            state: 'ENABLED',
        });
        const jobQueue = new batch.CfnJobQueue(this, 'JobQueue', {
            computeEnvironmentOrder: [
                {
                    order: 1,
                    computeEnvironment: computeEnvironment.ref,
                },
            ],
            state: 'ENABLED',
            priority: 1,
            jobQueueName: 'jobs',
        });
        jobQueue.addDependsOn(computeEnvironment);
        // Permissions
        this.bucket.grantReadWrite(this.ecsTaskRole);
        this.bucket.grantReadWrite(this.stepFunctionsRole);
        this.bucket.grantReadWrite(this.batchS3TaskRole);
        this.bucket.grantReadWrite(this.ecsRole);
        this.table.grantReadWriteData(this.stepFunctionsRole);
        this.table.grantReadWriteData(this.batchS3TaskRole);
        this.eventBus.grantPutEventsTo(this.stepFunctionsRole);
        logGroup.grantWrite(this.ecsTaskRole);
        this.database.credentials.grantRead(this.ecsExecutionRole);
        // Cloudwatch
        new constructs_1.MetaflowDashboard(this, 'dashboard', {
            dashboardName: 'MetaflowDashboard',
            bucketName: this.bucket.bucketName,
            ecsService: metaflowFargate.fargateService,
            period: 15,
        });
        // Outputs
        new constructs_1.MetaflowExports(this, 'metaflow-exports', {
            bucketName: this.bucket.bucketName,
            tableName: this.table.tableName,
            nlbDnsName: metaflowNlb.nlb.loadBalancerDnsName,
            migrateLambdaName: this.dbMigrateLambda.functionName,
            batchS3TaskRoleArn: this.batchS3TaskRole.roleArn,
            batchExecutionRoleArn: this.batchExecutionRole.roleArn,
            stepFunctionsRoleArn: this.stepFunctionsRole.roleArn,
            eventBridgeRoleArn: this.eventBridgeRole.roleArn,
            ecsTaskRoleArn: this.ecsTaskRole.roleArn,
            ecsRoleArn: this.ecsRole.roleArn,
            ecsExecutionRoleArn: this.ecsExecutionRole.roleArn,
            lambdaECSExecuteRoleArn: this.lambdaECSExecuteRole.roleArn,
        });
    }
}
exports.Metaflow = Metaflow;
_a = JSII_RTTI_SYMBOL_1;
Metaflow[_a] = { fqn: "cdk-metaflow.Metaflow", version: "0.0.12" };
//# sourceMappingURL=data:application/json;base64,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