"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.DynamoDBModelTransformer = exports.directiveDefinition = exports.CONDITIONS_MINIMUM_VERSION = void 0;
const cloudform_types_1 = require("cloudform-types");
const graphql_transformer_common_1 = require("graphql-transformer-common");
const graphql_transformer_core_1 = require("graphql-transformer-core");
const definitions_1 = require("./definitions");
const ModelDirectiveArgs_1 = require("./ModelDirectiveArgs");
const resources_1 = require("./resources");
const METADATA_KEY = 'DynamoDBTransformerMetadata';
exports.CONDITIONS_MINIMUM_VERSION = 5;
exports.directiveDefinition = graphql_transformer_core_1.gql `
  directive @model(
    queries: ModelQueryMap
    mutations: ModelMutationMap
    subscriptions: ModelSubscriptionMap
    timestamps: TimestampConfiguration
  ) on OBJECT
  input ModelMutationMap {
    create: String
    update: String
    delete: String
  }
  input ModelQueryMap {
    get: String
    list: String
  }
  input ModelSubscriptionMap {
    onCreate: [String]
    onUpdate: [String]
    onDelete: [String]
    level: ModelSubscriptionLevel
  }
  enum ModelSubscriptionLevel {
    off
    public
    on
  }
  input TimestampConfiguration {
    createdAt: String
    updatedAt: String
  }
`;
class DynamoDBModelTransformer extends graphql_transformer_core_1.Transformer {
    constructor(opts = {}) {
        super('DynamoDBModelTransformer', exports.directiveDefinition);
        this.before = (ctx) => {
            const template = this.resources.initTemplate();
            ctx.mergeResources(template.Resources);
            ctx.mergeParameters(template.Parameters);
            ctx.mergeOutputs(template.Outputs);
            ctx.mergeConditions(template.Conditions);
        };
        this.after = (ctx) => {
            const ddbMetata = ctx.metadata.get(METADATA_KEY);
            if (ddbMetata) {
                Object.entries(ddbMetata.hoistedRequestMappingContent || {}).forEach(([resourceId, hoistedContentGenerator]) => {
                    const hoistedContent = hoistedContentGenerator();
                    if (hoistedContent) {
                        const resource = ctx.getResource(resourceId);
                        resource.Properties.RequestMappingTemplate = [hoistedContent, resource.Properties.RequestMappingTemplate].join('\n');
                        ctx.setResource(resourceId, resource);
                    }
                });
            }
        };
        this.object = (def, directive, ctx) => {
            const isTypeNameReserved = def.name.value === ctx.getQueryTypeName() ||
                def.name.value === ctx.getMutationTypeName() ||
                def.name.value === ctx.getSubscriptionTypeName();
            if (isTypeNameReserved && ctx.featureFlags.getBoolean('validateTypeNameReservedWords', true)) {
                throw new graphql_transformer_core_1.InvalidDirectiveError(`'${def.name.value}' is a reserved type name and currently in use within the default schema element.`);
            }
            const stackName = def.name.value;
            const nonModelArray = definitions_1.getNonModelObjectArray(def, ctx, new Map());
            nonModelArray.forEach((value) => {
                const nonModelObject = definitions_1.makeNonModelInputObject(value, nonModelArray, ctx);
                if (!this.typeExist(nonModelObject.name.value, ctx)) {
                    ctx.addInput(nonModelObject);
                }
            });
            this.addIdField(def, directive, ctx);
            const typeName = def.name.value;
            this.setSyncConfig(ctx, typeName);
            const isSyncEnabled = this.opts.SyncConfig ? true : false;
            const tableLogicalID = graphql_transformer_common_1.ModelResourceIDs.ModelTableResourceID(typeName);
            const iamRoleLogicalID = graphql_transformer_common_1.ModelResourceIDs.ModelTableIAMRoleID(typeName);
            const dataSourceRoleLogicalID = graphql_transformer_common_1.ModelResourceIDs.ModelTableDataSourceID(typeName);
            const deletionPolicy = this.opts.EnableDeletionProtection ? cloudform_types_1.DeletionPolicy.Retain : cloudform_types_1.DeletionPolicy.Delete;
            ctx.setResource(tableLogicalID, this.resources.makeModelTable(typeName, undefined, undefined, deletionPolicy, isSyncEnabled));
            ctx.mapResourceToStack(stackName, tableLogicalID);
            ctx.setResource(iamRoleLogicalID, this.resources.makeIAMRole(typeName, this.opts.SyncConfig));
            ctx.mapResourceToStack(stackName, iamRoleLogicalID);
            ctx.setResource(dataSourceRoleLogicalID, this.resources.makeDynamoDBDataSource(tableLogicalID, iamRoleLogicalID, typeName, isSyncEnabled));
            ctx.mapResourceToStack(stackName, dataSourceRoleLogicalID);
            const streamArnOutputId = `GetAtt${graphql_transformer_common_1.ModelResourceIDs.ModelTableStreamArn(typeName)}`;
            ctx.setOutput(streamArnOutputId, this.resources.makeTableStreamArnOutput(tableLogicalID));
            ctx.mapResourceToStack(stackName, streamArnOutputId);
            const datasourceOutputId = `GetAtt${dataSourceRoleLogicalID}Name`;
            ctx.setOutput(datasourceOutputId, this.resources.makeDataSourceOutput(dataSourceRoleLogicalID));
            ctx.mapResourceToStack(stackName, datasourceOutputId);
            const tableNameOutputId = `GetAtt${tableLogicalID}Name`;
            ctx.setOutput(tableNameOutputId, this.resources.makeTableNameOutput(tableLogicalID));
            ctx.mapResourceToStack(stackName, tableNameOutputId);
            this.createQueries(def, directive, ctx);
            this.createMutations(def, directive, ctx, nonModelArray);
            this.createSubscriptions(def, directive, ctx);
            this.updateMutationConditionInput(ctx, def);
            if (isSyncEnabled) {
                const obj = ctx.getObject(def.name.value);
                const newFields = [
                    ...obj.fields,
                    graphql_transformer_common_1.makeField('_version', [], graphql_transformer_common_1.wrapNonNull(graphql_transformer_common_1.makeNamedType('Int'))),
                    graphql_transformer_common_1.makeField('_deleted', [], graphql_transformer_common_1.makeNamedType('Boolean')),
                    graphql_transformer_common_1.makeField('_lastChangedAt', [], graphql_transformer_common_1.wrapNonNull(graphql_transformer_common_1.makeNamedType('AWSTimestamp'))),
                ];
                const newObj = {
                    ...obj,
                    fields: newFields,
                };
                ctx.updateObject(newObj);
            }
            this.addTimestampFields(def, directive, ctx);
        };
        this.createMutations = (def, directive, ctx, nonModelArray) => {
            const typeName = def.name.value;
            const isSyncEnabled = this.opts.SyncConfig ? true : false;
            const mutationFields = [];
            const directiveArguments = graphql_transformer_core_1.getDirectiveArguments(directive);
            let shouldMakeCreate = true;
            let shouldMakeUpdate = true;
            let shouldMakeDelete = true;
            let createFieldNameOverride = undefined;
            let updateFieldNameOverride = undefined;
            let deleteFieldNameOverride = undefined;
            const createdAtField = ModelDirectiveArgs_1.getCreatedAtFieldName(directive);
            const updatedAtField = ModelDirectiveArgs_1.getUpdatedAtFieldName(directive);
            const existingCreatedAtField = def.fields.find(f => f.name.value === createdAtField);
            const existingUpdatedAtField = def.fields.find(f => f.name.value === updatedAtField);
            const timestampFields = {
                createdAtField: DynamoDBModelTransformer.isTimestampCompatibleField(existingCreatedAtField) ? createdAtField : undefined,
                updatedAtField: DynamoDBModelTransformer.isTimestampCompatibleField(existingUpdatedAtField) ? updatedAtField : undefined,
            };
            if (directiveArguments.mutations === null) {
                shouldMakeCreate = false;
                shouldMakeUpdate = false;
                shouldMakeDelete = false;
            }
            else if (directiveArguments.mutations) {
                if (!directiveArguments.mutations.create) {
                    shouldMakeCreate = false;
                }
                else {
                    createFieldNameOverride = directiveArguments.mutations.create;
                }
                if (!directiveArguments.mutations.update) {
                    shouldMakeUpdate = false;
                }
                else {
                    updateFieldNameOverride = directiveArguments.mutations.update;
                }
                if (!directiveArguments.mutations.delete) {
                    shouldMakeDelete = false;
                }
                else {
                    deleteFieldNameOverride = directiveArguments.mutations.delete;
                }
            }
            const conditionInputName = graphql_transformer_common_1.ModelResourceIDs.ModelConditionInputTypeName(typeName);
            if (shouldMakeCreate) {
                const createInput = definitions_1.makeCreateInputObject(def, directive, nonModelArray, ctx, isSyncEnabled);
                if (!ctx.getType(createInput.name.value)) {
                    ctx.addInput(createInput);
                }
                const createResolver = this.resources.makeCreateResolver({
                    type: def.name.value,
                    nameOverride: createFieldNameOverride,
                    syncConfig: this.opts.SyncConfig,
                });
                const resourceId = graphql_transformer_common_1.ResolverResourceIDs.DynamoDBCreateResolverResourceID(typeName);
                this.addInitalizationMetadata(ctx, resourceId, () => {
                    const inputObj = ctx.getType(createInput.name.value);
                    if (inputObj) {
                        return this.resources.initalizeDefaultInputForCreateMutation(inputObj, timestampFields);
                    }
                });
                ctx.setResource(resourceId, createResolver);
                ctx.mapResourceToStack(typeName, resourceId);
                const args = [graphql_transformer_common_1.makeInputValueDefinition('input', graphql_transformer_common_1.makeNonNullType(graphql_transformer_common_1.makeNamedType(createInput.name.value)))];
                if (this.supportsConditions(ctx)) {
                    args.push(graphql_transformer_common_1.makeInputValueDefinition('condition', graphql_transformer_common_1.makeNamedType(conditionInputName)));
                }
                mutationFields.push(graphql_transformer_common_1.makeField(createResolver.Properties.FieldName.toString(), args, graphql_transformer_common_1.makeNamedType(def.name.value)));
            }
            if (shouldMakeUpdate) {
                const updateInput = definitions_1.makeUpdateInputObject(def, nonModelArray, ctx, isSyncEnabled);
                const optionalNonNullableFields = definitions_1.getFieldsOptionalNonNullableField(updateInput.fields.map(r => r), def);
                if (!ctx.getType(updateInput.name.value)) {
                    ctx.addInput(updateInput);
                }
                const updateResolver = this.resources.makeUpdateResolver({
                    type: def.name.value,
                    nameOverride: updateFieldNameOverride,
                    syncConfig: this.opts.SyncConfig,
                    timestamps: timestampFields,
                    optionalNonNullableFields,
                });
                const resourceId = graphql_transformer_common_1.ResolverResourceIDs.DynamoDBUpdateResolverResourceID(typeName);
                ctx.setResource(resourceId, updateResolver);
                ctx.mapResourceToStack(typeName, resourceId);
                const args = [graphql_transformer_common_1.makeInputValueDefinition('input', graphql_transformer_common_1.makeNonNullType(graphql_transformer_common_1.makeNamedType(updateInput.name.value)))];
                if (this.supportsConditions(ctx)) {
                    args.push(graphql_transformer_common_1.makeInputValueDefinition('condition', graphql_transformer_common_1.makeNamedType(conditionInputName)));
                }
                mutationFields.push(graphql_transformer_common_1.makeField(updateResolver.Properties.FieldName.toString(), args, graphql_transformer_common_1.makeNamedType(def.name.value)));
            }
            if (shouldMakeDelete) {
                const deleteInput = definitions_1.makeDeleteInputObject(def, isSyncEnabled);
                if (!ctx.getType(deleteInput.name.value)) {
                    ctx.addInput(deleteInput);
                }
                const deleteResolver = this.resources.makeDeleteResolver({
                    type: def.name.value,
                    nameOverride: deleteFieldNameOverride,
                    syncConfig: this.opts.SyncConfig,
                });
                const resourceId = graphql_transformer_common_1.ResolverResourceIDs.DynamoDBDeleteResolverResourceID(typeName);
                ctx.setResource(resourceId, deleteResolver);
                ctx.mapResourceToStack(typeName, resourceId);
                const args = [graphql_transformer_common_1.makeInputValueDefinition('input', graphql_transformer_common_1.makeNonNullType(graphql_transformer_common_1.makeNamedType(deleteInput.name.value)))];
                if (this.supportsConditions(ctx)) {
                    args.push(graphql_transformer_common_1.makeInputValueDefinition('condition', graphql_transformer_common_1.makeNamedType(conditionInputName)));
                }
                mutationFields.push(graphql_transformer_common_1.makeField(deleteResolver.Properties.FieldName.toString(), args, graphql_transformer_common_1.makeNamedType(def.name.value)));
            }
            ctx.addMutationFields(mutationFields);
            if (shouldMakeCreate || shouldMakeUpdate || shouldMakeDelete) {
                this.generateConditionInputs(ctx, def);
            }
        };
        this.createQueries = (def, directive, ctx) => {
            const typeName = def.name.value;
            const queryFields = [];
            const directiveArguments = graphql_transformer_core_1.getDirectiveArguments(directive);
            let shouldMakeGet = true;
            let shouldMakeList = true;
            let getFieldNameOverride = undefined;
            let listFieldNameOverride = undefined;
            const isSyncEnabled = this.opts.SyncConfig ? true : false;
            if (directiveArguments.queries === null) {
                shouldMakeGet = false;
                shouldMakeList = false;
            }
            else if (directiveArguments.queries) {
                if (!directiveArguments.queries.get) {
                    shouldMakeGet = false;
                }
                else {
                    getFieldNameOverride = directiveArguments.queries.get;
                }
                if (!directiveArguments.queries.list) {
                    shouldMakeList = false;
                }
                else {
                    listFieldNameOverride = directiveArguments.queries.list;
                }
            }
            if (shouldMakeList) {
                if (!this.typeExist('ModelSortDirection', ctx)) {
                    const tableSortDirection = definitions_1.makeModelSortDirectionEnumObject();
                    ctx.addEnum(tableSortDirection);
                }
            }
            if (isSyncEnabled) {
                const syncResolver = this.resources.makeSyncResolver(typeName);
                const syncResourceID = graphql_transformer_common_1.ResolverResourceIDs.SyncResolverResourceID(typeName);
                ctx.setResource(syncResourceID, syncResolver);
                ctx.mapResourceToStack(typeName, syncResourceID);
                this.generateModelXConnectionType(ctx, def, isSyncEnabled);
                this.generateFilterInputs(ctx, def);
                queryFields.push(graphql_transformer_common_1.makeField(syncResolver.Properties.FieldName.toString(), [
                    graphql_transformer_common_1.makeInputValueDefinition('filter', graphql_transformer_common_1.makeNamedType(graphql_transformer_common_1.ModelResourceIDs.ModelFilterInputTypeName(def.name.value))),
                    graphql_transformer_common_1.makeInputValueDefinition('limit', graphql_transformer_common_1.makeNamedType('Int')),
                    graphql_transformer_common_1.makeInputValueDefinition('nextToken', graphql_transformer_common_1.makeNamedType('String')),
                    graphql_transformer_common_1.makeInputValueDefinition('lastSync', graphql_transformer_common_1.makeNamedType('AWSTimestamp')),
                ], graphql_transformer_common_1.makeNamedType(graphql_transformer_common_1.ModelResourceIDs.ModelConnectionTypeName(def.name.value))));
            }
            if (shouldMakeGet) {
                const getResolver = this.resources.makeGetResolver(def.name.value, getFieldNameOverride, isSyncEnabled, ctx.getQueryTypeName());
                const resourceId = graphql_transformer_common_1.ResolverResourceIDs.DynamoDBGetResolverResourceID(typeName);
                ctx.setResource(resourceId, getResolver);
                ctx.mapResourceToStack(typeName, resourceId);
                queryFields.push(graphql_transformer_common_1.makeField(getResolver.Properties.FieldName.toString(), [graphql_transformer_common_1.makeInputValueDefinition('id', graphql_transformer_common_1.makeNonNullType(graphql_transformer_common_1.makeNamedType('ID')))], graphql_transformer_common_1.makeNamedType(def.name.value)));
            }
            if (shouldMakeList) {
                this.generateModelXConnectionType(ctx, def);
                const listResolver = this.resources.makeListResolver(def.name.value, ctx.featureFlags.getBoolean('improvePluralization'), listFieldNameOverride, isSyncEnabled, ctx.getQueryTypeName());
                const resourceId = graphql_transformer_common_1.ResolverResourceIDs.DynamoDBListResolverResourceID(typeName);
                ctx.setResource(resourceId, listResolver);
                ctx.mapResourceToStack(typeName, resourceId);
                queryFields.push(graphql_transformer_common_1.makeConnectionField(listResolver.Properties.FieldName.toString(), def.name.value));
                this.generateFilterInputs(ctx, def);
            }
            ctx.addQueryFields(queryFields);
        };
        this.createSubscriptions = (def, directive, ctx) => {
            const typeName = def.name.value;
            const subscriptionFields = [];
            const directiveArguments = graphql_transformer_core_1.getDirectiveArguments(directive);
            const subscriptionsArgument = directiveArguments.subscriptions;
            const createResolver = ctx.getResource(graphql_transformer_common_1.ResolverResourceIDs.DynamoDBCreateResolverResourceID(typeName));
            const updateResolver = ctx.getResource(graphql_transformer_common_1.ResolverResourceIDs.DynamoDBUpdateResolverResourceID(typeName));
            const deleteResolver = ctx.getResource(graphql_transformer_common_1.ResolverResourceIDs.DynamoDBDeleteResolverResourceID(typeName));
            if (subscriptionsArgument === null) {
                return;
            }
            if (subscriptionsArgument && subscriptionsArgument.level === 'off') {
                return;
            }
            if (subscriptionsArgument && (subscriptionsArgument.onCreate || subscriptionsArgument.onUpdate || subscriptionsArgument.onDelete)) {
                const subscriptionToMutationsMap = {};
                const onCreate = subscriptionsArgument.onCreate || [];
                const onUpdate = subscriptionsArgument.onUpdate || [];
                const onDelete = subscriptionsArgument.onDelete || [];
                const subFields = [...onCreate, ...onUpdate, ...onDelete];
                for (const field of subFields) {
                    subscriptionToMutationsMap[field] = [];
                }
                for (const field of Object.keys(subscriptionToMutationsMap)) {
                    if (onCreate.includes(field) && createResolver) {
                        subscriptionToMutationsMap[field].push(createResolver.Properties.FieldName);
                    }
                    if (onUpdate.includes(field) && updateResolver) {
                        subscriptionToMutationsMap[field].push(updateResolver.Properties.FieldName);
                    }
                    if (onDelete.includes(field) && deleteResolver) {
                        subscriptionToMutationsMap[field].push(deleteResolver.Properties.FieldName);
                    }
                }
                for (const subFieldName of Object.keys(subscriptionToMutationsMap)) {
                    const subField = definitions_1.makeSubscriptionField(subFieldName, typeName, subscriptionToMutationsMap[subFieldName]);
                    subscriptionFields.push(subField);
                }
            }
            else {
                if (createResolver) {
                    const onCreateField = definitions_1.makeSubscriptionField(graphql_transformer_common_1.ModelResourceIDs.ModelOnCreateSubscriptionName(typeName), typeName, [
                        createResolver.Properties.FieldName,
                    ]);
                    subscriptionFields.push(onCreateField);
                }
                if (updateResolver) {
                    const onUpdateField = definitions_1.makeSubscriptionField(graphql_transformer_common_1.ModelResourceIDs.ModelOnUpdateSubscriptionName(typeName), typeName, [
                        updateResolver.Properties.FieldName,
                    ]);
                    subscriptionFields.push(onUpdateField);
                }
                if (deleteResolver) {
                    const onDeleteField = definitions_1.makeSubscriptionField(graphql_transformer_common_1.ModelResourceIDs.ModelOnDeleteSubscriptionName(typeName), typeName, [
                        deleteResolver.Properties.FieldName,
                    ]);
                    subscriptionFields.push(onDeleteField);
                }
            }
            ctx.addSubscriptionFields(subscriptionFields);
        };
        this.opts = this.getOpts(opts);
        this.resources = new resources_1.ResourceFactory();
    }
    addTimestampFields(def, directive, ctx) {
        const createdAtField = ModelDirectiveArgs_1.getCreatedAtFieldName(directive);
        const updatedAtField = ModelDirectiveArgs_1.getUpdatedAtFieldName(directive);
        const existingCreatedAtField = def.fields.find(f => f.name.value === createdAtField);
        const existingUpdatedAtField = def.fields.find(f => f.name.value === updatedAtField);
        if (!DynamoDBModelTransformer.isTimestampCompatibleField(existingCreatedAtField)) {
            console.log(`${def.name.value}.${existingCreatedAtField.name.value} is of type ${graphql_transformer_common_1.getBaseType(existingCreatedAtField.type)}. To support auto population change the type to AWSDateTime or String`);
        }
        if (!DynamoDBModelTransformer.isTimestampCompatibleField(existingUpdatedAtField)) {
            console.log(`${def.name.value}.${existingUpdatedAtField.name.value} is of type ${graphql_transformer_common_1.getBaseType(existingUpdatedAtField.type)}. To support auto population change the type to AWSDateTime or String`);
        }
        const obj = ctx.getObject(def.name.value);
        const newObj = {
            ...obj,
            fields: [
                ...obj.fields,
                ...(createdAtField && !existingCreatedAtField ? [graphql_transformer_common_1.makeField(createdAtField, [], graphql_transformer_common_1.wrapNonNull(graphql_transformer_common_1.makeNamedType('AWSDateTime')))] : []),
                ...(updatedAtField && !existingUpdatedAtField ? [graphql_transformer_common_1.makeField(updatedAtField, [], graphql_transformer_common_1.wrapNonNull(graphql_transformer_common_1.makeNamedType('AWSDateTime')))] : []),
            ],
        };
        ctx.updateObject(newObj);
    }
    addIdField(def, directive, ctx) {
        const hasIdField = def.fields.find(f => f.name.value === 'id');
        if (!hasIdField) {
            const obj = ctx.getObject(def.name.value);
            const newObj = {
                ...obj,
                fields: [graphql_transformer_common_1.makeField('id', [], graphql_transformer_common_1.wrapNonNull(graphql_transformer_common_1.makeNamedType('ID'))), ...obj.fields],
            };
            ctx.updateObject(newObj);
        }
    }
    typeExist(type, ctx) {
        return Boolean(type in ctx.nodeMap);
    }
    generateModelXConnectionType(ctx, def, isSync = false) {
        const tableXConnectionName = graphql_transformer_common_1.ModelResourceIDs.ModelConnectionTypeName(def.name.value);
        if (this.typeExist(tableXConnectionName, ctx)) {
            return;
        }
        const connectionType = graphql_transformer_common_1.blankObject(tableXConnectionName);
        ctx.addObject(connectionType);
        ctx.addObjectExtension(definitions_1.makeModelConnectionType(def.name.value, isSync));
    }
    generateFilterInputs(ctx, def) {
        const scalarFilters = definitions_1.makeScalarFilterInputs(this.supportsConditions(ctx));
        for (const filter of scalarFilters) {
            if (!this.typeExist(filter.name.value, ctx)) {
                ctx.addInput(filter);
            }
        }
        const enumFilters = definitions_1.makeEnumFilterInputObjects(def, ctx, this.supportsConditions(ctx));
        for (const filter of enumFilters) {
            if (!this.typeExist(filter.name.value, ctx)) {
                ctx.addInput(filter);
            }
        }
        const tableXQueryFilterInput = definitions_1.makeModelXFilterInputObject(def, ctx, this.supportsConditions(ctx));
        if (!this.typeExist(tableXQueryFilterInput.name.value, ctx)) {
            ctx.addInput(tableXQueryFilterInput);
        }
        if (this.supportsConditions(ctx)) {
            const attributeTypeEnum = definitions_1.makeAttributeTypeEnum();
            if (!this.typeExist(attributeTypeEnum.name.value, ctx)) {
                ctx.addType(attributeTypeEnum);
            }
        }
    }
    generateConditionInputs(ctx, def) {
        const scalarFilters = definitions_1.makeScalarFilterInputs(this.supportsConditions(ctx));
        for (const filter of scalarFilters) {
            if (!this.typeExist(filter.name.value, ctx)) {
                ctx.addInput(filter);
            }
        }
        const enumFilters = definitions_1.makeEnumFilterInputObjects(def, ctx, this.supportsConditions(ctx));
        for (const filter of enumFilters) {
            if (!this.typeExist(filter.name.value, ctx)) {
                ctx.addInput(filter);
            }
        }
        if (this.supportsConditions(ctx)) {
            const tableXMutationConditionInput = definitions_1.makeModelXConditionInputObject(def, ctx, this.supportsConditions(ctx));
            if (!this.typeExist(tableXMutationConditionInput.name.value, ctx)) {
                ctx.addInput(tableXMutationConditionInput);
            }
            const attributeTypeEnum = definitions_1.makeAttributeTypeEnum();
            if (!this.typeExist(attributeTypeEnum.name.value, ctx)) {
                ctx.addType(attributeTypeEnum);
            }
        }
    }
    getOpts(opts) {
        const defaultOpts = {
            EnableDeletionProtection: false,
        };
        return {
            ...defaultOpts,
            ...opts,
        };
    }
    setSyncConfig(ctx, typeName) {
        let syncConfig;
        const resolverConfig = ctx.getResolverConfig();
        if (resolverConfig && resolverConfig.project) {
            syncConfig = resolverConfig.project;
        }
        if (resolverConfig && resolverConfig.models && resolverConfig.models[typeName]) {
            const typeResolverConfig = resolverConfig.models[typeName];
            if (typeResolverConfig.ConflictDetection && typeResolverConfig.ConflictHandler) {
                syncConfig = typeResolverConfig;
            }
            else {
                console.warn(`Invalid resolverConfig for type ${typeName}. Using the project resolverConfig instead.`);
            }
        }
        return (this.opts.SyncConfig = syncConfig);
    }
    updateMutationConditionInput(ctx, type) {
        if (this.supportsConditions(ctx)) {
            const tableXMutationConditionInputName = graphql_transformer_common_1.ModelResourceIDs.ModelConditionInputTypeName(type.name.value);
            if (this.typeExist(tableXMutationConditionInputName, ctx)) {
                const tableXMutationConditionInput = ctx.getType(tableXMutationConditionInputName);
                const keyDirectives = type.directives.filter(d => d.name.value === 'key');
                if (keyDirectives && keyDirectives.length > 0) {
                    return;
                }
                const idField = tableXMutationConditionInput.fields.find(f => f.name.value === 'id');
                if (idField) {
                    const reducedFields = tableXMutationConditionInput.fields.filter(f => Boolean(f.name.value !== 'id'));
                    const updatedInput = {
                        ...tableXMutationConditionInput,
                        fields: reducedFields,
                    };
                    ctx.putType(updatedInput);
                }
            }
        }
    }
    supportsConditions(context) {
        return context.getTransformerVersion() >= exports.CONDITIONS_MINIMUM_VERSION;
    }
    static isTimestampCompatibleField(field) {
        if (field && !(graphql_transformer_common_1.getBaseType(field.type) === 'AWSDateTime' || graphql_transformer_common_1.getBaseType(field.type) === 'String')) {
            return false;
        }
        return true;
    }
    addInitalizationMetadata(ctx, resourceId, initCodeGenerator) {
        const ddbMetadata = ctx.metadata.has(METADATA_KEY) ? ctx.metadata.get(METADATA_KEY) : {};
        ddbMetadata.hoistedRequestMappingContent = {
            ...ddbMetadata === null || ddbMetadata === void 0 ? void 0 : ddbMetadata.hoistedRequestMappingContent,
            [resourceId]: initCodeGenerator,
        };
        ctx.metadata.set(METADATA_KEY, ddbMetadata);
    }
}
exports.DynamoDBModelTransformer = DynamoDBModelTransformer;
//# sourceMappingURL=DynamoDBModelTransformer.js.map