import json
import httpx
import argparse  
from typing import Any
from mcp.server.fastmcp import FastMCP

# 初始化 MCP 服务器
mcp = FastMCP("WeatherServer")

# OpenWeather API 配置
OPENWEATHER_API_BASE = "https://api.openweathermap.org/data/2.5/weather"
API_KEY = "1b1440376b5a9ab8569a723c254c51e4"  
USER_AGENT = "weather-app/1.0"

async def fetch_weather(city: str) -> dict[str, Any] | None:
    """
    从 OpenWeather API 获取天气信息。
    """
    if API_KEY is None:
        return {"error": "API_KEY 未设置，请提供有效的 OpenWeather API Key。"}

    params = {
        "q": city,
        "appid": API_KEY,
        "units": "metric",
        "lang": "zh_cn"
    }
    headers = {"User-Agent": USER_AGENT}

    async with httpx.AsyncClient() as client:
        try:
            response = await client.get(OPENWEATHER_API_BASE, params=params, headers=headers, timeout=30.0)
            response.raise_for_status()
            return response.json()
        except httpx.HTTPStatusError as e:
            return {"error": f"HTTP 错误: {e.response.status_code}"}
        except Exception as e:
            return {"error": f"请求失败: {str(e)}"}

def format_weather(data: dict[str, Any] | str) -> str:
    """
    将天气数据格式化为易读文本。
    """
    if isinstance(data, str):
        try:
            data = json.loads(data)
        except Exception as e:
            return f"无法解析天气数据: {e}"

    if "error" in data:
        return f"⚠️ {data['error']}"

    city = data.get("name", "未知")
    country = data.get("sys", {}).get("country", "未知")
    temp = data.get("main", {}).get("temp", "N/A")
    humidity = data.get("main", {}).get("humidity", "N/A")
    wind_speed = data.get("wind", {}).get("speed", "N/A")
    weather_list = data.get("weather", [{}])
    description = weather_list[0].get("description", "未知")

    return (
        f"🌍 {city}, {country}\n"
        f"🌡 温度: {temp}°C\n"
        f"💧 湿度: {humidity}%\n"
        f"🌬 风速: {wind_speed} m/s\n"
        f"🌤 天气: {description}\n"
    )

@mcp.tool()
async def query_weather(city: str) -> str:
    """
    输入指定城市的英文名称，返回今日天气查询结果。
    """
    data = await fetch_weather(city)
    return format_weather(data)

@mcp.tool()
def add(a: int, b: int) -> int:
    """
    计算 a 和 b 的和。
    """
    return a + b

def main():
    """主入口函数，支持配置 SSE 服务器的主机和端口"""
    parser = argparse.ArgumentParser(
        description="辰知MCP工具 - 提供天气查询和计算功能",
        formatter_class=argparse.RawDescriptionHelpFormatter,
        epilog="""
使用示例：
  # 本地运行（仅本机访问）
  mcp-chenzhi
  
  # 允许外部访问（绑定到所有网络接口）
  mcp-chenzhi --host 0.0.0.0
  
  # 自定义端口
  mcp-chenzhi --host 0.0.0.0 --port 6060
  
  # 完整配置
  mcp-chenzhi --host 0.0.0.0 --port 6060
        """
    )
    parser.add_argument(
        "--host",
        type=str,
        default="127.0.0.1",
        help="服务器绑定的主机地址（默认: 127.0.0.1，使用 0.0.0.0 允许外部访问）"
    )
    parser.add_argument(
        "--port",
        type=int,
        default=6060,
        help="服务器监听的端口（默认: 6060）"
    )
    
    args = parser.parse_args()
    
    import sys
    print(f"🚀 启动 MCP SSE 服务器...", file=sys.stderr)
    print(f"📍 服务器名称: WeatherServer", file=sys.stderr)
    print(f"🔧 传输协议: SSE", file=sys.stderr)
    print(f"🌐 绑定地址: {args.host}:{args.port}", file=sys.stderr)
    
    if args.host == "0.0.0.0":
        print(f"⚠️  服务器允许外部访问，请确保防火墙已配置", file=sys.stderr)
    
    try:
        mcp.run(transport='sse', host=args.host, port=args.port)
    except Exception as e:
        print(f"❌ 启动失败: {e}", file=sys.stderr)
        raise

if __name__ == "__main__":
    main()