from unittest import skipIf
from unittest.mock import MagicMock, patch

from django.apps import apps
from django.conf import settings
from django.contrib.auth import get_user_model
from django.test import TestCase

from environment.entities import (
    EnvironmentStatus,
    InstanceType,
    Region,
    ResearchEnvironment,
)
from environment.exceptions import (
    BillingVerificationFailed,
    ChangeEnvironmentInstanceTypeFailed,
    DeleteEnvironmentFailed,
    EnvironmentCreationFailed,
    GetAvailableEnvironmentsFailed,
    IdentityProvisioningFailed,
    StartEnvironmentFailed,
    StopEnvironmentFailed,
)
from environment.services import (
    change_environment_machine_type,
    create_cloud_identity,
    create_research_environment,
    delete_environment,
    get_environments_with_projects,
    start_stopped_environment,
    stop_running_environment,
    verify_billing_and_create_workspace,
)
from environment.tests.helpers import (
    create_user_with_cloud_identity,
    create_user_without_cloud_identity,
)
from environment.tests.mocks import get_workspace_list_json

PublishedProject = apps.get_model("project", "PublishedProject")


User = get_user_model()


@skipIf(
    not settings.ENABLE_CLOUD_RESEARCH_ENVIRONMENTS,
    "Research environments are disabled",
)
class CreateCloudIdentityTestCase(TestCase):
    def setUp(self):
        self.user = create_user_without_cloud_identity()

    @patch("environment.api.create_cloud_identity")
    def test_raises_if_request_fails(self, mock_create_cloud_identity):
        mock_create_cloud_identity.return_value.ok = False
        self.assertRaises(
            IdentityProvisioningFailed,
            create_cloud_identity,
            self.user,
        )

    @patch("environment.api.create_cloud_identity")
    def test_creates_cloud_identity_if_request_succeeds(
        self, mock_create_cloud_identity
    ):
        mock_otp = "top"
        mock_email = "email"
        mock_create_cloud_identity.return_value.ok = True
        mock_create_cloud_identity.return_value.json.return_value = {
            "one-time-password": mock_otp,
            "email-id": mock_email,
        }

        otp, identity = create_cloud_identity(self.user)
        self.assertEqual(otp, mock_otp)
        self.assertEqual(identity.gcp_user_id, f"researcher_{self.user.username}")
        self.assertEqual(identity.email, mock_email)
        self.assertEqual(self.user.cloud_identity, identity)


@skipIf(
    not settings.ENABLE_CLOUD_RESEARCH_ENVIRONMENTS,
    "Research environments are disabled",
)
class CreateResearchEnvironmentTestCase(TestCase):
    def setUp(self):
        self.project = MagicMock()
        self.project.slug = "slug"
        self.project.get_project_file_root.return_value = "bucket"
        self.user = create_user_with_cloud_identity()

    @patch("environment.api.create_workbench")
    def test_raises_if_request_fails(self, mock_create_workbench):
        mock_create_workbench.return_value.ok = False
        self.assertRaises(
            EnvironmentCreationFailed,
            create_research_environment,
            self.user,
            self.project,
            "us-central1",
            "n1-standard-2",
            "enviornment_type",
            "100",
        )

    @patch("environment.api.create_workbench")
    def test_returns_api_response_if_request_succeeds(self, mock_create_workbench):
        mock_create_workbench.return_value.ok = True
        result = create_research_environment(
            self.user,
            self.project,
            "us-central1",
            "n1-standard-2",
            "enviornment_type",
            "100",
        )
        self.assertEqual(result, mock_create_workbench.return_value)


@skipIf(
    not settings.ENABLE_CLOUD_RESEARCH_ENVIRONMENTS,
    "Research environments are disabled",
)
class StopRunningEnvironmentTestCase(TestCase):
    def setUp(self):
        self.user = create_user_with_cloud_identity()

    @patch("environment.api.stop_workbench")
    def test_raises_if_request_fails(self, mock_stop_workbench):
        mock_stop_workbench.return_value.ok = False
        self.assertRaises(
            StopEnvironmentFailed,
            stop_running_environment,
            self.user,
            "workbench_id",
            Region.AUSTRALIA_SOUTHEAST,
        )

    @patch("environment.api.stop_workbench")
    def test_raises_if_request_succeeds(self, mock_stop_workbench):
        mock_stop_workbench.return_value.ok = True
        result = stop_running_environment(
            self.user, "workbench_id", Region.AUSTRALIA_SOUTHEAST
        )
        self.assertEqual(result, mock_stop_workbench.return_value)


@skipIf(
    not settings.ENABLE_CLOUD_RESEARCH_ENVIRONMENTS,
    "Research environments are disabled",
)
class StartStoppedEnvironmentTestCase(TestCase):
    def setUp(self):
        self.user = create_user_with_cloud_identity()

    @patch("environment.api.start_workbench")
    def test_raises_if_request_fails(self, mock_start_workbench):
        mock_start_workbench.return_value.ok = False
        self.assertRaises(
            StartEnvironmentFailed,
            start_stopped_environment,
            self.user,
            "workbench_id",
            Region.AUSTRALIA_SOUTHEAST,
        )

    @patch("environment.api.start_workbench")
    def test_raises_if_request_succeeds(self, mock_stop_workbench):
        mock_stop_workbench.return_value.ok = True
        result = start_stopped_environment(
            self.user, "workbench_id", Region.AUSTRALIA_SOUTHEAST
        )
        self.assertEqual(result, mock_stop_workbench.return_value)


@skipIf(
    not settings.ENABLE_CLOUD_RESEARCH_ENVIRONMENTS,
    "Research environments are disabled",
)
class ChangeEnvironmentInstanceTypeTestCase(TestCase):
    def setUp(self):
        self.user = create_user_with_cloud_identity()

    @patch("environment.api.change_workbench_machine_type")
    def test_raises_if_request_fails(self, mock_change_workbench_machine_type):
        mock_change_workbench_machine_type.return_value.ok = False
        self.assertRaises(
            ChangeEnvironmentInstanceTypeFailed,
            change_environment_machine_type,
            self.user,
            "workbench_id",
            Region.AUSTRALIA_SOUTHEAST,
            InstanceType.N1_STANDARD_2,
        )

    @patch("environment.api.change_workbench_machine_type")
    def test_raises_if_request_succeeds(self, mock_change_workbench_machine_type):
        mock_change_workbench_machine_type.return_value.ok = True
        result = change_environment_machine_type(
            self.user,
            "workbench_id",
            Region.AUSTRALIA_SOUTHEAST,
            InstanceType.N1_STANDARD_2,
        )
        self.assertEqual(result, mock_change_workbench_machine_type.return_value)


@skipIf(
    not settings.ENABLE_CLOUD_RESEARCH_ENVIRONMENTS,
    "Research environments are disabled",
)
class DeleteEnvironmentTestCase(TestCase):
    def setUp(self):
        self.user = create_user_with_cloud_identity()

    @patch("environment.api.delete_workbench")
    def test_raises_if_request_fails(self, mock_delete_workbench):
        mock_delete_workbench.return_value.ok = False
        self.assertRaises(
            DeleteEnvironmentFailed,
            delete_environment,
            self.user,
            "workbench_id",
            Region.AUSTRALIA_SOUTHEAST,
        )

    @patch("environment.api.delete_workbench")
    def test_raises_if_request_succeeds(self, mock_delete_workbench):
        mock_delete_workbench.return_value.ok = True
        result = delete_environment(
            self.user, "workbench_id", Region.AUSTRALIA_SOUTHEAST
        )
        self.assertEqual(result, mock_delete_workbench.return_value)


@skipIf(
    not settings.ENABLE_CLOUD_RESEARCH_ENVIRONMENTS,
    "Research environments are disabled",
)
class VerifyBillingAndCreateWorkspaceTestCase(TestCase):
    def setUp(self):
        self.user = create_user_with_cloud_identity()
        self.some_billing_id = "XXXXXX-XXXXXX-XXXXXX"

    @patch("environment.api.create_workspace")
    def test_raises_if_request_fails(self, mock_create_workspace):
        mock_create_workspace.return_value.ok = False
        self.assertRaises(
            BillingVerificationFailed,
            verify_billing_and_create_workspace,
            self.user,
            self.some_billing_id,
        )

    @patch("environment.api.create_workspace")
    def test_not_raises_if_request_succeeds(self, mock_create_workspace):
        mock_create_workspace.return_value.ok = True
        verify_billing_and_create_workspace(self.user, self.some_billing_id)


@skipIf(
    not settings.ENABLE_CLOUD_RESEARCH_ENVIRONMENTS,
    "Research environments are disabled",
)
class GetAvailableEnvironmentsWithProjectsTestCase(TestCase):
    def setUp(self):
        self.user = create_user_with_cloud_identity()

    @patch("environment.api.get_workspace_list")
    def test_fetches_environments_and_parses_to_entity(self, mock_get_workspace_list):
        mock_get_workspace_list.return_value.json.return_value = get_workspace_list_json

        environment_project_pairs = get_environments_with_projects(self.user)

        self.assertTrue(
            all(
                True if environment.status != EnvironmentStatus.DESTROYED else False
                for environment, _project in environment_project_pairs
            )
        )
        self.assertIsInstance(environment_project_pairs, list)
        self.assertTrue(
            all(
                True if isinstance(environment, ResearchEnvironment) else False
                for environment, project in environment_project_pairs
            )
        )

    @patch("environment.api.get_workspace_list")
    def test_matches_running_environments_with_projects(self, mock_get_workspace_list):
        mock_get_workspace_list.return_value.json.return_value = get_workspace_list_json
        demopsn_project = PublishedProject.objects.get(slug="demopsn")

        environment_project_pairs = get_environments_with_projects(self.user)
        self.assertEqual(len(environment_project_pairs), 1)
        self.assertEqual(
            environment_project_pairs[0][0].group_granting_data_access, "demopsn"
        )
        self.assertEqual(environment_project_pairs[0][1], demopsn_project)

    @patch("environment.api.get_workspace_list")
    def test_raises_if_request_fails(self, mock_get_workspace_list):
        mock_get_workspace_list.return_value.ok = False
        self.assertRaises(
            GetAvailableEnvironmentsFailed,
            get_environments_with_projects,
            self.user,
        )
