"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.AlbLogsTable = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const region_info_1 = require("aws-cdk-lib/region-info");
const athena_1 = require("../athena");
const glue_1 = require("../glue");
const data_format_1 = require("../glue/lib/data-format");
class AlbLogsTable extends glue_1.Table {
    /**
       * Creates a new instance of the AlbLogsTable class.
       *
       * @param scope A CDK Construct that will serve as this stack's parent in the construct tree.
       * @param id A name to be associated with the stack and used in resource naming. Must be unique
       * within the context of 'scope'.
       * @param props Arguments related to the configuration of the resource.
       */
    constructor(scope, id, props) {
        const projectionYear = new Date().getFullYear() - 1;
        super(scope, id, {
            columns: [
                new glue_1.BasicColumn({
                    name: 'type',
                    type: 'string',
                }),
                new glue_1.BasicColumn({
                    name: 'time',
                    type: 'string',
                }),
                new glue_1.BasicColumn({
                    name: 'elb',
                    type: 'string',
                }),
                new glue_1.BasicColumn({
                    name: 'client_ip',
                    type: 'string',
                }),
                new glue_1.BasicColumn({
                    name: 'client_port',
                    type: 'int',
                }),
                new glue_1.BasicColumn({
                    name: 'target_ip',
                    type: 'string',
                }),
                new glue_1.BasicColumn({
                    name: 'target_port',
                    type: 'int',
                }),
                new glue_1.BasicColumn({
                    name: 'request_processing_time',
                    type: 'double',
                }),
                new glue_1.BasicColumn({
                    name: 'target_processing_time',
                    type: 'double',
                }),
                new glue_1.BasicColumn({
                    name: 'response_processing_time',
                    type: 'double',
                }),
                new glue_1.BasicColumn({
                    name: 'elb_status_code',
                    type: 'int',
                }),
                new glue_1.BasicColumn({
                    name: 'target_status_code',
                    type: 'string',
                }),
                new glue_1.BasicColumn({
                    name: 'received_bytes',
                    type: 'bigint',
                }),
                new glue_1.BasicColumn({
                    name: 'sent_bytes',
                    type: 'bigint',
                }),
                new glue_1.BasicColumn({
                    name: 'request_verb',
                    type: 'string',
                }),
                new glue_1.BasicColumn({
                    name: 'request_url',
                    type: 'string',
                }),
                new glue_1.BasicColumn({
                    name: 'request_proto',
                    type: 'string',
                }),
                new glue_1.BasicColumn({
                    name: 'user_agent',
                    type: 'string',
                }),
                new glue_1.BasicColumn({
                    name: 'ssl_cipher',
                    type: 'string',
                }),
                new glue_1.BasicColumn({
                    name: 'ssl_protocol',
                    type: 'string',
                }),
                new glue_1.BasicColumn({
                    name: 'target_group_arn',
                    type: 'string',
                }),
                new glue_1.BasicColumn({
                    name: 'trace_id',
                    type: 'string',
                }),
                new glue_1.BasicColumn({
                    name: 'domain_name',
                    type: 'string',
                }),
                new glue_1.BasicColumn({
                    name: 'chosen_cert_arn',
                    type: 'string',
                }),
                new glue_1.BasicColumn({
                    name: 'matched_rule_priority',
                    type: 'string',
                }),
                new glue_1.BasicColumn({
                    name: 'request_creation_time',
                    type: 'string',
                }),
                new glue_1.BasicColumn({
                    name: 'actions_executed',
                    type: 'string',
                }),
                new glue_1.BasicColumn({
                    name: 'redirect_url',
                    type: 'string',
                }),
                new glue_1.BasicColumn({
                    name: 'lambda_error_reason',
                    type: 'string',
                }),
                new glue_1.BasicColumn({
                    name: 'target_port_list',
                    type: 'string',
                }),
                new glue_1.BasicColumn({
                    name: 'target_status_code_list',
                    type: 'string',
                }),
                new glue_1.BasicColumn({
                    name: 'classification',
                    type: 'string',
                }),
                new glue_1.BasicColumn({
                    name: 'classification_reason',
                    type: 'string',
                }),
            ],
            compressed: false,
            dataFormat: {
                inputFormat: data_format_1.InputFormat.TEXT,
                outputFormat: data_format_1.OutputFormat.HIVE_IGNORE_KEY_TEXT,
                serializationLibrary: data_format_1.SerializationLibrary.REGEXP,
            },
            database: props.database,
            description: 'Table used for querying ALB access logs.',
            location: `s3://${props.bucket.bucketName}/${props.s3Prefix ?? ''}AWSLogs/`,
            name: props.name,
            owner: 'hadoop',
            parameters: {
                'EXTERNAL': 'TRUE',
                'projection.day.format': 'yyyy/MM/dd',
                'projection.day.interval': '1',
                'projection.day.range': `${projectionYear}/01/01,NOW`,
                'projection.day.type': 'date',
                'projection.day.interval.unit': 'DAYS',
                'projection.logname.type': 'enum',
                'projection.logname.values': 'elasticloadbalancing',
                'projection.regionname.type': 'enum',
                'projection.regionname.values': region_info_1.RegionInfo.regions.map((x) => {
                    return x.name;
                }).join(','),
                'projection.source.type': 'enum',
                'projection.source.values': aws_cdk_lib_1.Lazy.string({
                    produce: () => {
                        if (aws_cdk_lib_1.Token.isUnresolved(this.stack.account)) {
                            aws_cdk_lib_1.Annotations.of(this).addWarning([
                                `ALB logs table stack for ${this.node.path} is environment agnostic.`,
                                'Cross account partition projection cannot be configured on environment',
                                'agnostic stacks. To enable partition projection please specify an account',
                                `for the ${this.stack.node.path} stack.`,
                            ].join(' '));
                            return this.stack.account;
                        }
                        else {
                            return [...new Set(this.node.root.node.findAll().reduce((prev, cur) => {
                                    if ((cur instanceof aws_cdk_lib_1.Stage || cur instanceof aws_cdk_lib_1.Stack) && cur.account && !aws_cdk_lib_1.Token.isUnresolved(cur.account)) {
                                        prev.push(cur.account);
                                    }
                                    return prev;
                                }, []))].join(',');
                        }
                    },
                }),
                'projection.enabled': 'true',
                'storage.location.template': `s3://${props.bucket.bucketName}/${props.s3Prefix ?? ''}AWSLogs/\${source}/\${logname}/\${regionname}/\${day}`,
            },
            partitionKeys: [
                new glue_1.BasicColumn({
                    name: 'source',
                    type: 'string',
                }),
                new glue_1.BasicColumn({
                    name: 'logname',
                    type: 'string',
                }),
                new glue_1.BasicColumn({
                    name: 'regionname',
                    type: 'string',
                }),
                new glue_1.BasicColumn({
                    name: 'day',
                    type: 'string',
                }),
            ],
            serdeParameters: {
                'input.regex': '([^ ]*) ([^ ]*) ([^ ]*) ([^ ]*):([0-9]*) ([^ ]*)[:-]([0-9]*) ([-.0-9]*) ([-.0-9]*) ([-.0-9]*) (|[-0-9]*) (-|[-0-9]*) ([-0-9]*) ([-0-9]*) \"([^ ]*) (.*) (- |[^ ]*)\" \"([^\"]*)\" ([A-Z0-9-_]+) ([A-Za-z0-9.-]*) ([^ ]*) \"([^\"]*)\" \"([^\"]*)\" \"([^\"]*)\" ([-.0-9]*) ([^ ]*) \"([^\"]*)\" \"([^\"]*)\" \"([^ ]*)\" \"([^\s]+?)\" \"([^\s]+)\" \"([^ ]*)\" \"([^ ]*)\"',
                'serialization.format': '1',
            },
            storedAsSubDirectories: false,
            tableType: glue_1.TableType.EXTERNAL_TABLE,
        });
        this.createQueries = props.createQueries ?? true;
        this.friendlyQueryNames = props.friendlyQueryNames ?? false;
        if (this.createQueries) {
            this.topIpsNamedQuery = new athena_1.NamedQuery(this, 'top-ips-named-query', {
                database: this.database,
                description: 'Gets the 100 most actvie IP addresses by request count.',
                name: this.friendlyQueryNames ? 'alb-top-ips' : undefined,
                queryString: [
                    'SELECT client_ip,',
                    '    COUNT(*) AS requests,',
                    '    COUNT_IF(elb_status_code BETWEEN 400 AND 499) AS errors_4xx,',
                    '    COUNT_IF(elb_status_code BETWEEN 500 AND 599) AS errors_5xx,',
                    '    SUM(sent_bytes) AS sent,',
                    '    SUM(received_bytes) AS received,',
                    '    SUM(sent_bytes + received_bytes) AS total,',
                    '    ARBITRARY(user_agent) as user_agent',
                    `FROM ${this.tableName}`,
                    "WHERE day >= DATE_FORMAT(NOW() - PARSE_DURATION('1d'), '%Y/%m/%d')",
                    "    AND FROM_ISO8601_TIMESTAMP(time) >= NOW() - PARSE_DURATION('1d')",
                    'GROUP BY client_ip',
                    'ORDER by total DESC LIMIT 100;',
                ].join('\n'),
            });
            this.status5xxNamedQuery = new athena_1.NamedQuery(this, 'status-5xx-named-query', {
                database: this.database,
                description: 'Gets the 100 most recent ELB 5XX responses.',
                name: this.friendlyQueryNames ? 'alb-5xx-errors' : undefined,
                queryString: [
                    "SELECT FROM_ISO8601_TIMESTAMP(time) AT TIME ZONE 'UTC' AS time,",
                    "    CONCAT(client_ip, ':', CAST(client_port AS varchar)) AS client,",
                    "    CONCAT(target_ip, ':', CAST(target_port AS varchar)) AS target,",
                    '    CASE',
                    "        WHEN CAST(elb_status_code AS varchar) != target_status_code THEN CONCAT(CAST(elb_status_code AS varchar), ' (', target_status_code, ')')",
                    '        ELSE CAST(elb_status_code AS varchar)',
                    '    END as status,',
                    '    request_verb AS verb,',
                    '    request_url AS url,',
                    '    source AS account,',
                    '    elb',
                    `FROM ${this.tableName}`,
                    'WHERE elb_status_code BETWEEN 500 AND 599',
                    "    AND day >= DATE_FORMAT(NOW() - PARSE_DURATION('1d'), '%Y/%m/%d')",
                    "    AND FROM_ISO8601_TIMESTAMP(time) >= NOW() - PARSE_DURATION('1d')",
                    'ORDER BY time DESC LIMIT 100;',
                ].join('\n'),
            });
        }
    }
}
exports.AlbLogsTable = AlbLogsTable;
_a = JSII_RTTI_SYMBOL_1;
AlbLogsTable[_a] = { fqn: "cdk-extensions.glue_tables.AlbLogsTable", version: "0.0.20" };
//# sourceMappingURL=data:application/json;base64,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