"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.FlowLogsTable = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const region_info_1 = require("aws-cdk-lib/region-info");
const athena_1 = require("../athena");
const ec2_1 = require("../ec2");
const glue_1 = require("../glue");
const formatting_1 = require("../utils/formatting");
class FlowLogsTable extends glue_1.Table {
    /**
       * Creates a new instance of the FlowLogsTable class.
       *
       * @param scope A CDK Construct that will serve as this stack's parent in the construct tree.
       * @param id A name to be associated with the stack and used in resource naming. Must be unique
       * within the context of 'scope'.
       * @param props Arguments related to the configuration of the resource.
       */
    constructor(scope, id, props) {
        const projectionYear = new Date().getFullYear() - 1;
        super(scope, id, {
            compressed: false,
            dataFormat: {
                inputFormat: glue_1.InputFormat.TEXT,
                outputFormat: glue_1.OutputFormat.HIVE_IGNORE_KEY_TEXT,
                serializationLibrary: glue_1.SerializationLibrary.LAZY_SIMPLE,
            },
            database: props.database,
            description: 'Table used for querying network traffic flow.',
            location: `s3://${props.bucket.bucketName}/${props.s3Prefix ?? ''}AWSLogs/`,
            name: props.name,
            owner: 'hadoop',
            parameters: {
                'EXTERNAL': 'TRUE',
                'projection.day.format': 'yyyy/MM/dd',
                'projection.day.interval': '1',
                'projection.day.range': `${projectionYear}/01/01,NOW`,
                'projection.day.type': 'date',
                'projection.day.interval.unit': 'DAYS',
                'projection.logname.type': 'enum',
                'projection.logname.values': 'vpcflowlogs',
                'projection.regionname.type': 'enum',
                'projection.regionname.values': region_info_1.RegionInfo.regions.map((x) => {
                    return x.name;
                }).join(','),
                'projection.source.type': 'enum',
                'projection.source.values': aws_cdk_lib_1.Lazy.string({
                    produce: () => {
                        if (aws_cdk_lib_1.Token.isUnresolved(this.stack.account)) {
                            aws_cdk_lib_1.Annotations.of(this).addWarning([
                                `Flow logs table stack for ${this.node.path} is environment agnostic.`,
                                'Cross account partition projection cannot be configured on environment',
                                'agnostic stacks. To enable partition projection please specify an account',
                                `for the ${this.stack.node.path} stack.`,
                            ].join(' '));
                            return this.stack.account;
                        }
                        else {
                            return [...new Set(this.node.root.node.findAll().reduce((prev, cur) => {
                                    if ((cur instanceof aws_cdk_lib_1.Stage || cur instanceof aws_cdk_lib_1.Stack) && cur.account && !aws_cdk_lib_1.Token.isUnresolved(cur.account)) {
                                        prev.push(cur.account);
                                    }
                                    return prev;
                                }, []))].join(',');
                        }
                    },
                }),
                'projection.enabled': 'true',
                'skip.header.line.count': '1',
                'storage.location.template': `s3://${props.bucket.bucketName}/${props.s3Prefix ?? ''}AWSLogs/\${source}/\${logname}/\${regionname}/\${day}`,
            },
            partitionKeys: [
                new glue_1.BasicColumn({
                    name: 'source',
                    type: 'string',
                }),
                new glue_1.BasicColumn({
                    name: 'logname',
                    type: 'string',
                }),
                new glue_1.BasicColumn({
                    name: 'regionname',
                    type: 'string',
                }),
                new glue_1.BasicColumn({
                    name: 'day',
                    type: 'string',
                }),
            ],
            serdeParameters: {
                'field.delim': ' ',
                'serialization.format': ' ',
            },
            storedAsSubDirectories: false,
            tableType: glue_1.TableType.EXTERNAL_TABLE,
        });
        this.createQueries = props.createQueries ?? true;
        this.format = props.format ?? ec2_1.FlowLogFormat.V2;
        this.friendlyQueryNames = props.friendlyQueryNames ?? false;
        const internalRejectedFields = [
            ec2_1.FlowLogField.ACTION,
            ec2_1.FlowLogField.DSTADDR,
            ec2_1.FlowLogField.DSTPORT,
            ec2_1.FlowLogField.PROTOCOL,
            ec2_1.FlowLogField.SRCADDR,
            ec2_1.FlowLogField.SRCPORT,
            ec2_1.FlowLogField.START,
        ];
        if (this.createQueries) {
            if (formatting_1.includesAll(this.format.fields, internalRejectedFields)) {
                this.internalRejectedNamedQuery = new athena_1.NamedQuery(this, 'internal-rejected-named-query', {
                    database: this.database,
                    description: 'Gets the 100 most recent rejected packets that stayed within the private network ranges.',
                    name: this.friendlyQueryNames ? 'flow-logs-internal-rejected' : undefined,
                    queryString: [
                        'SELECT FROM_UNIXTIME("start", \'UTC\') AS "timestamp",',
                        '    CASE',
                        "        WHEN protocol = 1 THEN 'ICMP'",
                        "        WHEN protocol = 6 THEN 'TCP'",
                        "        WHEN protocol = 17 THEN 'UDP'",
                        '        ELSE CAST(protocol AS varchar)',
                        '    END AS proto,',
                        "    IF(protocol IN (6, 17), CONCAT(srcaddr, ':', CAST(srcport AS varchar)), srcaddr) AS source,",
                        "    IF(protocol IN (6, 17), CONCAT(dstaddr, ':', CAST(dstport AS varchar)), dstaddr) AS destination,",
                        '    action',
                        `FROM ${this.tableName}`,
                        "WHERE REGEXP_LIKE(srcaddr, '^(10\.|192\.168\.|172\.(1[6-9]|2[0-9]|3[0-1])\.).*')",
                        "    AND REGEXP_LIKE(dstaddr, '^(10\.|192\.168\.|172\.(1[6-9]|2[0-9]|3[0-1])\.).*')",
                        "    AND day >= DATE_FORMAT(NOW() - PARSE_DURATION('1d'), '%Y/%m/%d')",
                        "    AND start >= TO_UNIXTIME(NOW() - PARSE_DURATION('1d'))",
                        "    AND action = 'REJECT'",
                        'ORDER BY start DESC LIMIT 100;',
                    ].join('\n'),
                });
            }
        }
    }
    renderStorageDescriptor() {
        this.format.fields.forEach((x) => {
            this.addColumn(new glue_1.BasicColumn({
                name: x.name.toLowerCase().replace(/[^a-z0-9]+/g, ''),
                type: x.type === ec2_1.FlowLogDataType.STRING ? 'string' : 'int',
            }));
        });
        return super.renderStorageDescriptor();
    }
}
exports.FlowLogsTable = FlowLogsTable;
_a = JSII_RTTI_SYMBOL_1;
FlowLogsTable[_a] = { fqn: "cdk-extensions.glue_tables.FlowLogsTable", version: "0.0.20" };
//# sourceMappingURL=data:application/json;base64,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