"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.Job = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const aws_glue_1 = require("aws-cdk-lib/aws-glue");
const aws_iam_1 = require("aws-cdk-lib/aws-iam");
const aws_logs_1 = require("aws-cdk-lib/aws-logs");
const formatting_1 = require("../utils/formatting");
class Job extends aws_cdk_lib_1.Resource {
    constructor(scope, id, props) {
        super(scope, id, props);
        // Internal properties
        this._arguments = {};
        this._connections = [];
        this.allocatedCapacity = props.allocatedCapacity;
        this.continuousLogging = props.continuousLogging;
        this.description = props.description;
        this.executable = props.executable;
        this.maxCapacity = props.maxCapacity;
        this.maxConcurrentRuns = props.maxConcurrentRuns;
        this.maxRetries = props.maxRetries;
        this.name = props.name;
        this.notifyDelayAfter = props.notifyDelayAfter;
        this.securityConfiguration = props.securityConfiguration;
        this.timeout = props.timeout;
        this.workerCount = props.workerCount;
        this.workerType = props.workerType;
        const executable = props.executable.bind();
        if (props.enableProfilingMetrics) {
            this.addArgument('--enable-metrics', '');
        }
        if (this.continuousLogging?.enabled) {
            this.logGroup = this.configureContinuousLogging(this.continuousLogging);
        }
        if (this.workerType?.name === 'G.1X' && (this.workerCount ?? 0) > 299) {
            aws_cdk_lib_1.Annotations.of(this).addError('The maximum number of workers you can define for G.1X is 299.');
        }
        else if (this.workerType?.name === 'G.2X' && (this.workerCount ?? 0) > 149) {
            aws_cdk_lib_1.Annotations.of(this).addError('The maximum number of workers you can define for G.2X is 149.');
        }
        this.role = props.role ?? new aws_iam_1.Role(this, 'role', {
            assumedBy: new aws_iam_1.ServicePrincipal('glue.amazonaws.com'),
            managedPolicies: [
                aws_iam_1.ManagedPolicy.fromAwsManagedPolicyName('service-role/AWSGlueServiceRole'),
            ],
        });
        this.resource = new aws_glue_1.CfnJob(this, 'Resource', {
            allocatedCapacity: this.allocatedCapacity,
            command: {
                name: executable.type.name,
                pythonVersion: executable.pythonVersion,
                scriptLocation: this.buildCodeS3ObjectUrl(executable.script),
            },
            connections: aws_cdk_lib_1.Lazy.uncachedAny({
                produce: () => {
                    return !!!this._connections.length ? undefined : {
                        connections: this._connections.map((x) => {
                            return x.connectionName;
                        }),
                    };
                },
            }),
            defaultArguments: aws_cdk_lib_1.Lazy.uncachedAny({
                produce: () => {
                    return !!Object.keys(this._arguments).length ? this._arguments : undefined;
                },
            }),
            description: this.description,
            executionProperty: formatting_1.undefinedIfNoKeys({
                maxConcurrentRuns: this.maxConcurrentRuns,
            }),
            glueVersion: executable.glueVersion.name,
            maxCapacity: this.maxCapacity,
            maxRetries: this.maxRetries,
            name: this.name,
            notificationProperty: formatting_1.undefinedIfNoKeys({
                notifyDelayAfter: this.notifyDelayAfter?.toMinutes(),
            }),
            numberOfWorkers: this.workerCount,
            role: this.role.roleArn,
            securityConfiguration: this.securityConfiguration?.securityConfigurationName,
            timeout: this.timeout?.toMinutes(),
            workerType: this.workerType?.name,
        });
        this.jobArn = this.stack.formatArn({
            resource: 'job',
            resourceName: this.resource.ref,
            service: 'glue',
        });
        this.jobName = this.resource.ref;
        const args = props.defaultArguments ?? {};
        Object.keys(args).forEach((x) => {
            this.addArgument(x, args[x]);
        });
        props.connections?.forEach((x) => {
            this.addConnection(x);
        });
    }
    addArgument(key, value) {
        this._arguments[key] = value;
    }
    addConnection(connection) {
        this._connections.push(connection);
    }
    buildCodeS3ObjectUrl(code) {
        const s3Location = code.bind(this, this.role).s3Location;
        return `s3://${s3Location.bucketName}/${s3Location.objectKey}`;
    }
    configureContinuousLogging(props) {
        const logGroup = props?.logGroup ?? new aws_logs_1.LogGroup(this, 'log-group', {
            removalPolicy: aws_cdk_lib_1.RemovalPolicy.DESTROY,
            retention: aws_logs_1.RetentionDays.TWO_WEEKS,
        });
        logGroup.grantWrite(this.role);
        this.addArgument('--enable-continuous-cloudwatch-log', 'true');
        this.addArgument('--enable-continuous-log-filter', `${props.quiet ?? true}`);
        this.addArgument('--continuous-log-logGroup', logGroup.logGroupName);
        if (props.logStreamPrefix) {
            this.addArgument('--continuous-log-logStreamPrefix', props.logStreamPrefix);
        }
        if (props.conversionPattern) {
            this.addArgument('--continuous-log-conversionPattern', props.conversionPattern);
        }
        return logGroup;
    }
}
exports.Job = Job;
_a = JSII_RTTI_SYMBOL_1;
Job[_a] = { fqn: "cdk-extensions.glue.Job", version: "0.0.20" };
//# sourceMappingURL=data:application/json;base64,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