import { AssetOptions } from 'aws-cdk-lib';
import { IGrantable } from 'aws-cdk-lib/aws-iam';
import { IBucket, Location } from 'aws-cdk-lib/aws-s3';
import { Construct } from 'constructs';
/**
 * Represents a Glue Job's Code assets (an asset can be a scripts, a jar, a python file or any other file).
 */
export declare abstract class Code {
    /**
       * Job code as an S3 object.
       * @param bucket The S3 bucket
       * @param key The object key
       */
    static fromBucket(bucket: IBucket, key: string): S3Code;
    /**
       * Job code from a local disk path.
       *
       * @param path code file (not a directory).
       */
    static fromAsset(path: string, options?: AssetOptions): AssetCode;
    /**
       * Called when the Job is initialized to allow this object to bind.
       */
    abstract bind(scope: Construct, grantable: IGrantable): CodeConfig;
}
/**
 * Glue job Code from an S3 bucket.
 */
export declare class S3Code extends Code {
    private readonly bucket;
    private readonly key;
    constructor(bucket: IBucket, key: string);
    bind(_scope: Construct, grantable: IGrantable): CodeConfig;
}
/**
 * Job Code from a local file.
 */
export declare class AssetCode extends Code {
    private readonly path;
    private readonly options;
    private asset?;
    /**
       * @param path The path to the Code file.
       */
    constructor(path: string, options?: AssetOptions);
    bind(scope: Construct, grantable: IGrantable): CodeConfig;
    /**
       * Hash a string
       */
    private hashcode;
}
/**
 * Result of binding `Code` into a `Job`.
 */
export interface CodeConfig {
    /**
       * The location of the code in S3.
       */
    readonly s3Location: Location;
}
