"use strict";
var _a, _b, _c;
Object.defineProperty(exports, "__esModule", { value: true });
exports.AssetCode = exports.S3Code = exports.Code = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const crypto_1 = require("crypto");
const fs_1 = require("fs");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const aws_s3_assets_1 = require("aws-cdk-lib/aws-s3-assets");
/**
 * Represents a Glue Job's Code assets (an asset can be a scripts, a jar, a python file or any other file).
 */
class Code {
    /**
       * Job code as an S3 object.
       * @param bucket The S3 bucket
       * @param key The object key
       */
    static fromBucket(bucket, key) {
        return new S3Code(bucket, key);
    }
    /**
       * Job code from a local disk path.
       *
       * @param path code file (not a directory).
       */
    static fromAsset(path, options) {
        return new AssetCode(path, options);
    }
}
exports.Code = Code;
_a = JSII_RTTI_SYMBOL_1;
Code[_a] = { fqn: "cdk-extensions.glue.Code", version: "0.0.20" };
/**
 * Glue job Code from an S3 bucket.
 */
class S3Code extends Code {
    constructor(bucket, key) {
        super();
        this.bucket = bucket;
        this.key = key;
    }
    bind(_scope, grantable) {
        this.bucket.grantRead(grantable, this.key);
        return {
            s3Location: {
                bucketName: this.bucket.bucketName,
                objectKey: this.key,
            },
        };
    }
}
exports.S3Code = S3Code;
_b = JSII_RTTI_SYMBOL_1;
S3Code[_b] = { fqn: "cdk-extensions.glue.S3Code", version: "0.0.20" };
/**
 * Job Code from a local file.
 */
class AssetCode extends Code {
    /**
       * @param path The path to the Code file.
       */
    constructor(path, options = {}) {
        super();
        this.path = path;
        this.options = options;
        if (fs_1.lstatSync(this.path).isDirectory()) {
            throw new Error(`Code path ${this.path} is a directory. Only files are supported`);
        }
    }
    bind(scope, grantable) {
        // If the same AssetCode is used multiple times, retain only the first instantiation.
        if (!this.asset) {
            this.asset = new aws_s3_assets_1.Asset(scope, `Code${this.hashcode(this.path)}`, {
                path: this.path,
                ...this.options,
            });
        }
        else if (aws_cdk_lib_1.Stack.of(this.asset) !== aws_cdk_lib_1.Stack.of(scope)) {
            throw new Error([
                `Asset is already associated with another stack '${aws_cdk_lib_1.Stack.of(this.asset).stackName}'.` +
                    'Create a new Code instance for every stack.',
            ].join(' '));
        }
        this.asset.grantRead(grantable);
        return {
            s3Location: {
                bucketName: this.asset.s3BucketName,
                objectKey: this.asset.s3ObjectKey,
            },
        };
    }
    /**
       * Hash a string
       */
    hashcode(s) {
        const hash = crypto_1.createHash('md5');
        hash.update(s);
        return hash.digest('hex');
    }
    ;
}
exports.AssetCode = AssetCode;
_c = JSII_RTTI_SYMBOL_1;
AssetCode[_c] = { fqn: "cdk-extensions.glue.AssetCode", version: "0.0.20" };
//# sourceMappingURL=data:application/json;base64,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