"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.SecurityConfiguration = exports.JobBookmarksEncryptionMode = exports.CloudWatchEncryptionMode = exports.S3EncryptionMode = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const aws_glue_1 = require("aws-cdk-lib/aws-glue");
const aws_kms_1 = require("aws-cdk-lib/aws-kms");
const formatting_1 = require("../utils/formatting");
/**
 * Encryption mode for S3.
 * @see https://docs.aws.amazon.com/glue/latest/webapi/API_S3Encryption.html#Glue-Type-S3Encryption-S3EncryptionMode
 */
var S3EncryptionMode;
(function (S3EncryptionMode) {
    /**
       * Server-side encryption (SSE) with an AWS KMS key managed by the account owner.
       *
       * @see https://docs.aws.amazon.com/AmazonS3/latest/dev/UsingKMSEncryption.html
       */
    S3EncryptionMode["KMS"] = "SSE-KMS";
    /**
       * Server side encryption (SSE) with an Amazon S3-managed key.
       *
       * @see https://docs.aws.amazon.com/AmazonS3/latest/dev/UsingServerSideEncryption.html
       */
    S3EncryptionMode["S3_MANAGED"] = "SSE-S3";
})(S3EncryptionMode = exports.S3EncryptionMode || (exports.S3EncryptionMode = {}));
/**
 * Encryption mode for CloudWatch Logs.
 * @see https://docs.aws.amazon.com/glue/latest/webapi/API_CloudWatchEncryption.html#Glue-Type-CloudWatchEncryption-CloudWatchEncryptionMode
 */
var CloudWatchEncryptionMode;
(function (CloudWatchEncryptionMode) {
    /**
       * Server-side encryption (SSE) with an AWS KMS key managed by the account owner.
       *
       * @see https://docs.aws.amazon.com/AmazonS3/latest/dev/UsingKMSEncryption.html
       */
    CloudWatchEncryptionMode["KMS"] = "SSE-KMS";
})(CloudWatchEncryptionMode = exports.CloudWatchEncryptionMode || (exports.CloudWatchEncryptionMode = {}));
/**
 * Encryption mode for Job Bookmarks.
 * @see https://docs.aws.amazon.com/glue/latest/webapi/API_JobBookmarksEncryption.html#Glue-Type-JobBookmarksEncryption-JobBookmarksEncryptionMode
 */
var JobBookmarksEncryptionMode;
(function (JobBookmarksEncryptionMode) {
    /**
       * Client-side encryption (CSE) with an AWS KMS key managed by the account owner.
       *
       * @see https://docs.aws.amazon.com/AmazonS3/latest/dev/UsingClientSideEncryption.html
       */
    JobBookmarksEncryptionMode["CLIENT_SIDE_KMS"] = "CSE-KMS";
})(JobBookmarksEncryptionMode = exports.JobBookmarksEncryptionMode || (exports.JobBookmarksEncryptionMode = {}));
class SecurityConfiguration extends aws_cdk_lib_1.Resource {
    constructor(scope, id, props) {
        super(scope, id, props);
        this.name = props.name ?? aws_cdk_lib_1.Names.uniqueId(this);
        let cloudWatchKey = undefined;
        let jobBookmarksKey = undefined;
        let s3Key = undefined;
        const fetchKey = (key) => {
            return key ?? this.node.tryFindChild('key') ?? new aws_kms_1.Key(this, 'key');
        };
        if (this.cloudWatchEncryption?.mode === CloudWatchEncryptionMode.KMS) {
            cloudWatchKey = fetchKey(this.cloudWatchEncryption.kmsKey);
        }
        else if (this.jobBookmarksEncryption?.mode === JobBookmarksEncryptionMode.CLIENT_SIDE_KMS && !!!this.jobBookmarksEncryption.kmsKey) {
            jobBookmarksKey = fetchKey(this.jobBookmarksEncryption.kmsKey);
        }
        else if (this.s3Encryption?.mode === S3EncryptionMode.KMS && !!!this.s3Encryption.kmsKey) {
            s3Key = fetchKey(this.s3Encryption.kmsKey);
        }
        this.key = this.node.tryFindChild('key');
        this.resource = new aws_glue_1.CfnSecurityConfiguration(this, 'Resource', {
            encryptionConfiguration: {
                cloudWatchEncryption: formatting_1.undefinedIfNoKeys({
                    cloudWatchEncryptionMode: this.cloudWatchEncryption?.mode,
                    kmsKeyArn: cloudWatchKey?.keyArn,
                }),
                jobBookmarksEncryption: formatting_1.undefinedIfNoKeys({
                    jobBookmarksEncryptionMode: this.jobBookmarksEncryption?.mode,
                    kmsKeyArn: jobBookmarksKey?.keyArn,
                }),
                s3Encryptions: !!!this.s3Encryption ? undefined : [{
                        s3EncryptionMode: this.s3Encryption?.mode,
                        kmsKeyArn: s3Key?.keyArn,
                    }],
            },
            name: this.name,
        });
        this.securityConfigurationName = this.resource.ref;
    }
}
exports.SecurityConfiguration = SecurityConfiguration;
_a = JSII_RTTI_SYMBOL_1;
SecurityConfiguration[_a] = { fqn: "cdk-extensions.glue.SecurityConfiguration", version: "0.0.20" };
//# sourceMappingURL=data:application/json;base64,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