"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.S3Destination = exports.S3CompressionFormat = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const aws_iam_1 = require("aws-cdk-lib/aws-iam");
const formatting_1 = require("../../../utils/formatting");
const delivery_stream_destination_1 = require("./delivery-stream-destination");
var S3CompressionFormat;
(function (S3CompressionFormat) {
    S3CompressionFormat["GZIP"] = "GZIP";
    S3CompressionFormat["HADOOP_SNAPPY"] = "HADOOP_SNAPPY";
    S3CompressionFormat["SNAPPY"] = "Snappy";
    S3CompressionFormat["UNCOMPRESSED"] = "UNCOMPRESSED";
    S3CompressionFormat["ZIP"] = "ZIP";
})(S3CompressionFormat = exports.S3CompressionFormat || (exports.S3CompressionFormat = {}));
class S3Destination extends delivery_stream_destination_1.DeliveryStreamDestination {
    constructor(bucket, options = {}) {
        super();
        this.bucket = bucket;
        this.buffering = options.buffering;
        this.cloudwatchLoggingConfiguration = options.cloudwatchLoggingConfiguration;
        this.compressionFormat = options.compressionFormat;
        this.encryptionEnabled = options.encryptionEnabled ?? !!options.encryptionKey;
        this.encryptionKey = options.encryptionKey;
        this.errorOutputPrefix = options.errorOutputPrefix;
        this.keyPrefix = options.keyPrefix;
        this._role = options.role;
    }
    // Private accessors
    get role() {
        return this._role;
    }
    bind(scope) {
        return {
            s3DestinationConfiguration: this.buildConfiguration(scope),
        };
    }
    buildConfiguration(scope) {
        const stack = aws_cdk_lib_1.Stack.of(scope);
        const isCrossAccount = this.bucket.stack.account !== stack.account;
        this._role = this._role ?? scope.node.tryFindChild('s3-delivery-role') ?? new aws_iam_1.Role(scope, 's3-delivery-role', {
            assumedBy: new aws_iam_1.PrincipalWithConditions(new aws_iam_1.ServicePrincipal('firehose.amazonaws.com'), {
                StringEquals: {
                    'sts:ExternalId': aws_cdk_lib_1.Stack.of(scope).account,
                },
            }),
        });
        this._role.addToPrincipalPolicy(new aws_iam_1.PolicyStatement({
            actions: [
                's3:GetBucketLocation',
                's3:ListBucket',
                's3:ListBucketMultipartUploads',
            ],
            effect: aws_iam_1.Effect.ALLOW,
            resources: [
                this.bucket.bucketArn,
            ],
        }));
        this._role.addToPrincipalPolicy(new aws_iam_1.PolicyStatement({
            actions: [
                's3:AbortMultipartUpload',
                's3:GetObject',
                's3:PutObject',
                ...(isCrossAccount ? ['s3:PutObjectAcl'] : []),
            ],
            effect: aws_iam_1.Effect.ALLOW,
            resources: [
                this.bucket.arnForObjects('*'),
            ],
        }));
        if (this.encryptionKey && this.encryptionEnabled) {
            this._role.addToPrincipalPolicy(new aws_iam_1.PolicyStatement({
                actions: [
                    'kms:Decrypt',
                    'kms:GenerateDataKey',
                ],
                conditions: {
                    StringEquals: {
                        'kms:ViaService': `s3.${aws_cdk_lib_1.Stack.of(scope).region}.amazonaws.com`,
                    },
                    StringLike: {
                        'kms:EncryptionContext:aws:s3:arn': [
                            this.bucket.arnForObjects(`${this.keyPrefix ?? ''}*`),
                            ...(this.errorOutputPrefix ? this.bucket.arnForObjects(`${this.errorOutputPrefix ?? ''}*`) : []),
                        ],
                    },
                },
                effect: aws_iam_1.Effect.ALLOW,
                resources: [
                    this.encryptionKey.keyArn,
                ],
            }));
        }
        // Configure cross account bucket access
        if (!aws_cdk_lib_1.Token.isUnresolved(stack.account) && isCrossAccount) {
            this.bucket.addToResourcePolicy(new aws_iam_1.PolicyStatement({
                actions: [
                    's3:GetBucketLocation',
                    's3:ListBucket',
                    's3:ListBucketMultipartUploads',
                ],
                effect: aws_iam_1.Effect.ALLOW,
                principals: [
                    new aws_iam_1.AccountPrincipal(stack.account),
                ],
                resources: [
                    stack.resolve(this.bucket.bucketArn),
                ],
            }));
            this.bucket.addToResourcePolicy(new aws_iam_1.PolicyStatement({
                actions: [
                    's3:AbortMultipartUpload',
                    's3:GetObject',
                    's3:PutObject',
                    's3:PutObjectAcl',
                ],
                effect: aws_iam_1.Effect.ALLOW,
                principals: [
                    new aws_iam_1.AccountPrincipal(stack.account),
                ],
                resources: [
                    this.bucket.arnForObjects('*'),
                ],
            }));
        }
        if (this.cloudwatchLoggingConfiguration?.enabled) {
            const logGroupScope = this.cloudwatchLoggingConfiguration.logGroup?.logGroupName ?? '*';
            const logStreamScope = this.cloudwatchLoggingConfiguration.logStream?.logStreamName ?? '*';
            this._role.addToPrincipalPolicy(new aws_iam_1.PolicyStatement({
                actions: [
                    'logs:PutLogEvents',
                ],
                effect: aws_iam_1.Effect.ALLOW,
                resources: [
                    aws_cdk_lib_1.Stack.of(this.cloudwatchLoggingConfiguration.logGroup ?? scope).formatArn({
                        arnFormat: aws_cdk_lib_1.ArnFormat.COLON_RESOURCE_NAME,
                        resource: 'log-group',
                        resourceName: `${logGroupScope}:log-stream:${logStreamScope}`,
                        service: 'logs',
                    }),
                ],
            }));
        }
        return {
            bucketArn: this.bucket.bucketArn,
            bufferingHints: this.buffering?.bind(scope),
            cloudWatchLoggingOptions: this.cloudwatchLoggingConfiguration?.bind(scope),
            compressionFormat: this.compressionFormat,
            encryptionConfiguration: formatting_1.definedFieldsOrUndefined({
                kmsEncryptionConfig: this.encryptionKey === undefined ? undefined : formatting_1.definedFieldsOrUndefined({
                    awskmsKeyArn: this.encryptionKey.keyArn,
                }),
                noEncryptionConfig: (!this.encryptionEnabled && this.encryptionKey) ? 'NoEncryption' : undefined,
            }),
            errorOutputPrefix: this.errorOutputPrefix,
            prefix: this.keyPrefix,
            roleArn: this._role.roleArn,
        };
    }
    renderBackupConfiguration(scope, enabled) {
        const mode = enabled ? 'Enabled' : 'Disabled';
        return {
            s3BackupConfiguration: this.buildConfiguration(scope),
            s3BackupMode: enabled === undefined ? undefined : mode,
        };
    }
}
exports.S3Destination = S3Destination;
_a = JSII_RTTI_SYMBOL_1;
S3Destination[_a] = { fqn: "cdk-extensions.kinesis_firehose.S3Destination", version: "0.0.20" };
//# sourceMappingURL=data:application/json;base64,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