"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.InboundResolver = void 0;
const aws_ec2_1 = require("aws-cdk-lib/aws-ec2");
const resolver_endpoint_base_1 = require("./resolver-endpoint-base");
class InboundResolver extends resolver_endpoint_base_1.ResolverEndpointBase {
    /**
       * Creates a new instance of the InboundResolver class.
       *
       * @param scope A CDK Construct that will serve as this stack's parent in the construct tree.
       * @param id A name to be associated with the stack and used in resource naming. Must be unique
       * within the context of 'scope'.
       * @param props Arguments related to the configuration of the resource.
       */
    constructor(scope, id, props) {
        super(scope, id, {
            ...props,
            direction: InboundResolver.DIRECTION,
            subnetOffset: InboundResolver.SUBNET_OFFSET,
        });
        this.inboundCidrs = props.inboundCidrs ?? InboundResolver.DEFAULT_INBOUND_CIDRS;
        for (let cidr of this.inboundCidrs) {
            this.securityGroup.addIngressRule(aws_ec2_1.Peer.ipv4(cidr), aws_ec2_1.Port.tcp(53), `TCP DNS from ${cidr}`);
            this.securityGroup.addIngressRule(aws_ec2_1.Peer.ipv4(cidr), aws_ec2_1.Port.udp(53), `UDP DNS from ${cidr}`);
        }
    }
}
exports.InboundResolver = InboundResolver;
// Static properties
InboundResolver.DEFAULT_INBOUND_CIDRS = [
    '10.0.0.0/8',
    '172.16.0.0/12',
    '192.168.0.0/16',
];
InboundResolver.DIRECTION = 'INBOUND';
InboundResolver.SUBNET_OFFSET = 4;
//# sourceMappingURL=data:application/json;base64,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