import { ArnComponents, ResourceProps } from 'aws-cdk-lib';
import { IVpc, SubnetSelection } from 'aws-cdk-lib/aws-ec2';
import { CfnResourceShare } from 'aws-cdk-lib/aws-ram';
import { Construct } from 'constructs';
import { ForwardResolverRule } from './forward-resolver-rule';
import { ResolverEndpointBase } from './resolver-endpoint-base';
import { ResolverRuleTargetIp } from './resolver-rule';
/**
 * Configuration for the Outbound Resolver resource.
 */
export interface OutboundResolverProps extends ResourceProps {
    /**
       * The Amazon Resource Name (ARN) of the AWS organization. If this is provided then any resolver
       * rules that get added will be automatically shared with the rest of the organization using AWS RAM.
       */
    readonly organizationArn?: ArnComponents;
    /**
       * A list of CIDR ranges that the Outbound Resolver should be able to connect to to make queries.
       *
       * @default OutboundResolver.DEFAULT_OUTBOUND_CIDRS
       */
    readonly outboundCidrs?: string[];
    /**
       * The selection criteria to use to determine which subnets in a VPC resolver endpoints should be
       * created in. As a matter of best practice, at least 3 subnets in different availablity zones should
       * be used.
       */
    readonly subnets: SubnetSelection;
    /**
       * The VPC where the resolver endpoints should be created.
       */
    readonly vpc: IVpc;
}
export declare class OutboundResolver extends ResolverEndpointBase {
    static readonly DEFAULT_OUTBOUND_CIDRS: string[];
    static readonly DIRECTION: string;
    static readonly SUBNET_OFFSET: number;
    readonly organizationArn?: ArnComponents;
    readonly outboundCidrs: string[];
    readonly resolverRules: ForwardResolverRule[];
    resourceShare?: CfnResourceShare;
    /**
       * Creates a new instance of the OutboundResolver class.
       *
       * @param scope A CDK Construct that will serve as this stack's parent in the construct tree.
       * @param id A name to be associated with the stack and used in resource naming. Must be unique
       * within the context of 'scope'.
       * @param props Arguments related to the configuration of the resource.
       */
    constructor(scope: Construct, id: string, props: OutboundResolverProps);
    /**
       * Adds a new Resolver Rule for a given domain to the Outbound Resolver. Also adds the created
       * rule to a RAM Resource Share if an organization ARN was specified when the Outbound Resolver
       * was created.
       *
       * @param domain The domain you would like to add the rule for.
       * @param targets The IP addresses of the external resolver that should be used to resolve the
       * domain.
       */
    addRule(domain: string, targets: ResolverRuleTargetIp[]): ForwardResolverRule;
}
