"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.ResolverRule = exports.ResolverRuleType = void 0;
const aws_cdk_lib_1 = require("aws-cdk-lib");
const aws_route53resolver_1 = require("aws-cdk-lib/aws-route53resolver");
const resolver_rule_association_1 = require("./resolver-rule-association");
var ResolverRuleType;
(function (ResolverRuleType) {
    ResolverRuleType["FORWARD"] = "FORWARD";
    ResolverRuleType["RECURSIVE"] = "RECURSIVE";
    ResolverRuleType["SYSTEM"] = "SYSTEM";
})(ResolverRuleType = exports.ResolverRuleType || (exports.ResolverRuleType = {}));
class ResolverRuleBase extends aws_cdk_lib_1.Resource {
    /**
       * Creates an association between a VPC and a resolver rule.
       *
       * A resolver rule that isn't associated with any VPC's will have no
       * effect, even in the VPC containing the rule's outbound resolver.
       *
       * VPC associations don't need to be in the same account as
       * the resolver rule for rules shared with [AWS RAM](https://docs.aws.amazon.com/ram/latest/userguide/what-is.html).
       *
       * @param vpc The VPC you want to create the association for.
       * @param options Additional configuration for the association.
       * @returns The association that was created.
       */
    associateVpc(vpc, options) {
        return new resolver_rule_association_1.ResolverRuleAssociation(this, `association-${vpc.node.addr}`, {
            name: options?.name,
            resolverRule: this,
            vpc: vpc,
        });
    }
}
class ResolverRule extends ResolverRuleBase {
    /**
       * Creates a new instance of the ResolverRule class.
       *
       * @param scope A CDK Construct that will serve as this stack's parent in the construct tree.
       * @param id A name to be associated with the stack and used in resource naming. Must be unique
       * within the context of 'scope'.
       * @param props Arguments related to the configuration of the resource.
       */
    constructor(scope, id, props) {
        super(scope, id, props);
        // Internal properties
        this._targetIps = [];
        this.domainName = props.domainName;
        this.resolverEndpoint = props.resolverEndpoint;
        this.ruleName = props.ruleName;
        this.ruleType = props.ruleType;
        if (this.ruleType === ResolverRuleType.RECURSIVE) {
            aws_cdk_lib_1.Annotations.of(this).addError("Currently, only Resolver can create rules that have a value of 'RECURSIVE' for 'RuleType'.");
        }
        this.resource = new aws_route53resolver_1.CfnResolverRule(this, 'Resource', {
            domainName: this.domainName,
            ruleType: this.ruleType,
            name: this.ruleName,
            resolverEndpointId: this.resolverEndpoint?.resolverEndpointId,
            targetIps: aws_cdk_lib_1.Lazy.any({
                produce: () => {
                    return this._targetIps.map((x) => {
                        return {
                            ip: x.address,
                            port: x.port?.toString(),
                        };
                    });
                },
            }, {
                omitEmptyArray: true,
            }),
        });
        this.resolverRuleArn = this.resource.attrArn;
        this.resolverRuleDomainName = this.resource.attrDomainName;
        this.resolverRuleEndpointId = this.resource.attrResolverEndpointId;
        this.resolverRuleId = this.resource.attrResolverRuleId;
        this.resolverRuleName = this.resource.attrName;
        this.resolverRuleTargetIps = this.resource.attrTargetIps;
        props.targetIps?.map((x) => {
            this.addTargetIp(x);
        });
    }
    static fromResolverRuleId(scope, id, resolverRuleId) {
        class Import extends ResolverRuleBase {
            constructor() {
                super(...arguments);
                this.resolverRuleArn = aws_cdk_lib_1.Stack.of(scope).formatArn({
                    resource: 'resolver-rule',
                    resourceName: resolverRuleId,
                    service: 'route53resolver',
                });
                this.resolverRuleDomainName = aws_cdk_lib_1.Lazy.string({
                    produce: () => {
                        throw new Error('Cannot get domain name for resolver rules imported using only an ID.');
                    },
                });
                this.resolverRuleEndpointId = aws_cdk_lib_1.Lazy.string({
                    produce: () => {
                        throw new Error('Cannot get endpoint ID for resolver rules imported using only an ID.');
                    },
                });
                this.resolverRuleId = resolverRuleId;
                this.resolverRuleName = aws_cdk_lib_1.Lazy.string({
                    produce: () => {
                        throw new Error('Cannot get name for resolver rules imported using only an ID.');
                    },
                });
                this.resolverRuleTargetIps = aws_cdk_lib_1.Lazy.any({
                    produce: () => {
                        throw new Error("Cannot get target IP's for resolver rules imported using only an ID.");
                    },
                });
            }
        }
        return new Import(scope, id);
    }
    addTargetIp(target) {
        const port = target.port ?? 53;
        if ((port < 0) || (port > 65535)) {
            aws_cdk_lib_1.Annotations.of(this).addError(`Specified resolver target port (${port}) is  invalid (must be between 0 and 65535).`);
        }
        if (this.ruleType === ResolverRuleType.SYSTEM) {
            aws_cdk_lib_1.Annotations.of(this).addError("Cannot add target IP's to rules of type 'SYSTEM'.");
        }
        this._targetIps.push(target);
        return this;
    }
}
exports.ResolverRule = ResolverRule;
//# sourceMappingURL=data:application/json;base64,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