"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.BucketEncryption = exports.BucketEncryptionAlgorithm = void 0;
const aws_cdk_lib_1 = require("aws-cdk-lib");
const aws_iam_1 = require("aws-cdk-lib/aws-iam");
const aws_kms_1 = require("aws-cdk-lib/aws-kms");
/**
 * Algorithms for enabling at rest encryption on S3 buckets.
 */
var BucketEncryptionAlgorithm;
(function (BucketEncryptionAlgorithm) {
    /**
       * Encrypt files using S3.
       */
    BucketEncryptionAlgorithm["KMS"] = "aws:kms";
    /**
       * Encrypt files using S3 managed AES256 encryption.
       */
    BucketEncryptionAlgorithm["AES256"] = "AES256";
})(BucketEncryptionAlgorithm = exports.BucketEncryptionAlgorithm || (exports.BucketEncryptionAlgorithm = {}));
class BucketEncryption {
    constructor(options = {}) {
        this.algorithm = options.algorithm;
        this.bucketKeyEnabled = options.bucketKeyEnabled;
        this.generateKey = options.generateKey;
    }
    static kms(options) {
        return new BucketEncryption({
            algorithm: BucketEncryptionAlgorithm.KMS,
            encryptionKey: options.encryptionKey,
            bucketKeyEnabled: options.bucketKeyEnabled,
            generateKey: !!options.encryptionKey,
        });
    }
    bind(scope) {
        if (!this.algorithm) {
            return {};
        }
        else if (this.algorithm === BucketEncryptionAlgorithm.AES256) {
            return {
                encryptionSettings: {
                    serverSideEncryptionConfiguration: [
                        {
                            serverSideEncryptionByDefault: {
                                sseAlgorithm: this.algorithm,
                            },
                        },
                    ],
                },
            };
        }
        else {
            let key = this.encryptionKey;
            if (key && this.generateKey) {
                aws_cdk_lib_1.Annotations.of(scope).addError([
                    'Cannot generate an encryption key as another KMS key was',
                    'explicitly specified.',
                ].join(' '));
            }
            else if (this.generateKey) {
                key = new aws_kms_1.Key(scope, 'encryption-key', {
                    admins: [
                        new aws_iam_1.AccountRootPrincipal(),
                    ],
                    description: `Handles bucket encryption for ${scope.node.path}.`,
                    enabled: true,
                    keySpec: aws_kms_1.KeySpec.SYMMETRIC_DEFAULT,
                    keyUsage: aws_kms_1.KeyUsage.ENCRYPT_DECRYPT,
                    removalPolicy: aws_cdk_lib_1.RemovalPolicy.RETAIN,
                });
            }
            return {
                encryptionKey: key,
                encryptionSettings: !key ? undefined : {
                    serverSideEncryptionConfiguration: [
                        {
                            bucketKeyEnabled: this.bucketKeyEnabled,
                            serverSideEncryptionByDefault: {
                                sseAlgorithm: this.algorithm,
                                kmsMasterKeyId: key.keyArn,
                            },
                        },
                    ],
                },
            };
        }
    }
}
exports.BucketEncryption = BucketEncryption;
BucketEncryption.KMS = new BucketEncryption({
    algorithm: BucketEncryptionAlgorithm.KMS,
    bucketKeyEnabled: true,
    generateKey: true,
});
BucketEncryption.KMS_MANAGED = new BucketEncryption({
    algorithm: BucketEncryptionAlgorithm.KMS,
});
BucketEncryption.S3_MANAGED = new BucketEncryption({
    algorithm: BucketEncryptionAlgorithm.AES256,
});
BucketEncryption.UNENCRYPTED = new BucketEncryption();
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiYnVja2V0LWVuY3J5cHRpb24uanMiLCJzb3VyY2VSb290IjoiIiwic291cmNlcyI6WyIuLi8uLi8uLi9zcmMvczMvbGliL2J1Y2tldC1lbmNyeXB0aW9uLnRzIl0sIm5hbWVzIjpbXSwibWFwcGluZ3MiOiI7OztBQUFBLDZDQUF5RDtBQUN6RCxpREFBMkQ7QUFDM0QsaURBQW1FO0FBS25FOztHQUVHO0FBQ0gsSUFBWSx5QkFVWDtBQVZELFdBQVkseUJBQXlCO0lBQ25DOztTQUVLO0lBQ0wsNENBQWUsQ0FBQTtJQUVmOztTQUVLO0lBQ0wsOENBQWlCLENBQUE7QUFDbkIsQ0FBQyxFQVZXLHlCQUF5QixHQUF6QixpQ0FBeUIsS0FBekIsaUNBQXlCLFFBVXBDO0FBbUJELE1BQWEsZ0JBQWdCO0lBaUMzQixZQUFvQixVQUFtQyxFQUFFO1FBQ3ZELElBQUksQ0FBQyxTQUFTLEdBQUcsT0FBTyxDQUFDLFNBQVMsQ0FBQztRQUNuQyxJQUFJLENBQUMsZ0JBQWdCLEdBQUcsT0FBTyxDQUFDLGdCQUFnQixDQUFDO1FBQ2pELElBQUksQ0FBQyxXQUFXLEdBQUcsT0FBTyxDQUFDLFdBQVcsQ0FBQztJQUN6QyxDQUFDO0lBbkJNLE1BQU0sQ0FBQyxHQUFHLENBQUMsT0FBNkI7UUFDN0MsT0FBTyxJQUFJLGdCQUFnQixDQUFDO1lBQzFCLFNBQVMsRUFBRSx5QkFBeUIsQ0FBQyxHQUFHO1lBQ3hDLGFBQWEsRUFBRSxPQUFPLENBQUMsYUFBYTtZQUNwQyxnQkFBZ0IsRUFBRSxPQUFPLENBQUMsZ0JBQWdCO1lBQzFDLFdBQVcsRUFBRSxDQUFDLENBQUMsT0FBTyxDQUFDLGFBQWE7U0FDckMsQ0FBQyxDQUFDO0lBQ0wsQ0FBQztJQWNNLElBQUksQ0FBQyxLQUFpQjtRQUMzQixJQUFJLENBQUMsSUFBSSxDQUFDLFNBQVMsRUFBRTtZQUNuQixPQUFPLEVBQUUsQ0FBQztTQUNYO2FBQU0sSUFBSSxJQUFJLENBQUMsU0FBUyxLQUFLLHlCQUF5QixDQUFDLE1BQU0sRUFBRTtZQUM5RCxPQUFPO2dCQUNMLGtCQUFrQixFQUFFO29CQUNsQixpQ0FBaUMsRUFBRTt3QkFDakM7NEJBQ0UsNkJBQTZCLEVBQUU7Z0NBQzdCLFlBQVksRUFBRSxJQUFJLENBQUMsU0FBUzs2QkFDN0I7eUJBQ0Y7cUJBQ0Y7aUJBQ0Y7YUFDRixDQUFDO1NBQ0g7YUFBTTtZQUNMLElBQUksR0FBRyxHQUFxQixJQUFJLENBQUMsYUFBYSxDQUFDO1lBQy9DLElBQUksR0FBRyxJQUFJLElBQUksQ0FBQyxXQUFXLEVBQUU7Z0JBQzNCLHlCQUFXLENBQUMsRUFBRSxDQUFDLEtBQUssQ0FBQyxDQUFDLFFBQVEsQ0FBQztvQkFDN0IsMERBQTBEO29CQUMxRCx1QkFBdUI7aUJBQ3hCLENBQUMsSUFBSSxDQUFDLEdBQUcsQ0FBQyxDQUFDLENBQUM7YUFDZDtpQkFBTSxJQUFJLElBQUksQ0FBQyxXQUFXLEVBQUU7Z0JBQzNCLEdBQUcsR0FBRyxJQUFJLGFBQUcsQ0FBQyxLQUFLLEVBQUUsZ0JBQWdCLEVBQUU7b0JBQ3JDLE1BQU0sRUFBRTt3QkFDTixJQUFJLDhCQUFvQixFQUFFO3FCQUMzQjtvQkFDRCxXQUFXLEVBQUUsaUNBQWlDLEtBQUssQ0FBQyxJQUFJLENBQUMsSUFBSSxHQUFHO29CQUNoRSxPQUFPLEVBQUUsSUFBSTtvQkFDYixPQUFPLEVBQUUsaUJBQU8sQ0FBQyxpQkFBaUI7b0JBQ2xDLFFBQVEsRUFBRSxrQkFBUSxDQUFDLGVBQWU7b0JBQ2xDLGFBQWEsRUFBRSwyQkFBYSxDQUFDLE1BQU07aUJBQ3BDLENBQUMsQ0FBQzthQUNKO1lBRUQsT0FBTztnQkFDTCxhQUFhLEVBQUUsR0FBRztnQkFDbEIsa0JBQWtCLEVBQUUsQ0FBQyxHQUFHLENBQUMsQ0FBQyxDQUFDLFNBQVMsQ0FBQyxDQUFDLENBQUM7b0JBQ3JDLGlDQUFpQyxFQUFFO3dCQUNqQzs0QkFDRSxnQkFBZ0IsRUFBRSxJQUFJLENBQUMsZ0JBQWdCOzRCQUN2Qyw2QkFBNkIsRUFBRTtnQ0FDN0IsWUFBWSxFQUFFLElBQUksQ0FBQyxTQUFTO2dDQUM1QixjQUFjLEVBQUUsR0FBRyxDQUFDLE1BQU07NkJBQzNCO3lCQUNGO3FCQUNGO2lCQUNGO2FBQ0YsQ0FBQztTQUNIO0lBQ0gsQ0FBQzs7QUF6RkgsNENBMEZDO0FBekZ3QixvQkFBRyxHQUFxQixJQUFJLGdCQUFnQixDQUFDO0lBQ2xFLFNBQVMsRUFBRSx5QkFBeUIsQ0FBQyxHQUFHO0lBQ3hDLGdCQUFnQixFQUFFLElBQUk7SUFDdEIsV0FBVyxFQUFFLElBQUk7Q0FDbEIsQ0FBQyxDQUFDO0FBRW9CLDRCQUFXLEdBQXFCLElBQUksZ0JBQWdCLENBQUM7SUFDMUUsU0FBUyxFQUFFLHlCQUF5QixDQUFDLEdBQUc7Q0FDekMsQ0FBQyxDQUFDO0FBRW9CLDJCQUFVLEdBQXFCLElBQUksZ0JBQWdCLENBQUM7SUFDekUsU0FBUyxFQUFFLHlCQUF5QixDQUFDLE1BQU07Q0FDNUMsQ0FBQyxDQUFDO0FBRW9CLDRCQUFXLEdBQXFCLElBQUksZ0JBQWdCLEVBQUUsQ0FBQyIsInNvdXJjZXNDb250ZW50IjpbImltcG9ydCB7IEFubm90YXRpb25zLCBSZW1vdmFsUG9saWN5IH0gZnJvbSAnYXdzLWNkay1saWInO1xuaW1wb3J0IHsgQWNjb3VudFJvb3RQcmluY2lwYWwgfSBmcm9tICdhd3MtY2RrLWxpYi9hd3MtaWFtJztcbmltcG9ydCB7IElLZXksIEtleSwgS2V5U3BlYywgS2V5VXNhZ2UgfSBmcm9tICdhd3MtY2RrLWxpYi9hd3Mta21zJztcbmltcG9ydCB7IENmbkJ1Y2tldCB9IGZyb20gJ2F3cy1jZGstbGliL2F3cy1zMyc7XG5pbXBvcnQgeyBJQ29uc3RydWN0IH0gZnJvbSAnY29uc3RydWN0cyc7XG5cblxuLyoqXG4gKiBBbGdvcml0aG1zIGZvciBlbmFibGluZyBhdCByZXN0IGVuY3J5cHRpb24gb24gUzMgYnVja2V0cy5cbiAqL1xuZXhwb3J0IGVudW0gQnVja2V0RW5jcnlwdGlvbkFsZ29yaXRobSB7XG4gIC8qKlxuICAgICAqIEVuY3J5cHQgZmlsZXMgdXNpbmcgUzMuXG4gICAgICovXG4gIEtNUyA9ICdhd3M6a21zJyxcblxuICAvKipcbiAgICAgKiBFbmNyeXB0IGZpbGVzIHVzaW5nIFMzIG1hbmFnZWQgQUVTMjU2IGVuY3J5cHRpb24uXG4gICAgICovXG4gIEFFUzI1NiA9ICdBRVMyNTYnXG59XG5cbmludGVyZmFjZSBCdWNrZXRFbmNyeXB0aW9uT3B0aW9ucyB7XG4gIHJlYWRvbmx5IGFsZ29yaXRobT86IEJ1Y2tldEVuY3J5cHRpb25BbGdvcml0aG07XG4gIHJlYWRvbmx5IGJ1Y2tldEtleUVuYWJsZWQ/OiBib29sZWFuO1xuICByZWFkb25seSBlbmNyeXB0aW9uS2V5PzogSUtleTtcbiAgcmVhZG9ubHkgZ2VuZXJhdGVLZXk/OiBib29sZWFuO1xufVxuXG5leHBvcnQgaW50ZXJmYWNlIEttc0VuY3J5cHRpb25PcHRpb25zIHtcbiAgcmVhZG9ubHkgYnVja2V0S2V5RW5hYmxlZD86IGJvb2xlYW47XG4gIHJlYWRvbmx5IGVuY3J5cHRpb25LZXk/OiBJS2V5O1xufVxuXG5leHBvcnQgaW50ZXJmYWNlIEJ1Y2tldEVuY3J5cHRpb25Db25maWd1cmF0aW9uIHtcbiAgcmVhZG9ubHkgZW5jcnlwdGlvbktleT86IElLZXk7XG4gIHJlYWRvbmx5IGVuY3J5cHRpb25TZXR0aW5ncz86IENmbkJ1Y2tldC5CdWNrZXRFbmNyeXB0aW9uUHJvcGVydHk7XG59XG5cbmV4cG9ydCBjbGFzcyBCdWNrZXRFbmNyeXB0aW9uIHtcbiAgcHVibGljIHN0YXRpYyByZWFkb25seSBLTVM6IEJ1Y2tldEVuY3J5cHRpb24gPSBuZXcgQnVja2V0RW5jcnlwdGlvbih7XG4gICAgYWxnb3JpdGhtOiBCdWNrZXRFbmNyeXB0aW9uQWxnb3JpdGhtLktNUyxcbiAgICBidWNrZXRLZXlFbmFibGVkOiB0cnVlLFxuICAgIGdlbmVyYXRlS2V5OiB0cnVlLFxuICB9KTtcblxuICBwdWJsaWMgc3RhdGljIHJlYWRvbmx5IEtNU19NQU5BR0VEOiBCdWNrZXRFbmNyeXB0aW9uID0gbmV3IEJ1Y2tldEVuY3J5cHRpb24oe1xuICAgIGFsZ29yaXRobTogQnVja2V0RW5jcnlwdGlvbkFsZ29yaXRobS5LTVMsXG4gIH0pO1xuXG4gIHB1YmxpYyBzdGF0aWMgcmVhZG9ubHkgUzNfTUFOQUdFRDogQnVja2V0RW5jcnlwdGlvbiA9IG5ldyBCdWNrZXRFbmNyeXB0aW9uKHtcbiAgICBhbGdvcml0aG06IEJ1Y2tldEVuY3J5cHRpb25BbGdvcml0aG0uQUVTMjU2LFxuICB9KTtcblxuICBwdWJsaWMgc3RhdGljIHJlYWRvbmx5IFVORU5DUllQVEVEOiBCdWNrZXRFbmNyeXB0aW9uID0gbmV3IEJ1Y2tldEVuY3J5cHRpb24oKTtcblxuXG4gIHB1YmxpYyBzdGF0aWMga21zKG9wdGlvbnM6IEttc0VuY3J5cHRpb25PcHRpb25zKTogQnVja2V0RW5jcnlwdGlvbiB7XG4gICAgcmV0dXJuIG5ldyBCdWNrZXRFbmNyeXB0aW9uKHtcbiAgICAgIGFsZ29yaXRobTogQnVja2V0RW5jcnlwdGlvbkFsZ29yaXRobS5LTVMsXG4gICAgICBlbmNyeXB0aW9uS2V5OiBvcHRpb25zLmVuY3J5cHRpb25LZXksXG4gICAgICBidWNrZXRLZXlFbmFibGVkOiBvcHRpb25zLmJ1Y2tldEtleUVuYWJsZWQsXG4gICAgICBnZW5lcmF0ZUtleTogISFvcHRpb25zLmVuY3J5cHRpb25LZXksXG4gICAgfSk7XG4gIH1cblxuXG4gIHB1YmxpYyByZWFkb25seSBhbGdvcml0aG0/OiBCdWNrZXRFbmNyeXB0aW9uQWxnb3JpdGhtO1xuICBwdWJsaWMgcmVhZG9ubHkgYnVja2V0S2V5RW5hYmxlZD86IGJvb2xlYW47XG4gIHB1YmxpYyByZWFkb25seSBlbmNyeXB0aW9uS2V5PzogSUtleTtcbiAgcHVibGljIHJlYWRvbmx5IGdlbmVyYXRlS2V5PzogYm9vbGVhbjtcblxuICBwcml2YXRlIGNvbnN0cnVjdG9yKG9wdGlvbnM6IEJ1Y2tldEVuY3J5cHRpb25PcHRpb25zID0ge30pIHtcbiAgICB0aGlzLmFsZ29yaXRobSA9IG9wdGlvbnMuYWxnb3JpdGhtO1xuICAgIHRoaXMuYnVja2V0S2V5RW5hYmxlZCA9IG9wdGlvbnMuYnVja2V0S2V5RW5hYmxlZDtcbiAgICB0aGlzLmdlbmVyYXRlS2V5ID0gb3B0aW9ucy5nZW5lcmF0ZUtleTtcbiAgfVxuXG4gIHB1YmxpYyBiaW5kKHNjb3BlOiBJQ29uc3RydWN0KTogQnVja2V0RW5jcnlwdGlvbkNvbmZpZ3VyYXRpb24ge1xuICAgIGlmICghdGhpcy5hbGdvcml0aG0pIHtcbiAgICAgIHJldHVybiB7fTtcbiAgICB9IGVsc2UgaWYgKHRoaXMuYWxnb3JpdGhtID09PSBCdWNrZXRFbmNyeXB0aW9uQWxnb3JpdGhtLkFFUzI1Nikge1xuICAgICAgcmV0dXJuIHtcbiAgICAgICAgZW5jcnlwdGlvblNldHRpbmdzOiB7XG4gICAgICAgICAgc2VydmVyU2lkZUVuY3J5cHRpb25Db25maWd1cmF0aW9uOiBbXG4gICAgICAgICAgICB7XG4gICAgICAgICAgICAgIHNlcnZlclNpZGVFbmNyeXB0aW9uQnlEZWZhdWx0OiB7XG4gICAgICAgICAgICAgICAgc3NlQWxnb3JpdGhtOiB0aGlzLmFsZ29yaXRobSxcbiAgICAgICAgICAgICAgfSxcbiAgICAgICAgICAgIH0sXG4gICAgICAgICAgXSxcbiAgICAgICAgfSxcbiAgICAgIH07XG4gICAgfSBlbHNlIHtcbiAgICAgIGxldCBrZXk6IElLZXkgfCB1bmRlZmluZWQgPSB0aGlzLmVuY3J5cHRpb25LZXk7XG4gICAgICBpZiAoa2V5ICYmIHRoaXMuZ2VuZXJhdGVLZXkpIHtcbiAgICAgICAgQW5ub3RhdGlvbnMub2Yoc2NvcGUpLmFkZEVycm9yKFtcbiAgICAgICAgICAnQ2Fubm90IGdlbmVyYXRlIGFuIGVuY3J5cHRpb24ga2V5IGFzIGFub3RoZXIgS01TIGtleSB3YXMnLFxuICAgICAgICAgICdleHBsaWNpdGx5IHNwZWNpZmllZC4nLFxuICAgICAgICBdLmpvaW4oJyAnKSk7XG4gICAgICB9IGVsc2UgaWYgKHRoaXMuZ2VuZXJhdGVLZXkpIHtcbiAgICAgICAga2V5ID0gbmV3IEtleShzY29wZSwgJ2VuY3J5cHRpb24ta2V5Jywge1xuICAgICAgICAgIGFkbWluczogW1xuICAgICAgICAgICAgbmV3IEFjY291bnRSb290UHJpbmNpcGFsKCksXG4gICAgICAgICAgXSxcbiAgICAgICAgICBkZXNjcmlwdGlvbjogYEhhbmRsZXMgYnVja2V0IGVuY3J5cHRpb24gZm9yICR7c2NvcGUubm9kZS5wYXRofS5gLFxuICAgICAgICAgIGVuYWJsZWQ6IHRydWUsXG4gICAgICAgICAga2V5U3BlYzogS2V5U3BlYy5TWU1NRVRSSUNfREVGQVVMVCxcbiAgICAgICAgICBrZXlVc2FnZTogS2V5VXNhZ2UuRU5DUllQVF9ERUNSWVBULFxuICAgICAgICAgIHJlbW92YWxQb2xpY3k6IFJlbW92YWxQb2xpY3kuUkVUQUlOLFxuICAgICAgICB9KTtcbiAgICAgIH1cblxuICAgICAgcmV0dXJuIHtcbiAgICAgICAgZW5jcnlwdGlvbktleToga2V5LFxuICAgICAgICBlbmNyeXB0aW9uU2V0dGluZ3M6ICFrZXkgPyB1bmRlZmluZWQgOiB7XG4gICAgICAgICAgc2VydmVyU2lkZUVuY3J5cHRpb25Db25maWd1cmF0aW9uOiBbXG4gICAgICAgICAgICB7XG4gICAgICAgICAgICAgIGJ1Y2tldEtleUVuYWJsZWQ6IHRoaXMuYnVja2V0S2V5RW5hYmxlZCxcbiAgICAgICAgICAgICAgc2VydmVyU2lkZUVuY3J5cHRpb25CeURlZmF1bHQ6IHtcbiAgICAgICAgICAgICAgICBzc2VBbGdvcml0aG06IHRoaXMuYWxnb3JpdGhtLFxuICAgICAgICAgICAgICAgIGttc01hc3RlcktleUlkOiBrZXkua2V5QXJuLFxuICAgICAgICAgICAgICB9LFxuICAgICAgICAgICAgfSxcbiAgICAgICAgICBdLFxuICAgICAgICB9LFxuICAgICAgfTtcbiAgICB9XG4gIH1cbn0iXX0=