"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.DeleteMediaInsightsPipelineConfiguration = exports.UpdateMediaInsightsPipelineConfiguration = exports.CreateMediaInsightsPipelineConfiguration = void 0;
const client_chime_sdk_media_pipelines_1 = require("@aws-sdk/client-chime-sdk-media-pipelines");
const client_ssm_1 = require("@aws-sdk/client-ssm");
const ssmClient = new client_ssm_1.SSMClient({ region: process.env.AWS_REGION });
const chimeSDKMediaPipelineClient = new client_chime_sdk_media_pipelines_1.ChimeSDKMediaPipelinesClient({
    region: process.env.AWS_REGION,
});
function capitalizeKeys(obj) {
    if (Array.isArray(obj)) {
        return obj.map((item) => capitalizeKeys(item));
    }
    if (typeof obj === 'object') {
        const capitalizedObj = {};
        for (const key in obj) {
            if (obj.hasOwnProperty(key)) {
                const capitalizedKey = key.charAt(0).toUpperCase() + key.slice(1);
                const value = obj[key];
                capitalizedObj[capitalizedKey] =
                    typeof value === 'string' && !isNaN(parseFloat(value))
                        ? parseFloat(value)
                        : capitalizeKeys(value);
            }
        }
        return capitalizedObj;
    }
    return obj;
}
function capitalizeElementsKeys(elements) {
    return capitalizeKeys(elements);
}
let formattedElements = [];
let formattedTags = [];
let formattedRealTimeAlertConfiguration = {};
let formattedRealTimeAlertRules = [];
let createMediaInsightsPipelineConfigurationResponse;
let createMediaInsightsPipelineConfigurationParams;
const formatProps = (props) => {
    console.info(`Formatting Props: ${JSON.stringify(props)}`);
    if (props.elements) {
        console.info(`Formatting Elements: ${JSON.stringify(props.elements)}`);
        props.elements.forEach((element) => {
            formattedElements.push(capitalizeElementsKeys(element));
        });
        console.info(`Formatted Elements: ${JSON.stringify(formattedElements)}`);
    }
    if (props.tags) {
        console.info(`Formatting Tags: ${JSON.stringify(props.tags)}`);
        props.tags.forEach((tag) => {
            formattedTags.push(capitalizeKeys(tag));
        });
        console.info(`Formatted Tags: ${JSON.stringify(formattedTags)}`);
    }
    if (props.realTimeAlertConfiguration) {
        console.info(`Formatting Real Time Alert Configuration: ${JSON.stringify(props.realTimeAlertConfiguration)}`);
        props.realTimeAlertConfiguration.rules.forEach((rule) => {
            formattedRealTimeAlertRules.push(capitalizeKeys(rule));
        });
        formattedRealTimeAlertConfiguration = {
            Disabled: props.realTimeAlertConfiguration.disabled,
            Rules: formattedRealTimeAlertRules,
        };
        console.info(`Formatted Real Time Alert Configuration: ${JSON.stringify(formattedRealTimeAlertConfiguration)}`);
    }
    const mediaInsightsPipelineConfigurationParams = {
        ResourceAccessRoleArn: props.resourceAccessRoleArn,
        MediaInsightsPipelineConfigurationName: props.mediaInsightsPipelineConfigurationName,
        ...(props.tags && { Tags: formattedTags }),
        ...(props.clientRequestToken && {
            ClientRequestToken: props.clientRequestToken,
        }),
        ...(props.elements && {
            Elements: formattedElements,
        }),
        ...(props.realTimeAlertConfiguration && {
            RealTimeAlertConfiguration: formattedRealTimeAlertConfiguration,
        }),
    };
    console.info(`mediaInsightsPipelineConfigurationParams: ${JSON.stringify(mediaInsightsPipelineConfigurationParams)}`);
    return mediaInsightsPipelineConfigurationParams;
};
exports.CreateMediaInsightsPipelineConfiguration = async (uid, props) => {
    console.log(`Creating Media Insights Pipeline Configuration: ${uid}`);
    console.log(`Create Media Insights Pipeline Configuration props: ${JSON.stringify(props)}`);
    createMediaInsightsPipelineConfigurationParams = formatProps(props);
    console.log(`Create Media Insights Pipeline Configuration params: ${JSON.stringify(createMediaInsightsPipelineConfigurationParams)}`);
    try {
        createMediaInsightsPipelineConfigurationResponse =
            await chimeSDKMediaPipelineClient.send(new client_chime_sdk_media_pipelines_1.CreateMediaInsightsPipelineConfigurationCommand(createMediaInsightsPipelineConfigurationParams));
        console.log(`createMediaInsightsPipelineConfigurationResponse: ${JSON.stringify(createMediaInsightsPipelineConfigurationResponse)}`);
    }
    catch (error) {
        if (error instanceof Error) {
            console.error(error);
            throw error;
        }
    }
    try {
        await ssmClient.send(new client_ssm_1.PutParameterCommand({
            Name: '/chime/MediaInsightsPipelineConfigurationArn' + uid,
            Value: createMediaInsightsPipelineConfigurationResponse.MediaInsightsPipelineConfiguration
                .MediaInsightsPipelineConfigurationArn,
            Description: 'Media Insight Pipeline Configuration ARN' + uid,
            Overwrite: true,
            Type: 'String',
        }));
    }
    catch (error) {
        if (error instanceof Error) {
            console.error(error);
            throw error;
        }
    }
    return createMediaInsightsPipelineConfigurationResponse.MediaInsightsPipelineConfiguration;
};
let updateMediaInsightsPipelineConfigurationParams;
let updateMediaInsightsPipelineConfigurationResponse;
let updateMediaInsightsPipelineIdentifier;
let getParameterCommandOutput;
exports.UpdateMediaInsightsPipelineConfiguration = async (uid, props) => {
    console.log(`Updating Media Insights Pipeline Configuration: ${uid}`);
    console.log(`Updating Media Insights Pipeline Configuration props: ${JSON.stringify(props)}`);
    try {
        getParameterCommandOutput = await ssmClient.send(new client_ssm_1.GetParameterCommand({
            Name: '/chime/MediaInsightsPipelineConfigurationArn' + uid,
        }));
    }
    catch (error) {
        if (error instanceof Error) {
            console.error(error);
            throw error;
        }
    }
    updateMediaInsightsPipelineIdentifier =
        getParameterCommandOutput.Parameter.Value;
    updateMediaInsightsPipelineConfigurationParams = {
        ...formatProps(props),
        Identifier: updateMediaInsightsPipelineIdentifier,
    };
    console.log(`Update Media Insights Pipeline Configuration params: ${JSON.stringify(updateMediaInsightsPipelineConfigurationParams)}`);
    try {
        updateMediaInsightsPipelineConfigurationResponse =
            await chimeSDKMediaPipelineClient.send(new client_chime_sdk_media_pipelines_1.UpdateMediaInsightsPipelineConfigurationCommand(updateMediaInsightsPipelineConfigurationParams));
        console.log(`createMediaInsightsPipelineConfigurationResponse: ${JSON.stringify(createMediaInsightsPipelineConfigurationResponse)}`);
    }
    catch (error) {
        if (error instanceof Error) {
            console.error(error);
            throw error;
        }
    }
    return updateMediaInsightsPipelineConfigurationResponse.MediaInsightsPipelineConfiguration;
};
let deleteMediaInsightsPipelineIdentifier;
exports.DeleteMediaInsightsPipelineConfiguration = async (uid) => {
    console.log(`Deleting Media Insights Pipeline Configuration: ${uid}`);
    try {
        getParameterCommandOutput = await ssmClient.send(new client_ssm_1.GetParameterCommand({
            Name: '/chime/MediaInsightsPipelineConfigurationArn' + uid,
        }));
    }
    catch (error) {
        if (error instanceof Error) {
            console.error(error);
            throw error;
        }
    }
    try {
        await ssmClient.send(new client_ssm_1.DeleteParameterCommand({
            Name: '/chime/MediaInsightsPipelineConfigurationArn' + uid,
        }));
    }
    catch (error) {
        if (error instanceof Error) {
            console.error(error);
            throw error;
        }
    }
    deleteMediaInsightsPipelineIdentifier =
        getParameterCommandOutput.Parameter.Value;
    console.info(`deleteMediaInsightsPipelineIdentifier: ${deleteMediaInsightsPipelineIdentifier}`);
    try {
        await chimeSDKMediaPipelineClient.send(new client_chime_sdk_media_pipelines_1.DeleteMediaInsightsPipelineConfigurationCommand({
            Identifier: deleteMediaInsightsPipelineIdentifier,
        }));
    }
    catch (error) {
        if (error instanceof Error) {
            console.error(error);
            throw error;
        }
    }
    return;
};
//# sourceMappingURL=data:application/json;base64,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