"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const cfn = require("@aws-cdk/aws-cloudformation");
const cpactions = require("@aws-cdk/aws-codepipeline-actions");
const cdk = require("@aws-cdk/core");
const cxapi = require("@aws-cdk/cx-api");
/**
 * A class to deploy a stack that is part of a CDK App, using CodePipeline.
 * This composite Action takes care of preparing and executing a CloudFormation ChangeSet.
 *
 * It currently does *not* support stacks that make use of ``Asset``s, and
 * requires the deployed stack is in the same account and region where the
 * CodePipeline is hosted.
 */
class PipelineDeployStackAction {
    constructor(props) {
        this.stack = props.stack;
        const assets = this.stack.node.metadata.filter(md => md.type === cxapi.ASSET_METADATA);
        if (assets.length > 0) {
            // FIXME: Implement the necessary actions to publish assets
            throw new Error(`Cannot deploy the stack ${this.stack.stackName} because it references ${assets.length} asset(s)`);
        }
        const createChangeSetRunOrder = props.createChangeSetRunOrder || 1;
        const executeChangeSetRunOrder = props.executeChangeSetRunOrder || (createChangeSetRunOrder + 1);
        if (createChangeSetRunOrder >= executeChangeSetRunOrder) {
            throw new Error(`createChangeSetRunOrder (${createChangeSetRunOrder}) must be < executeChangeSetRunOrder (${executeChangeSetRunOrder})`);
        }
        const changeSetName = props.changeSetName || 'CDK-CodePipeline-ChangeSet';
        const capabilities = cfnCapabilities(props.adminPermissions, props.capabilities);
        this.prepareChangeSetAction = new cpactions.CloudFormationCreateReplaceChangeSetAction({
            actionName: 'ChangeSet',
            changeSetName,
            runOrder: createChangeSetRunOrder,
            stackName: props.stack.stackName,
            templatePath: props.input.atPath(props.stack.templateFile),
            adminPermissions: props.adminPermissions,
            deploymentRole: props.role,
            capabilities,
        });
        this.executeChangeSetAction = new cpactions.CloudFormationExecuteChangeSetAction({
            actionName: 'Execute',
            changeSetName,
            runOrder: executeChangeSetRunOrder,
            stackName: this.stack.stackName,
        });
    }
    bind(scope, stage, options) {
        if (this.stack.environment !== cdk.Stack.of(scope).environment) {
            // FIXME: Add the necessary to extend to stacks in a different account
            throw new Error(`Cross-environment deployment is not supported`);
        }
        stage.addAction(this.prepareChangeSetAction);
        this._deploymentRole = this.prepareChangeSetAction.deploymentRole;
        return this.executeChangeSetAction.bind(scope, stage, options);
    }
    get deploymentRole() {
        if (!this._deploymentRole) {
            throw new Error(`Use this action in a pipeline first before accessing 'deploymentRole'`);
        }
        return this._deploymentRole;
    }
    /**
     * Add policy statements to the role deploying the stack.
     *
     * This role is passed to CloudFormation and must have the IAM permissions
     * necessary to deploy the stack or you can grant this role `adminPermissions`
     * by using that option during creation. If you do not grant
     * `adminPermissions` you need to identify the proper statements to add to
     * this role based on the CloudFormation Resources in your stack.
     */
    addToDeploymentRolePolicy(statement) {
        this.deploymentRole.addToPolicy(statement);
    }
    onStateChange(name, target, options) {
        return this.executeChangeSetAction.onStateChange(name, target, options);
    }
    get actionProperties() {
        return this.executeChangeSetAction.actionProperties;
    }
}
exports.PipelineDeployStackAction = PipelineDeployStackAction;
function cfnCapabilities(adminPermissions, capabilities) {
    if (adminPermissions && capabilities === undefined) {
        // admin true default capability to NamedIAM and AutoExpand
        return [cfn.CloudFormationCapabilities.NAMED_IAM, cfn.CloudFormationCapabilities.AUTO_EXPAND];
    }
    else if (capabilities === undefined) {
        // else capabilities are undefined set AnonymousIAM and AutoExpand
        return [cfn.CloudFormationCapabilities.ANONYMOUS_IAM, cfn.CloudFormationCapabilities.AUTO_EXPAND];
    }
    else {
        // else capabilities are defined use them
        return capabilities;
    }
}
//# sourceMappingURL=data:application/json;base64,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