"use strict";
const assert_1 = require("@aws-cdk/assert");
const iam = require("@aws-cdk/aws-iam");
const kms = require("@aws-cdk/aws-kms");
const cdk = require("@aws-cdk/core");
const sns = require("../lib");
module.exports = {
    'topic tests': {
        'all defaults'(test) {
            const stack = new cdk.Stack();
            new sns.Topic(stack, 'MyTopic');
            assert_1.expect(stack).toMatch({
                'Resources': {
                    'MyTopic86869434': {
                        'Type': 'AWS::SNS::Topic',
                    },
                },
            });
            test.done();
        },
        'specify topicName'(test) {
            const stack = new cdk.Stack();
            new sns.Topic(stack, 'MyTopic', {
                topicName: 'topicName',
            });
            assert_1.expect(stack).toMatch({
                'Resources': {
                    'MyTopic86869434': {
                        'Type': 'AWS::SNS::Topic',
                        'Properties': {
                            'TopicName': 'topicName',
                        },
                    },
                },
            });
            test.done();
        },
        'specify displayName'(test) {
            const stack = new cdk.Stack();
            new sns.Topic(stack, 'MyTopic', {
                displayName: 'displayName',
            });
            assert_1.expect(stack).toMatch({
                'Resources': {
                    'MyTopic86869434': {
                        'Type': 'AWS::SNS::Topic',
                        'Properties': {
                            'DisplayName': 'displayName',
                        },
                    },
                },
            });
            test.done();
        },
        'specify kmsMasterKey'(test) {
            const stack = new cdk.Stack();
            const key = new kms.Key(stack, 'CustomKey');
            new sns.Topic(stack, 'MyTopic', {
                masterKey: key,
            });
            assert_1.expect(stack).to(assert_1.haveResource('AWS::SNS::Topic', {
                'KmsMasterKeyId': { 'Ref': 'CustomKey1E6D0D07' },
            }));
            test.done();
        },
        'specify both'(test) {
            const stack = new cdk.Stack();
            new sns.Topic(stack, 'MyTopic', {
                topicName: 'topicName',
                displayName: 'displayName',
            });
            assert_1.expect(stack).toMatch({
                'Resources': {
                    'MyTopic86869434': {
                        'Type': 'AWS::SNS::Topic',
                        'Properties': {
                            'DisplayName': 'displayName',
                            'TopicName': 'topicName',
                        },
                    },
                },
            });
            test.done();
        },
    },
    'can add a policy to the topic'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const topic = new sns.Topic(stack, 'Topic');
        // WHEN
        topic.addToResourcePolicy(new iam.PolicyStatement({
            resources: ['*'],
            actions: ['sns:*'],
            principals: [new iam.ArnPrincipal('arn')],
        }));
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::SNS::TopicPolicy', {
            PolicyDocument: {
                Version: '2012-10-17',
                Statement: [{
                        'Sid': '0',
                        'Action': 'sns:*',
                        'Effect': 'Allow',
                        'Principal': { 'AWS': 'arn' },
                        'Resource': '*',
                    }],
            },
        }));
        test.done();
    },
    'give publishing permissions'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const topic = new sns.Topic(stack, 'Topic');
        const user = new iam.User(stack, 'User');
        // WHEN
        topic.grantPublish(user);
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::IAM::Policy', {
            'PolicyDocument': {
                Version: '2012-10-17',
                'Statement': [
                    {
                        'Action': 'sns:Publish',
                        'Effect': 'Allow',
                        'Resource': stack.resolve(topic.topicArn),
                    },
                ],
            },
        }));
        test.done();
    },
    'topic resource policy includes unique SIDs'(test) {
        const stack = new cdk.Stack();
        const topic = new sns.Topic(stack, 'MyTopic');
        topic.addToResourcePolicy(new iam.PolicyStatement({ actions: ['service:statement0'] }));
        topic.addToResourcePolicy(new iam.PolicyStatement({ actions: ['service:statement1'] }));
        assert_1.expect(stack).toMatch({
            'Resources': {
                'MyTopic86869434': {
                    'Type': 'AWS::SNS::Topic',
                },
                'MyTopicPolicy12A5EC17': {
                    'Type': 'AWS::SNS::TopicPolicy',
                    'Properties': {
                        'PolicyDocument': {
                            'Statement': [
                                {
                                    'Action': 'service:statement0',
                                    'Effect': 'Allow',
                                    'Sid': '0',
                                },
                                {
                                    'Action': 'service:statement1',
                                    'Effect': 'Allow',
                                    'Sid': '1',
                                },
                            ],
                            'Version': '2012-10-17',
                        },
                        'Topics': [
                            {
                                'Ref': 'MyTopic86869434',
                            },
                        ],
                    },
                },
            },
        });
        test.done();
    },
    'fromTopicArn'(test) {
        // GIVEN
        const stack2 = new cdk.Stack();
        // WHEN
        const imported = sns.Topic.fromTopicArn(stack2, 'Imported', 'arn:aws:sns:*:123456789012:my_corporate_topic');
        // THEN
        test.deepEqual(imported.topicName, 'my_corporate_topic');
        test.deepEqual(imported.topicArn, 'arn:aws:sns:*:123456789012:my_corporate_topic');
        test.done();
    },
    'test metrics'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const topic = new sns.Topic(stack, 'Topic');
        // THEN
        test.deepEqual(stack.resolve(topic.metricNumberOfMessagesPublished()), {
            dimensions: { TopicName: { 'Fn::GetAtt': ['TopicBFC7AF6E', 'TopicName'] } },
            namespace: 'AWS/SNS',
            metricName: 'NumberOfMessagesPublished',
            period: cdk.Duration.minutes(5),
            statistic: 'Sum',
        });
        test.deepEqual(stack.resolve(topic.metricPublishSize()), {
            dimensions: { TopicName: { 'Fn::GetAtt': ['TopicBFC7AF6E', 'TopicName'] } },
            namespace: 'AWS/SNS',
            metricName: 'PublishSize',
            period: cdk.Duration.minutes(5),
            statistic: 'Average',
        });
        test.done();
    },
    'subscription is created under the topic scope by default'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const topic = new sns.Topic(stack, 'Topic');
        // WHEN
        topic.addSubscription({
            bind: () => ({
                protocol: sns.SubscriptionProtocol.HTTP,
                endpoint: 'http://foo/bar',
                subscriberId: 'my-subscription',
            }),
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::SNS::Subscription'));
        test.done();
    },
    'if "scope" is defined, subscription will be created under that scope'(test) {
        // GIVEN
        const app = new cdk.App();
        const stack = new cdk.Stack(app, 'A');
        const stack2 = new cdk.Stack(app, 'B');
        const topic = new sns.Topic(stack, 'Topic');
        // WHEN
        topic.addSubscription({
            bind: () => ({
                protocol: sns.SubscriptionProtocol.HTTP,
                endpoint: 'http://foo/bar',
                subscriberScope: stack2,
                subscriberId: 'subscriberId',
            }),
        });
        // THEN
        assert_1.expect(stack).notTo(assert_1.haveResource('AWS::SNS::Subscription'));
        assert_1.expect(stack2).to(assert_1.haveResource('AWS::SNS::Subscription'));
        test.done();
    },
};
//# sourceMappingURL=data:application/json;base64,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