#
# Copyright © 2012–2022 Michal Čihař <michal@cihar.com>
#
# This file is part of Weblate <https://weblate.org/>
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <https://www.gnu.org/licenses/>.
#

import boto3
from django.conf import settings

from weblate.machinery.base import MachineTranslation


class AWSTranslation(MachineTranslation):
    """AWS machine translation."""

    name = "AWS"
    max_score = 88
    language_map = {
        "zh_Hant": "zh-TW",
        "zh_Hans": "zh",
    }

    def __init__(self):
        super().__init__()
        self.client = boto3.client(
            "translate",
            region_name=settings.MT_AWS_REGION,
            aws_access_key_id=settings.MT_AWS_ACCESS_KEY_ID,
            aws_secret_access_key=settings.MT_AWS_SECRET_ACCESS_KEY,
        )

    def map_language_code(self, code):
        """Convert language to service specific code."""
        return super().map_language_code(code).replace("_", "-").split("@")[0]

    def download_languages(self):
        """
        Hardcoded list of supported languages as there is no API to get this.

        Can be generated by HTML scraping using
        ./scripts/generate-aws-languages
        """
        return (
            "af",
            "am",
            "ar",
            "az",
            "bg",
            "bn",
            "bs",
            "ca",
            "cs",
            "cy",
            "da",
            "de",
            "el",
            "en",
            "es",
            "es-MX",
            "et",
            "fa",
            "fa-AF",
            "fi",
            "fr",
            "fr-CA",
            "gu",
            "ha",
            "he",
            "hi",
            "hr",
            "ht",
            "hu",
            "hy",
            "id",
            "is",
            "it",
            "ja",
            "ka",
            "kk",
            "kn",
            "ko",
            "lt",
            "lv",
            "mk",
            "ml",
            "mn",
            "ms",
            "mt",
            "nl",
            "no",
            "pl",
            "ps",
            "pt",
            "ro",
            "ru",
            "si",
            "sk",
            "sl",
            "so",
            "sq",
            "sr",
            "sv",
            "sw",
            "ta",
            "te",
            "th",
            "tl",
            "tr",
            "uk",
            "ur",
            "uz",
            "vi",
            "zh",
            "zh-TW",
        )

    def download_translations(
        self,
        source,
        language,
        text: str,
        unit,
        user,
        search: bool,
        threshold: int = 75,
    ):
        response = self.client.translate_text(
            Text=text, SourceLanguageCode=source, TargetLanguageCode=language
        )
        yield {
            "text": response["TranslatedText"],
            "quality": self.max_score,
            "service": self.name,
            "source": text,
        }
