"use strict";
var _a, _b;
Object.defineProperty(exports, "__esModule", { value: true });
exports.Job = exports.MetricType = exports.JobState = exports.WorkerType = void 0;
const jsiiDeprecationWarnings = require("../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const cloudwatch = require("aws-cdk-lib/aws-cloudwatch");
const events = require("aws-cdk-lib/aws-events");
const iam = require("aws-cdk-lib/aws-iam");
const s3 = require("aws-cdk-lib/aws-s3");
const cdk = require("aws-cdk-lib");
const _1 = require(".");
const aws_glue_1 = require("aws-cdk-lib/aws-glue");
/**
 * The type of predefined worker that is allocated when a job runs.
 *
 * If you need to use a WorkerType that doesn't exist as a static member, you
 * can instantiate a `WorkerType` object, e.g: `WorkerType.of('other type')`.
 */
class WorkerType {
    constructor(name) {
        this.name = name;
    }
    /**
     * Custom worker type
     * @param workerType custom worker type
     */
    static of(workerType) {
        return new WorkerType(workerType);
    }
}
exports.WorkerType = WorkerType;
_a = JSII_RTTI_SYMBOL_1;
WorkerType[_a] = { fqn: "@aws-cdk/aws-glue-alpha.WorkerType", version: "2.66.0-alpha.0" };
/**
 * Each worker provides 4 vCPU, 16 GB of memory and a 50GB disk, and 2 executors per worker.
 */
WorkerType.STANDARD = new WorkerType('Standard');
/**
 * Each worker maps to 1 DPU (4 vCPU, 16 GB of memory, 64 GB disk), and provides 1 executor per worker. Suitable for memory-intensive jobs.
 */
WorkerType.G_1X = new WorkerType('G.1X');
/**
 * Each worker maps to 2 DPU (8 vCPU, 32 GB of memory, 128 GB disk), and provides 1 executor per worker. Suitable for memory-intensive jobs.
 */
WorkerType.G_2X = new WorkerType('G.2X');
/**
 * Each worker maps to 0.25 DPU (2 vCPU, 4 GB of memory, 64 GB disk), and provides 1 executor per worker. Suitable for low volume streaming jobs.
 */
WorkerType.G_025X = new WorkerType('G.025X');
/**
 * Each worker maps to 2 high-memory DPU [M-DPU] (8 vCPU, 64 GB of memory, 128 GB disk). Supported in Ray jobs.
 */
WorkerType.Z_2X = new WorkerType('Z.2X');
/**
 * Job states emitted by Glue to CloudWatch Events.
 *
 * @see https://docs.aws.amazon.com/AmazonCloudWatch/latest/events/EventTypes.html#glue-event-types for more information.
 */
var JobState;
(function (JobState) {
    /**
     * State indicating job run succeeded
     */
    JobState["SUCCEEDED"] = "SUCCEEDED";
    /**
     * State indicating job run failed
     */
    JobState["FAILED"] = "FAILED";
    /**
     * State indicating job run timed out
     */
    JobState["TIMEOUT"] = "TIMEOUT";
    /**
     * State indicating job is starting
     */
    JobState["STARTING"] = "STARTING";
    /**
     * State indicating job is running
     */
    JobState["RUNNING"] = "RUNNING";
    /**
     * State indicating job is stopping
     */
    JobState["STOPPING"] = "STOPPING";
    /**
     * State indicating job stopped
     */
    JobState["STOPPED"] = "STOPPED";
})(JobState = exports.JobState || (exports.JobState = {}));
/**
 * The Glue CloudWatch metric type.
 *
 * @see https://docs.aws.amazon.com/glue/latest/dg/monitoring-awsglue-with-cloudwatch-metrics.html
 */
var MetricType;
(function (MetricType) {
    /**
     * A value at a point in time.
     */
    MetricType["GAUGE"] = "gauge";
    /**
     * An aggregate number.
     */
    MetricType["COUNT"] = "count";
})(MetricType = exports.MetricType || (exports.MetricType = {}));
class JobBase extends cdk.Resource {
    /**
     * Create a CloudWatch Event Rule for this Glue Job when it's in a given state
     *
     * @param id construct id
     * @param options event options. Note that some values are overridden if provided, these are
     *  - eventPattern.source = ['aws.glue']
     *  - eventPattern.detailType = ['Glue Job State Change', 'Glue Job Run Status']
     *  - eventPattern.detail.jobName = [this.jobName]
     *
     * @see https://docs.aws.amazon.com/AmazonCloudWatch/latest/events/EventTypes.html#glue-event-types
     */
    onEvent(id, options = {}) {
        const rule = new events.Rule(this, id, options);
        rule.addTarget(options.target);
        rule.addEventPattern({
            source: ['aws.glue'],
            detailType: ['Glue Job State Change', 'Glue Job Run Status'],
            detail: {
                jobName: [this.jobName],
            },
        });
        return rule;
    }
    /**
     * Create a CloudWatch Event Rule for the transition into the input jobState.
     *
     * @param id construct id.
     * @param jobState the job state.
     * @param options optional event options.
     */
    onStateChange(id, jobState, options = {}) {
        const rule = this.onEvent(id, {
            description: `Rule triggered when Glue job ${this.jobName} is in ${jobState} state`,
            ...options,
        });
        rule.addEventPattern({
            detail: {
                state: [jobState],
            },
        });
        return rule;
    }
    /**
     * Create a CloudWatch Event Rule matching JobState.SUCCEEDED.
     *
     * @param id construct id.
     * @param options optional event options. default is {}.
     */
    onSuccess(id, options = {}) {
        return this.onStateChange(id, JobState.SUCCEEDED, options);
    }
    /**
     * Return a CloudWatch Event Rule matching FAILED state.
     *
     * @param id construct id.
     * @param options optional event options. default is {}.
     */
    onFailure(id, options = {}) {
        return this.onStateChange(id, JobState.FAILED, options);
    }
    /**
     * Return a CloudWatch Event Rule matching TIMEOUT state.
     *
     * @param id construct id.
     * @param options optional event options. default is {}.
     */
    onTimeout(id, options = {}) {
        return this.onStateChange(id, JobState.TIMEOUT, options);
    }
    /**
     * Create a CloudWatch metric.
     *
     * @param metricName name of the metric typically prefixed with `glue.driver.`, `glue.<executorId>.` or `glue.ALL.`.
     * @param type the metric type.
     * @param props metric options.
     *
     * @see https://docs.aws.amazon.com/glue/latest/dg/monitoring-awsglue-with-cloudwatch-metrics.html
     */
    metric(metricName, type, props) {
        return new cloudwatch.Metric({
            metricName,
            namespace: 'Glue',
            dimensionsMap: {
                JobName: this.jobName,
                JobRunId: 'ALL',
                Type: type,
            },
            ...props,
        }).attachTo(this);
    }
    /**
     * Return a CloudWatch Metric indicating job success.
     *
     * This metric is based on the Rule returned by no-args onSuccess() call.
     */
    metricSuccess(props) {
        return metricRule(this.metricJobStateRule('SuccessMetricRule', JobState.SUCCEEDED), props);
    }
    /**
     * Return a CloudWatch Metric indicating job failure.
     *
     * This metric is based on the Rule returned by no-args onFailure() call.
     */
    metricFailure(props) {
        return metricRule(this.metricJobStateRule('FailureMetricRule', JobState.FAILED), props);
    }
    /**
     * Return a CloudWatch Metric indicating job timeout.
     *
     * This metric is based on the Rule returned by no-args onTimeout() call.
     */
    metricTimeout(props) {
        return metricRule(this.metricJobStateRule('TimeoutMetricRule', JobState.TIMEOUT), props);
    }
    /**
     * Creates or retrieves a singleton event rule for the input job state for use with the metric JobState methods.
     *
     * @param id construct id.
     * @param jobState the job state.
     * @private
     */
    metricJobStateRule(id, jobState) {
        return this.node.tryFindChild(id) ?? this.onStateChange(id, jobState);
    }
}
/**
 * A Glue Job.
 */
class Job extends JobBase {
    constructor(scope, id, props) {
        super(scope, id, {
            physicalName: props.jobName,
        });
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_glue_alpha_JobProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, Job);
            }
            throw error;
        }
        const executable = props.executable.bind();
        this.role = props.role ?? new iam.Role(this, 'ServiceRole', {
            assumedBy: new iam.ServicePrincipal('glue.amazonaws.com'),
            managedPolicies: [iam.ManagedPolicy.fromAwsManagedPolicyName('service-role/AWSGlueServiceRole')],
        });
        this.grantPrincipal = this.role;
        const sparkUI = props.sparkUI?.enabled ? this.setupSparkUI(executable, this.role, props.sparkUI) : undefined;
        this.sparkUILoggingLocation = sparkUI?.location;
        const continuousLoggingArgs = props.continuousLogging?.enabled ? this.setupContinuousLogging(this.role, props.continuousLogging) : {};
        const profilingMetricsArgs = props.enableProfilingMetrics ? { '--enable-metrics': '' } : {};
        const defaultArguments = {
            ...this.executableArguments(executable),
            ...continuousLoggingArgs,
            ...profilingMetricsArgs,
            ...sparkUI?.args,
            ...this.checkNoReservedArgs(props.defaultArguments),
        };
        const jobResource = new aws_glue_1.CfnJob(this, 'Resource', {
            name: props.jobName,
            description: props.description,
            role: this.role.roleArn,
            command: {
                name: executable.type.name,
                scriptLocation: this.codeS3ObjectUrl(executable.script),
                pythonVersion: executable.pythonVersion,
            },
            glueVersion: executable.glueVersion.name,
            workerType: props.workerType?.name,
            numberOfWorkers: props.workerCount,
            maxCapacity: props.maxCapacity,
            maxRetries: props.maxRetries,
            executionProperty: props.maxConcurrentRuns ? { maxConcurrentRuns: props.maxConcurrentRuns } : undefined,
            notificationProperty: props.notifyDelayAfter ? { notifyDelayAfter: props.notifyDelayAfter.toMinutes() } : undefined,
            timeout: props.timeout?.toMinutes(),
            connections: props.connections ? { connections: props.connections.map((connection) => connection.connectionName) } : undefined,
            securityConfiguration: props.securityConfiguration?.securityConfigurationName,
            tags: props.tags,
            defaultArguments,
        });
        const resourceName = this.getResourceNameAttribute(jobResource.ref);
        this.jobArn = jobArn(this, resourceName);
        this.jobName = resourceName;
    }
    /**
     * Creates a Glue Job
     *
     * @param scope The scope creating construct (usually `this`).
     * @param id The construct's id.
     * @param attrs Import attributes
     */
    static fromJobAttributes(scope, id, attrs) {
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_glue_alpha_JobAttributes(attrs);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.fromJobAttributes);
            }
            throw error;
        }
        class Import extends JobBase {
            constructor() {
                super(...arguments);
                this.jobName = attrs.jobName;
                this.jobArn = jobArn(scope, attrs.jobName);
                this.grantPrincipal = attrs.role ?? new iam.UnknownPrincipal({ resource: this });
            }
        }
        return new Import(scope, id);
    }
    /**
     * Check no usage of reserved arguments.
     *
     * @see https://docs.aws.amazon.com/glue/latest/dg/aws-glue-programming-etl-glue-arguments.html
     */
    checkNoReservedArgs(defaultArguments) {
        if (defaultArguments) {
            const reservedArgs = new Set(['--debug', '--mode', '--JOB_NAME']);
            Object.keys(defaultArguments).forEach((arg) => {
                if (reservedArgs.has(arg)) {
                    throw new Error(`The ${arg} argument is reserved by Glue. Don't set it`);
                }
            });
        }
        return defaultArguments;
    }
    executableArguments(config) {
        const args = {};
        args['--job-language'] = config.language;
        if (config.className) {
            args['--class'] = config.className;
        }
        if (config.extraJars && config.extraJars?.length > 0) {
            args['--extra-jars'] = config.extraJars.map(code => this.codeS3ObjectUrl(code)).join(',');
        }
        if (config.extraPythonFiles && config.extraPythonFiles.length > 0) {
            args['--extra-py-files'] = config.extraPythonFiles.map(code => this.codeS3ObjectUrl(code)).join(',');
        }
        if (config.extraFiles && config.extraFiles.length > 0) {
            args['--extra-files'] = config.extraFiles.map(code => this.codeS3ObjectUrl(code)).join(',');
        }
        if (config.extraJarsFirst) {
            args['--user-jars-first'] = 'true';
        }
        return args;
    }
    setupSparkUI(executable, role, props) {
        if (_1.JobType.PYTHON_SHELL === executable.type) {
            throw new Error('Spark UI is not available for JobType.PYTHON_SHELL jobs');
        }
        else if (_1.JobType.RAY === executable.type) {
            throw new Error('Spark UI is not available for JobType.RAY jobs');
        }
        const bucket = props.bucket ?? new s3.Bucket(this, 'SparkUIBucket');
        bucket.grantReadWrite(role);
        const args = {
            '--enable-spark-ui': 'true',
            '--spark-event-logs-path': bucket.s3UrlForObject(props.prefix),
        };
        return {
            location: {
                prefix: props.prefix,
                bucket,
            },
            args,
        };
    }
    setupContinuousLogging(role, props) {
        const args = {
            '--enable-continuous-cloudwatch-log': 'true',
            '--enable-continuous-log-filter': (props.quiet ?? true).toString(),
        };
        if (props.logGroup) {
            args['--continuous-log-logGroup'] = props.logGroup.logGroupName;
            props.logGroup.grantWrite(role);
        }
        if (props.logStreamPrefix) {
            args['--continuous-log-logStreamPrefix'] = props.logStreamPrefix;
        }
        if (props.conversionPattern) {
            args['--continuous-log-conversionPattern'] = props.conversionPattern;
        }
        return args;
    }
    codeS3ObjectUrl(code) {
        const s3Location = code.bind(this, this.role).s3Location;
        return `s3://${s3Location.bucketName}/${s3Location.objectKey}`;
    }
}
exports.Job = Job;
_b = JSII_RTTI_SYMBOL_1;
Job[_b] = { fqn: "@aws-cdk/aws-glue-alpha.Job", version: "2.66.0-alpha.0" };
/**
 * Create a CloudWatch Metric that's based on Glue Job events
 * {@see https://docs.aws.amazon.com/AmazonCloudWatch/latest/events/EventTypes.html#glue-event-types}
 * The metric has namespace = 'AWS/Events', metricName = 'TriggeredRules' and RuleName = rule.ruleName dimension.
 *
 * @param rule for use in setting RuleName dimension value
 * @param props metric properties
 */
function metricRule(rule, props) {
    return new cloudwatch.Metric({
        namespace: 'AWS/Events',
        metricName: 'TriggeredRules',
        dimensionsMap: { RuleName: rule.ruleName },
        statistic: cloudwatch.Statistic.SUM,
        ...props,
    }).attachTo(rule);
}
/**
 * Returns the job arn
 * @param scope
 * @param jobName
 */
function jobArn(scope, jobName) {
    return cdk.Stack.of(scope).formatArn({
        service: 'glue',
        resource: 'job',
        resourceName: jobName,
    });
}
//# sourceMappingURL=data:application/json;base64,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