from typing import Any, Dict, List

from globus_sdk.exc import GlobusAPIError


class TimerAPIError(GlobusAPIError):
    """
    Error class to represent error responses from Timer.

    Has no particular additions to the base ``GlobusAPIError``, but implements a
    different method for parsing error responses from Timer due to the differences
    between pydantic-generated errors and other errors.
    """

    MESSAGE_FIELDS: List[str] = []  # we are overriding `_load_from_json` instead

    def _load_from_json(self, data: Dict[str, Any]) -> None:
        """
        Errors generated by Timer itself look like this:

        .. code-block:: json

            {
                "error": {
                    "code": "ERROR",
                    "detail": "Request failed successfully",
                    "status": 500
                }
            }


        While errors from Timer's pydantic implementation look like this:

        .. code-block:: json

            {
                "detail": [
                    {
                        "loc": ["body", "start"],
                        "msg": "field required",
                        "type": "value_error.missing"
                    },
                    {
                        "loc": ["body", "callback_url"],
                        "msg": "field required",
                        "type": "value_error.missing"
                    },
                    ...

        The ``msg`` might be like the above, or "value is not a valid <TYPE>", and so
        on.  ``loc`` however is always a list which indicates the location of the error
        within the request. It might say ``["body", "start"]`` for example, or it might
        start with "query" to point to a value in the query string.
        """
        if "error" in data:
            self.code = data["error"].get("code", "ERROR")
            self.message = data["error"].get("detail", "")
        elif "detail" in data:
            self.code = "Validation Error"
            self.message = "; ".join(
                e["msg"] + ": " + ".".join(k for k in e["loc"]) for e in data["detail"]
            )
        else:
            self.code = "Unknown Error"
            self.message = "Could not parse error details from the response"
