# AutoProperty library

- [What it used for](#What-it-used-for)
- [Why you need it?](#why-you-need-it)
- [Installation](#installation)
- [Requirements](#requirements)
- [How does this work?](#how-does-this-work)
- [Quick start](#quick-start)
    - [Syntax](#syntax)
    - [Annotations](#annotations)
- [Full example](#full-example)
- [Documentation](#documentation)
- [FAQ](#faq)
- [Known problems](#known-problems)
- [Perfomance](#performance)
- [Plans](#plans)
- [Feedback](#feedback)

## What it used for?

This open source library is used for creating autogenerated properties with type validation.  

I think it is very comfy to use and less code to write.  

I just have enough of writting millions and trillions of "@property" decorators and setters for them. It also have type validation, as a bonus.

## Why you need it?

This library provides ***more readability*** to the code, ***decreasing amount of lines*** and ***preventing from violation*** of the principles of object-oriented programming (**DRY**).

## Installation

To install this lib run this line in your terminal/console:
```bash
pip install autoproperty
```

## Requirements

The only it one dependency is pydantic library. If pip didnt install by itself then run this command in your env:

```bash
pip install pydantic
``` 

## How does this work?

Basically it is just a common **property** but with autogenerated getters and setters (no deleters for now, will be added later) like in Csharp but without access modifiers. I tried to achieve the better speed perfomance as the basic solution.  

It also has builtin field validation from pydantic but you can turn it off (turned on by default) and save some time on the runtime.

## Quick start

### Syntax

Base syntax.

```python
class Exmpl:
    @AutoProperty[int] # <- (Optional) generic for IDE syntax highlighting
    def some_prop(self) -> int: ... # <- not have to implement!
                          # ^ type annotation for checking object in setter
                          # and type annotation

obj = Exmpl()
obj.some_prop = 42
```

### Annotations

You **have to** add at **least one** type annotation to any of these **three places** (If you didnt turned off type validation). Be careful to **not mix them up**. They all have to **be the same**, otherwise will raise **an error**.

```python
class Exmpl:
    _some_prop: int # <- one

    @AutoProperty(annotation_type=int) # <- two
    def some_prop(self) -> int: ...
                          # ^ three
```


## Full example

```python
from autoproperty import AutoProperty


class Point:
    def __init__(self, x: int, y: int):
        self.X = x
        self.Y = y

    @AutoProperty[int]
    def X(self) -> int: ...

    @AutoProperty[int]
    def Y(self) -> int: ...

    def __repr__(self) -> str:
        return f"[{self.X};{self.Y}]"


myPointOne = Point(2, 6)

print(myPointOne.X) # 2

print(myPointOne) # [2; 6]
```

## Documentation

More info about syntax and options you can find in docs folder. Here is a [base example](autoproperty/docs/basic_example.md).

## FAQ

- > How to install this library?  

    Just run `pip install autoproperty`.

- > How much slower is the basic auto-property class compared to the basic solution?

    It's not much slower, the basic autoproperty class is only a few times slower than the basic solution, but there's a nanosecond count going on. For more details check [Perfomance](#performance).

- > Is lightweight version of autoproperty is available on all platforms?

    Not yet. Now it is only available on linux and mac because I dont have windows machine to compile the windows version. But I will add it ASAP.

- > Do this lib have a lot of dependencies?

    No. There is only one dependency: pydantic. Even this one i would make optional if user does not intend to use field validation.

- > Can I use this lib in production?

    It is hard to give a solid answer. I would prefer anyone can use this in production. But I do have a low amount of experience in such things. It need some tests and professional opinion to be sure for 100%. So **yes** you can, but be careful.

- > How often updates will come out?

    As more as I have powers for contributing. I do work alone, no sponsors, all on my own. If this project will like some people, even a couple, of course I would update it more often.

## Known problems

- Not tested yet with classmethods or staticmethods, only bound methods, but it should work as well.
- No windows version of LightProperty

## Performance

First of all, I want to note that the basic property solution in Python is 447% faster for the getter and 265% faster for the setter than the basic auto property from this library.

However, do not jump to conclusions. First of all, you should consider the convenience and readability of the code, and only then the performance. For high-load projects, where every nanosecond is important, a lightweight version of auto-action has been created. Its speed is impressive: by 414% for the getter and by 366% for the setter, which surpasses the basic auto performance.

Thus, lightweight autoproperty turns out to be faster than autoproperty and the lightweight autoproperty setter is even faster than in the basic Python implementation.

You can test it out on your own PC. Just launch test_time_comparing.py from /test/. 

There is result of tests:

```
autoproperty getter time:  204.9903589429996
autoproperty setter time:  187.70526755399987

light autoproperty getter time:  49.50247659899833
light autoproperty setter time:  51.2770686570002

basic property getter time 45.86868843100092
basic property setter time 70.93056209800125

diff between getters of autoproperty and it light version 4.141012188209338
diff between setters of autoproperty and it light version 3.6606083863644354

diff between getters of autoproperty and basic solution 4.469069553871401
diff between setters of autoproperty and basic solution 2.6463242642100706

Lets find out who is faster. More percent - faster
getters
If AutoProperty result = 100%, then LightAutoProperty = 414%, and basic property = 447%

setters
If AutoProperty result = 100%, then LightAutoProperty = 366%, and basic property = 265%
```

## Plans

Currently I'm planning to add these features to the lib:

- The opportunity to add your own validation.
- The opportunity to add handlers to "set" and "get" events.
- Modificators like "read-only"
- The opportunity to add your own getter and setter using protocols

## Feedback

If you want to ask me something, you have solution one of the above problems, you have an offer to me or any other reason, please open an issue or message me via email alecsw86@gmail.com.