#include <cassert>
#include <cmath>
#include <iostream>

#include <dune/fem/quadrature/pardgsimplexquadrature.hh>

namespace Dune {
namespace Fem {
namespace ParDGSimplexQuadrature
{

// 2d quadrature rules for the reference triangle [(0,0), (1,0), (0,1)]
//
// naming convention of quad2d_x: x denotes the order of the formula
//                                i.e. the maximum degree of polynimials
//                                that is integrated exact by the formula.
//
// exception quad2d_0: this is a dummy that has no points



// return a quadrature formula that has at least degree minimum_degree
template<>
const Quadrature2d& Quadrature2d::quadrature(int minimum_degree)
{
  if (minimum_degree < 0) return quad2d_0;

  switch (minimum_degree){
  case 0:
  case 1: return quad2d_1; break;
  case 2: return quad2d_2; break;
  // for order=3 we user the order=4 quad
  case 3: //return quad2d_3; break;
  case 4: return quad2d_4; break;
  case 5: return quad2d_5; break;
  case 6: return quad2d_6; break;
  case 7: return quad2d_7; break;
  case 8: return quad2d_8; break;
  case 9: return quad2d_9; break;
  case 10: return quad2d_10; break;
  case 11: return quad2d_11; break;
  case 12:
  case 13: return quad2d_13; break;
  default:
       {
         std::cerr << "Quadrature2d::quadrature: order " << minimum_degree << " not avaiable" << std::endl;
         assert(0);
         abort();
       }
  }

  return quad2d_0;
}



template<>
void Quadrature2d::check() const
{
  double total_error = 0.0;

  for(int m=0; m<=degree; m++){
    for(int n=0; n<=m; n++){
      int k = m-n;
      double sum = 0.0;
      for(int r=0; r<nop; r++){
  double x_n = 1.0;
  double y_k = 1.0;
  for(int i=1; i<=n; i++) x_n *= x(r)[0];
  for(int i=1; i<=k; i++) y_k *= x(r)[1];
  sum += w(r) * x_n * y_k;
      }

      long long a = 1;
      long long b = 1;
      for(int i=1; i<=n; i++) a *= i;
      for(int i=1; i<=k; i++) a *= i;
      for(int i=1; i<=n+k+2; i++) b *= i;

      double error = std::abs(sum - (double)a / (double)b);
      total_error += error;

      std::cout << n << "  "
    << k << "  "
    << "   error: " << error << "  "
    << std::endl;
    }
  }

  std::cout << "total error: " << total_error << std::endl;
}



// dummy
static const double quad2d_0_x[][3] = {{}};
const Quadrature2d quad2d_0(0, 0, quad2d_0_x);



// Quadrature rules for the reference triangle
//
// taken from
//
// Dunavant, D.A.
// High degree efficient symmetrical Gaussian quadrature rules for
// the triangle. (English)
// Int. J. Numer. Methods Eng. 21, 1129-1148 (1985)

static const double quad2d_1_x[][3] =
  {{1.0/3.0, 1.0/3.0,  0.5}
  };
const Quadrature2d quad2d_1(1, 1, quad2d_1_x);


static const double quad2d_2_x[][3] =
  {{1.0/6.0, 1.0/6.0,  1.0/6.0},
   {2.0/3.0, 1.0/6.0,  1.0/6.0},
   {1.0/6.0, 2.0/3.0,  1.0/6.0},
  };
const Quadrature2d quad2d_2(3, 2, quad2d_2_x);


// NOTE: this quadrature has negative weights
// which leads to nasty bugs when using higher order Lagrange elements
static const double quad2d_3_x[][3] =
  {{1.0/3.0, 1.0/3.0,   -9.0/32.0},
   {1.0/5.0, 1.0/5.0,   25.0/96.0},
   {3.0/5.0, 1.0/5.0,   25.0/96.0},
   {1.0/5.0, 3.0/5.0,   25.0/96.0},
  };
const Quadrature2d quad2d_3(4, 3, quad2d_3_x);


static const double quad2d_4_x[][3] =
  {{0.108103018168070, 0.445948490915965,   0.223381589678011*0.5},
   {0.445948490915965, 0.108103018168070,   0.223381589678011*0.5},
   {0.445948490915965, 0.445948490915965,   0.223381589678011*0.5},
   {0.816847572980459, 0.091576213509771,   0.109951743655322*0.5},
   {0.091576213509771, 0.816847572980459,   0.109951743655322*0.5},
   {0.091576213509771, 0.091576213509771,   0.109951743655322*0.5}
  };
const Quadrature2d quad2d_4(6, 4, quad2d_4_x);


static const double quad2d_5_x[][3] =
  {{1.0/3.0, 1.0/3.0,
    9.0/80.0},
   {(6.0+sqrt(15.0))/21.0, (6.0+sqrt(15.0))/21.0,
    (155.0+sqrt(15.0))/2400.0},
   {(9.0-2.0*sqrt(15.0))/21.0,(6.0+sqrt(15.0))/21.0,
    (155.0+sqrt(15.0))/2400.0},
   {(6.0+sqrt(15.0))/21.0,(9.0-2.0*sqrt(15.0))/21.0,
    (155.0+sqrt(15.0))/2400.0},
   {(6.0-sqrt(15.0))/21.0, (6.0-sqrt(15.0))/21.0,
    (155.0-sqrt(15.0))/2400.0},
   {(9.0+2.0*sqrt(15.0))/21.0, (6.0-sqrt(15.0))/21.0,
    (155.0-sqrt(15.0))/2400.0},
   {(6.0-sqrt(15.0))/21.0, (9.0+2.0*sqrt(15.0))/21.0,
    (155.0-sqrt(15.0))/2400.0}
  };
const Quadrature2d quad2d_5(7, 5, quad2d_5_x);



static const double quad2d_6_x[][3] =
  {{0.501426509658179, 0.249286745170910,   0.116786275726379*0.5}, // w0
   {0.249286745170910, 0.501426509658179,   0.116786275726379*0.5},
   {0.249286745170910, 0.249286745170910,   0.116786275726379*0.5},
   {0.873821971016996, 0.063089014491502,   0.050844906370207*0.5}, // w1
   {0.063089014491502, 0.873821971016996,   0.050844906370207*0.5},
   {0.063089014491502, 0.063089014491502,   0.050844906370207*0.5},
   {0.053145049844817, 0.310352451033784,   0.082851075618374*0.5}, // w2
   {0.053145049844817, 0.636502499121399,   0.082851075618374*0.5},
   {0.310352451033784, 0.053145049844817,   0.082851075618374*0.5},
   {0.310352451033784, 0.636502499121399,   0.082851075618374*0.5},
   {0.636502499121399, 0.053145049844817,   0.082851075618374*0.5},
   {0.636502499121399, 0.310352451033784,   0.082851075618374*0.5}
  };
const Quadrature2d quad2d_6(12, 6, quad2d_6_x);




static const double quad2d_7_x[][3] =
  {{1.0/3.0, 1.0/3.0,   -0.149570044467682*0.5}, // w0
   {0.479308067841920, 0.260345966079040,   0.175615257433208*0.5}, // w1
   {0.260345966079040, 0.479308067841920,   0.175615257433208*0.5},
   {0.260345966079040, 0.260345966079040,   0.175615257433208*0.5},
   {0.869739794195568, 0.065130102902216,   0.053347235608838*0.5}, // w2
   {0.065130102902216, 0.869739794195568,   0.053347235608838*0.5},
   {0.065130102902216, 0.065130102902216,   0.053347235608838*0.5},
   {0.048690315425316, 0.312865496004874,   0.077113760890257*0.5}, // w3
   {0.048690315425316, 0.638444188569810,   0.077113760890257*0.5},
   {0.312865496004874, 0.048690315425316,   0.077113760890257*0.5},
   {0.312865496004874, 0.638444188569810,   0.077113760890257*0.5},
   {0.638444188569810, 0.048690315425316,   0.077113760890257*0.5},
   {0.638444188569810, 0.312865496004874,   0.077113760890257*0.5}
  };
const Quadrature2d quad2d_7(13, 7, quad2d_7_x);



static const double quad2d_8_x[][3] =
  {{1.0/3.0, 1.0/3.0,   0.144315607677787*0.5}, // w0
   {0.081414823414554, 0.459292588292723,   0.095091634267285*0.5}, // w1
   {0.459292588292723, 0.081414823414554,   0.095091634267285*0.5},
   {0.459292588292723, 0.459292588292723,   0.095091634267285*0.5},
   {0.658861384496480, 0.170569307751760,   0.103217370534718*0.5}, // w2
   {0.170569307751760, 0.658861384496480,   0.103217370534718*0.5},
   {0.170569307751760, 0.170569307751760,   0.103217370534718*0.5},
   {0.898905543365938, 0.050547228317031,   0.032458497623198*0.5}, // w3
   {0.050547228317031, 0.898905543365938,   0.032458497623198*0.5},
   {0.050547228317031, 0.050547228317031,   0.032458497623198*0.5},
   {0.008394777409958, 0.263112829634638,   0.027230314174435*0.5}, // w4
   {0.008394777409958, 0.728492392955404,   0.027230314174435*0.5},
   {0.263112829634638, 0.008394777409958,   0.027230314174435*0.5},
   {0.263112829634638, 0.728492392955404,   0.027230314174435*0.5},
   {0.728492392955404, 0.008394777409958,   0.027230314174435*0.5},
   {0.728492392955404, 0.263112829634638,   0.027230314174435*0.5}
  };
const Quadrature2d quad2d_8(16, 8, quad2d_8_x);



static const double quad2d_9_x[][3] =
  {{1.0/3.0, 1.0/3.0,   0.097135796282799*0.5}, // w0
   {0.020634961602525, 0.489682519198738,   0.031334700227139*0.5}, // w1
   {0.489682519198738, 0.020634961602525,   0.031334700227139*0.5},
   {0.489682519198738, 0.489682519198738,   0.031334700227139*0.5},
   {0.125820817014127, 0.437089591492937,   0.077827541004774*0.5}, // w2
   {0.437089591492937, 0.125820817014127,   0.077827541004774*0.5},
   {0.437089591492937, 0.437089591492937,   0.077827541004774*0.5},
   {0.623592928761935, 0.188203535619033,   0.079647738927210*0.5}, // w3
   {0.188203535619033, 0.623592928761935,   0.079647738927210*0.5},
   {0.188203535619033, 0.188203535619033,   0.079647738927210*0.5},
   {0.910540973211095, 0.044729513394453,   0.025577675658698*0.5}, // w4
   {0.044729513394453, 0.910540973211095,   0.025577675658698*0.5},
   {0.044729513394453, 0.044729513394453,   0.025577675658698*0.5},
   {0.036838412054736, 0.221962989160766,   0.043283539377289*0.5}, // w5
   {0.036838412054736, 0.741198598784498,   0.043283539377289*0.5},
   {0.221962989160766, 0.036838412054736,   0.043283539377289*0.5},
   {0.221962989160766, 0.741198598784498,   0.043283539377289*0.5},
   {0.741198598784498, 0.036838412054736,   0.043283539377289*0.5},
   {0.741198598784498, 0.221962989160766,   0.043283539377289*0.5}
  };
const Quadrature2d quad2d_9(19, 9, quad2d_9_x);




static const double quad2d_10_x[][3] =
  {{1.0/3.0, 1.0/3.0,   0.090817990382754*0.5}, // w0
   {0.028844733232685, 0.485577633383657,   0.036725957756467*0.5}, // w1
   {0.485577633383657, 0.028844733232685,   0.036725957756467*0.5},
   {0.485577633383657, 0.485577633383657,   0.036725957756467*0.5},
   {0.781036849029926, 0.109481575485037,   0.045321059435528*0.5}, // w2
   {0.109481575485037, 0.781036849029926,   0.045321059435528*0.5},
   {0.109481575485037, 0.109481575485037,   0.045321059435528*0.5},
   {0.141707219414880, 0.307939838764121,   0.072757916845420*0.5}, // w3
   {0.141707219414880, 0.550352941820999,   0.072757916845420*0.5},
   {0.307939838764121, 0.141707219414880,   0.072757916845420*0.5},
   {0.307939838764121, 0.550352941820999,   0.072757916845420*0.5},
   {0.550352941820999, 0.141707219414880,   0.072757916845420*0.5},
   {0.550352941820999, 0.307939838764121,   0.072757916845420*0.5},
   {0.025003534762686, 0.246672560639903,   0.028327242531057*0.5}, // w4
   {0.025003534762686, 0.728323904597411,   0.028327242531057*0.5},
   {0.246672560639903, 0.025003534762686,   0.028327242531057*0.5},
   {0.246672560639903, 0.728323904597411,   0.028327242531057*0.5},
   {0.728323904597411, 0.025003534762686,   0.028327242531057*0.5},
   {0.728323904597411, 0.246672560639903,   0.028327242531057*0.5},
   {0.009540815400299, 0.066803251012200,   0.009421666963733*0.5}, // w5
   {0.009540815400299, 0.923655933587500,   0.009421666963733*0.5},
   {0.066803251012200, 0.009540815400299,   0.009421666963733*0.5},
   {0.066803251012200, 0.923655933587500,   0.009421666963733*0.5},
   {0.923655933587500, 0.009540815400299,   0.009421666963733*0.5},
   {0.923655933587500, 0.066803251012200,   0.009421666963733*0.5}
  };
const Quadrature2d quad2d_10(25, 10, quad2d_10_x);



// note: has negative weights
//       has negative baryzentric coordinates
static const double quad2d_11_x[][3] =
  {{0.935270103777448214, 0.032364948111275893,   6.8298655013389309e-3}, // w0
   {0.032364948111275893, 0.935270103777448214,   6.8298655013389309e-3},
   {0.032364948111275893, 0.032364948111275893,   6.8298655013389309e-3},
   {0.76129817543483740, 0.11935091228258130,     0.018092270251709039}, // w1
   {0.11935091228258130, 0.76129817543483740,     0.018092270251709039},
   {0.11935091228258130, 0.11935091228258130,     0.018092270251709039},
   {-0.06922209654151660, 0.53461104827075830,    4.6350316448033802e-4}, // w2
   {0.53461104827075830, -0.06922209654151660,    4.6350316448033802e-4},
   {0.53461104827075830, 0.53461104827075830,     4.6350316448033802e-4},
   {0.59338019913743506, 0.20330990043128247,     0.029661488690387036}, // w3
   {0.20330990043128247, 0.59338019913743506,     0.029661488690387036},
   {0.20330990043128247, 0.20330990043128247,     0.029661488690387036},
   {0.20206139406828956, 0.39896930296585522,     0.038574767457406561}, // w4
   {0.39896930296585522, 0.20206139406828956,     0.038574767457406561},
   {0.39896930296585522, 0.39896930296585522,     0.038574767457406561},
   {0.59320121342821275, 0.050178138310494665,    0.026168555981102035}, // w5
   {0.59320121342821275, 0.356620648261292585,    0.026168555981102035},
   {0.050178138310494665, 0.59320121342821275,    0.026168555981102035},
   {0.050178138310494665, 0.356620648261292585,   0.026168555981102035},
   {0.356620648261292585, 0.59320121342821275,    0.026168555981102035},
   {0.356620648261292585, 0.050178138310494665,   0.026168555981102035},
   {0.80748900315979215, 0.021022016536166297,    0.010353829819570344}, // w6
   {0.80748900315979215, 0.171488980304041553,    0.010353829819570344},
   {0.021022016536166297, 0.80748900315979215,    0.010353829819570344},
   {0.021022016536166297, 0.171488980304041553,   0.010353829819570344},
   {0.171488980304041553, 0.80748900315979215,    0.010353829819570344},
   {0.171488980304041553, 0.021022016536166297,   0.010353829819570344}
  };
const Quadrature2d quad2d_11(27, 11, quad2d_11_x);





static const double quad2d_13_x[][3] =
  {{1.0/3.0, 1.0/3.0,    0.029348039806359515}, // w0
   {0.5, 0.5,            2.6784518955454304e-3}, // w1
   {0.5, 0.0,            2.6784518955454304e-3},
   {0.0, 0.5,            2.6784518955454304e-3},
   {0.950785622713539564, 0.024607188643230218,   3.9253841480500401e-3}, // w2
   {0.024607188643230218, 0.950785622713539564,   3.9253841480500401e-3},
   {0.024607188643230218, 0.024607188643230218,   3.9253841480500401e-3},
   {0.15938249379761064, 0.42030875310119468,     0.025334476587943481}, // w3
   {0.42030875310119468, 0.15938249379761064,     0.025334476587943481},
   {0.42030875310119468, 0.42030875310119468,     0.025334476587943481},
   {0.54419948898767878, 0.22790025550616061,     0.025040163045254533}, // w4
   {0.22790025550616061, 0.54419948898767878,     0.025040163045254533},
   {0.22790025550616061, 0.22790025550616061,     0.025040163045254533},
   {0.76757388223296420, 0.11621305888351790,     0.015823557296149159}, // w5
   {0.11621305888351790, 0.76757388223296420,     0.015823557296149159},
   {0.11621305888351790, 0.11621305888351790,     0.015823557296149159},
   {0.04679403990184170, 0.47660298004907915,     0.015746281537984397}, // w6
   {0.47660298004907915, 0.04679403990184170,     0.015746281537984397},
   {0.47660298004907915, 0.47660298004907915,     0.015746281537984397},
   {0.85177558714541046, 0.022797894538248612,    7.9012661076303756e-3}, // w7
   {0.85177558714541046, 0.125426518316340928,    7.9012661076303756e-3},
   {0.022797894538248612, 0.85177558714541046,    7.9012661076303756e-3},
   {0.022797894538248612, 0.125426518316340928,   7.9012661076303756e-3},
   {0.125426518316340928, 0.85177558714541046,    7.9012661076303756e-3},
   {0.125426518316340928, 0.022797894538248612,   7.9012661076303756e-3},
   {0.69279731756666085, 0.016275770991088540,    7.9908188904642026e-3}, // w8
   {0.69279731756666085, 0.290926911442250610,    7.9908188904642026e-3},
   {0.016275770991088540, 0.69279731756666085,    7.9908188904642026e-3},
   {0.016275770991088540, 0.290926911442250610,   7.9908188904642026e-3},
   {0.290926911442250610, 0.69279731756666085,    7.9908188904642026e-3},
   {0.290926911442250610, 0.016275770991088540,   7.9908188904642026e-3},
   {0.63795588386420953, 0.089733060451605359,    0.018275751112048647}, // w9
   {0.63795588386420953, 0.272311055684185111,    0.018275751112048647},
   {0.089733060451605359, 0.63795588386420953,    0.018275751112048647},
   {0.089733060451605359, 0.272311055684185111,   0.018275751112048647},
   {0.272311055684185111, 0.63795588386420953,    0.018275751112048647},
   {0.272311055684185111, 0.089733060451605359,   0.018275751112048647}
  };
const Quadrature2d quad2d_13(37, 13, quad2d_13_x);


} // ParDGSimplexQuadrature
} // Fem
} // Dune
