"use strict";
const assert_1 = require("@aws-cdk/assert");
const lambda = require("@aws-cdk/aws-lambda");
const sns = require("@aws-cdk/aws-sns");
const cdk = require("@aws-cdk/core");
const nodeunit_1 = require("nodeunit");
const lib_1 = require("../lib");
class TestCustomResource extends cdk.Construct {
    constructor(scope, id, opts = {}) {
        super(scope, id);
        const singletonLambda = new lambda.SingletonFunction(this, 'Lambda', {
            uuid: 'TestCustomResourceProvider',
            code: new lambda.InlineCode('def hello(): pass'),
            runtime: lambda.Runtime.PYTHON_2_7,
            handler: 'index.hello',
            timeout: cdk.Duration.minutes(5),
        });
        this.resource = new lib_1.CustomResource(this, 'Resource', {
            ...opts,
            provider: lib_1.CustomResourceProvider.fromLambda(singletonLambda),
        });
    }
}
module.exports = nodeunit_1.testCase({
    'custom resources honor removalPolicy': {
        'unspecified (aka .Destroy)'(test) {
            // GIVEN
            const app = new cdk.App();
            const stack = new cdk.Stack(app, 'Test');
            // WHEN
            new TestCustomResource(stack, 'Custom');
            // THEN
            assert_1.expect(stack).to(assert_1.haveResource('AWS::CloudFormation::CustomResource', {}, assert_1.ResourcePart.CompleteDefinition));
            test.equal(app.synth().tryGetArtifact(stack.stackName).findMetadataByType('aws:cdk:protected').length, 0);
            test.done();
        },
        '.Destroy'(test) {
            // GIVEN
            const app = new cdk.App();
            const stack = new cdk.Stack(app, 'Test');
            // WHEN
            new TestCustomResource(stack, 'Custom', { removalPolicy: cdk.RemovalPolicy.DESTROY });
            // THEN
            assert_1.expect(stack).to(assert_1.haveResource('AWS::CloudFormation::CustomResource', {}, assert_1.ResourcePart.CompleteDefinition));
            test.equal(app.synth().tryGetArtifact(stack.stackName).findMetadataByType('aws:cdk:protected').length, 0);
            test.done();
        },
        '.Retain'(test) {
            // GIVEN
            const app = new cdk.App();
            const stack = new cdk.Stack(app, 'Test');
            // WHEN
            new TestCustomResource(stack, 'Custom', { removalPolicy: cdk.RemovalPolicy.RETAIN });
            // THEN
            assert_1.expect(stack).to(assert_1.haveResource('AWS::CloudFormation::CustomResource', {
                DeletionPolicy: 'Retain',
                UpdateReplacePolicy: 'Retain',
            }, assert_1.ResourcePart.CompleteDefinition));
            test.done();
        },
    },
    'custom resource is added twice, lambda is added once'(test) {
        // GIVEN
        const app = new cdk.App();
        const stack = new cdk.Stack(app, 'Test');
        // WHEN
        new TestCustomResource(stack, 'Custom1');
        new TestCustomResource(stack, 'Custom2');
        // THEN
        assert_1.expect(stack).toMatch({
            'Resources': {
                'SingletonLambdaTestCustomResourceProviderServiceRole81FEAB5C': {
                    'Type': 'AWS::IAM::Role',
                    'Properties': {
                        'AssumeRolePolicyDocument': {
                            'Statement': [
                                {
                                    'Action': 'sts:AssumeRole',
                                    'Effect': 'Allow',
                                    'Principal': {
                                        'Service': 'lambda.amazonaws.com',
                                    },
                                },
                            ],
                            'Version': '2012-10-17',
                        },
                        'ManagedPolicyArns': [
                            {
                                'Fn::Join': ['', [
                                        'arn:', { 'Ref': 'AWS::Partition' }, ':iam::aws:policy/service-role/AWSLambdaBasicExecutionRole',
                                    ]],
                            },
                        ],
                    },
                },
                'SingletonLambdaTestCustomResourceProviderA9255269': {
                    'Type': 'AWS::Lambda::Function',
                    'Properties': {
                        'Code': {
                            'ZipFile': 'def hello(): pass',
                        },
                        'Handler': 'index.hello',
                        'Role': {
                            'Fn::GetAtt': [
                                'SingletonLambdaTestCustomResourceProviderServiceRole81FEAB5C',
                                'Arn',
                            ],
                        },
                        'Runtime': 'python2.7',
                        'Timeout': 300,
                    },
                    'DependsOn': [
                        'SingletonLambdaTestCustomResourceProviderServiceRole81FEAB5C',
                    ],
                },
                'Custom1D319B237': {
                    'Type': 'AWS::CloudFormation::CustomResource',
                    'DeletionPolicy': 'Delete',
                    'UpdateReplacePolicy': 'Delete',
                    'Properties': {
                        'ServiceToken': {
                            'Fn::GetAtt': [
                                'SingletonLambdaTestCustomResourceProviderA9255269',
                                'Arn',
                            ],
                        },
                    },
                },
                'Custom2DD5FB44D': {
                    'Type': 'AWS::CloudFormation::CustomResource',
                    'DeletionPolicy': 'Delete',
                    'UpdateReplacePolicy': 'Delete',
                    'Properties': {
                        'ServiceToken': {
                            'Fn::GetAtt': [
                                'SingletonLambdaTestCustomResourceProviderA9255269',
                                'Arn',
                            ],
                        },
                    },
                },
            },
        });
        test.done();
    },
    'custom resources can specify a resource type that starts with Custom::'(test) {
        const app = new cdk.App();
        const stack = new cdk.Stack(app, 'Test');
        new lib_1.CustomResource(stack, 'MyCustomResource', {
            resourceType: 'Custom::MyCustomResourceType',
            provider: lib_1.CustomResourceProvider.fromTopic(new sns.Topic(stack, 'Provider')),
        });
        assert_1.expect(stack).to(assert_1.haveResource('Custom::MyCustomResourceType'));
        test.done();
    },
    'fails if custom resource type is invalid': {
        'does not start with "Custom::"'(test) {
            const app = new cdk.App();
            const stack = new cdk.Stack(app, 'Test');
            test.throws(() => {
                new lib_1.CustomResource(stack, 'MyCustomResource', {
                    resourceType: 'NoCustom::MyCustomResourceType',
                    provider: lib_1.CustomResourceProvider.fromTopic(new sns.Topic(stack, 'Provider')),
                });
            }, /Custom resource type must begin with "Custom::"/);
            test.done();
        },
        'has invalid characters'(test) {
            const app = new cdk.App();
            const stack = new cdk.Stack(app, 'Test');
            test.throws(() => {
                new lib_1.CustomResource(stack, 'MyCustomResource', {
                    resourceType: 'Custom::My Custom?ResourceType',
                    provider: lib_1.CustomResourceProvider.fromTopic(new sns.Topic(stack, 'Provider')),
                });
            }, /Custom resource type name can only include alphanumeric characters and/);
            test.done();
        },
        'is longer than 60 characters'(test) {
            const app = new cdk.App();
            const stack = new cdk.Stack(app, 'Test');
            test.throws(() => {
                new lib_1.CustomResource(stack, 'MyCustomResource', {
                    resourceType: 'Custom::0123456789012345678901234567890123456789012345678901234567891',
                    provider: lib_1.CustomResourceProvider.fromTopic(new sns.Topic(stack, 'Provider')),
                });
            }, /Custom resource type length > 60/);
            test.done();
        },
    },
    '.ref returns the intrinsic reference (physical name)'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const res = new TestCustomResource(stack, 'myResource');
        // THEN
        test.deepEqual(stack.resolve(res.resource.ref), { Ref: 'myResourceC6A188A9' });
        test.done();
    },
});
//# sourceMappingURL=data:application/json;base64,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