"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const core_1 = require("@aws-cdk/core");
const cxapi = require("@aws-cdk/cx-api");
const record_set_1 = require("./record-set");
const route53_generated_1 = require("./route53.generated");
const util_1 = require("./util");
class HostedZone extends core_1.Resource {
    constructor(scope, id, props) {
        super(scope, id);
        /**
         * VPCs to which this hosted zone will be added
         */
        this.vpcs = new Array();
        util_1.validateZoneName(props.zoneName);
        const resource = new route53_generated_1.CfnHostedZone(this, 'Resource', {
            name: props.zoneName + '.',
            hostedZoneConfig: props.comment ? { comment: props.comment } : undefined,
            queryLoggingConfig: props.queryLogsLogGroupArn ? { cloudWatchLogsLogGroupArn: props.queryLogsLogGroupArn } : undefined,
            vpcs: core_1.Lazy.anyValue({ produce: () => this.vpcs.length === 0 ? undefined : this.vpcs })
        });
        this.hostedZoneId = resource.ref;
        this.hostedZoneNameServers = resource.attrNameServers;
        this.zoneName = props.zoneName;
        for (const vpc of props.vpcs || []) {
            this.addVpc(vpc);
        }
    }
    static fromHostedZoneId(scope, id, hostedZoneId) {
        class Import extends core_1.Resource {
            constructor() {
                super(...arguments);
                this.hostedZoneId = hostedZoneId;
            }
            get zoneName() {
                throw new Error(`HostedZone.fromHostedZoneId doesn't support "zoneName"`);
            }
        }
        return new Import(scope, id);
    }
    /**
     * Imports a hosted zone from another stack.
     */
    static fromHostedZoneAttributes(scope, id, attrs) {
        class Import extends core_1.Resource {
            constructor() {
                super(...arguments);
                this.hostedZoneId = attrs.hostedZoneId;
                this.zoneName = attrs.zoneName;
            }
        }
        return new Import(scope, id);
    }
    /**
     * Lookup a hosted zone in the current account/region based on query parameters.
     */
    static fromLookup(scope, id, query) {
        const DEFAULT_HOSTED_ZONE = {
            Id: '/hostedzone/DUMMY',
            Name: query.domainName,
        };
        const response = core_1.ContextProvider.getValue(scope, {
            provider: cxapi.HOSTED_ZONE_PROVIDER,
            dummyValue: DEFAULT_HOSTED_ZONE,
            props: query
        });
        // CDK handles the '.' at the end, so remove it here
        if (response.Name.endsWith('.')) {
            response.Name = response.Name.substring(0, response.Name.length - 1);
        }
        return this.fromHostedZoneAttributes(scope, id, {
            hostedZoneId: response.Id,
            zoneName: response.Name,
        });
    }
    /**
     * Add another VPC to this private hosted zone.
     *
     * @param vpc the other VPC to add.
     */
    addVpc(vpc) {
        this.vpcs.push({ vpcId: vpc.vpcId, vpcRegion: core_1.Stack.of(vpc).region });
    }
}
exports.HostedZone = HostedZone;
/**
 * Create a Route53 public hosted zone.
 *
 * @resource AWS::Route53::HostedZone
 */
class PublicHostedZone extends HostedZone {
    static fromPublicHostedZoneId(scope, id, publicHostedZoneId) {
        class Import extends core_1.Resource {
            constructor() {
                super(...arguments);
                this.hostedZoneId = publicHostedZoneId;
            }
            get zoneName() { throw new Error(`cannot retrieve "zoneName" from an an imported hosted zone`); }
        }
        return new Import(scope, id);
    }
    constructor(scope, id, props) {
        super(scope, id, props);
        if (props.caaAmazon) {
            new record_set_1.CaaAmazonRecord(this, 'CaaAmazon', {
                zone: this
            });
        }
    }
    addVpc(_vpc) {
        throw new Error('Cannot associate public hosted zones with a VPC');
    }
    /**
     * Adds a delegation from this zone to a designated zone.
     *
     * @param delegate the zone being delegated to.
     * @param opts     options for creating the DNS record, if any.
     */
    addDelegation(delegate, opts = {}) {
        new record_set_1.ZoneDelegationRecord(this, `${this.zoneName} -> ${delegate.zoneName}`, {
            zone: this,
            recordName: delegate.zoneName,
            nameServers: delegate.hostedZoneNameServers,
            comment: opts.comment,
            ttl: opts.ttl,
        });
    }
}
exports.PublicHostedZone = PublicHostedZone;
/**
 * Create a Route53 private hosted zone for use in one or more VPCs.
 *
 * Note that `enableDnsHostnames` and `enableDnsSupport` must have been enabled
 * for the VPC you're configuring for private hosted zones.
 *
 * @resource AWS::Route53::HostedZone
 */
class PrivateHostedZone extends HostedZone {
    static fromPrivateHostedZoneId(scope, id, privateHostedZoneId) {
        class Import extends core_1.Resource {
            constructor() {
                super(...arguments);
                this.hostedZoneId = privateHostedZoneId;
            }
            get zoneName() { throw new Error(`cannot retrieve "zoneName" from an an imported hosted zone`); }
        }
        return new Import(scope, id);
    }
    constructor(scope, id, props) {
        super(scope, id, props);
        this.addVpc(props.vpc);
    }
}
exports.PrivateHostedZone = PrivateHostedZone;
//# sourceMappingURL=data:application/json;base64,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